# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    run()           --  run function of this test case
"""

from AutomationUtils.cvtestcase import CVTestCase
from VirtualServer.VSAUtils import VirtualServerHelper, OptionsHelper, VirtualServerUtils
from AutomationUtils import constants
from Reports.utils import TestCaseUtils
from Web.Common.cvbrowser import BrowserFactory, Browser
from Web.AdminConsole.Helper.LoginHelper import LoginMain
from Web.AdminConsole.Helper.VirtualServerHelper import AdminConsoleVirtualServer
from Web.AdminConsole.adminconsole import AdminConsole


class TestCase(CVTestCase):
    """Class for executing Basic acceptance Test of VMWARE Full Snap backup
    and Restore test case - v1/v2 File Indexing"""

    def __init__(self):
        """Initializes test case class object"""
        super(TestCase, self).__init__()
        self.name = "VSA VMWARE SNAP Full Backup and Restore Cases for v1/v2 file indexing"
        self.product = self.products_list.VIRTUALIZATIONVMWARE
        self.feature = self.features_list.DATAPROTECTION
        self.test_individual_status = True
        self.test_individual_failure_message = ""
        self.show_to_user = True
        self.tcinputs = {}
        self.utils = TestCaseUtils(self)
        self.browser = None
        self.vsa_obj = None

    def setup(self):
        """
        Sets up the variables required for running the test case
        """
        self.utils.reset_temp_dir()
        download_directory = self.utils.reset_temp_dir()
        VirtualServerUtils.decorative_log("Initializing browser objects")
        factory = BrowserFactory()
        self.browser = factory.create_browser_object()
        self.browser.set_downloads_dir(download_directory)
        self.browser.open()

        VirtualServerUtils.decorative_log("Creating a login object")
        self.admin_console = AdminConsole(self.browser, self.commcell.webconsole_hostname)
        self.admin_console.login( self.inputJSONnode['commcell']['commcellUsername'],
                                  self.inputJSONnode['commcell']['commcellPassword'])

        VirtualServerUtils.decorative_log("Creating an object for Virtual Server helper")
        self.vsa_obj = AdminConsoleVirtualServer( self.instance, self.browser,
                                                  self.commcell, self.csdb)
        self.vsa_obj.hypervisor = self.tcinputs['ClientName']
        self.vsa_obj.instance = self.tcinputs['InstanceName']
        self.vsa_obj.subclient = self.tcinputs['SubclientName']
        self.vsa_obj.subclient_obj = self.subclient
        self.vsa_obj.ci_enabled = True
        self.vsa_obj.vsa_discovery()

    def run(self):
        """Main function for test case execution"""

        try:
            auto_subclient = backup_options = None
            self.log.info("Started executing %s testcase", self.id)
            VirtualServerUtils.decorative_log("Initialize helper objects")
            auto_subclient = VirtualServerUtils.subclient_initialize(self)
            proxy_options = OptionsHelper.FullVMRestoreOptions(auto_subclient, self)
            proxy_used = proxy_options.proxy_client

            VirtualServerUtils.decorative_log("Backup")
            backup_options = OptionsHelper.BackupOptions(auto_subclient)
            _adv = {"create_backup_copy_immediately": True}
            backup_options.advance_options = _adv
            backup_options.backup_method = "SNAP"
            backup_options.cleanup_testdata_before_backup = False
            if backup_options.collect_metadata:
                raise Exception("Metadata collection is enabled")
            auto_subclient.backup(backup_options)

            self.log.info("Verifying that File Indexing job runs and completes successfully")
            file_indexing_job_id = auto_subclient.get_in_line_file_indexing_job()

            self.log.info("Validating File indexing job status")
            auto_subclient.check_status_file_indexing(file_indexing_job_id)
            try:
                self.log.info('Get details about the File Indexing job to extract child VM Backup Jobs, the VM GUIDs, and'
                              'proxy used for validation')

                file_indexing_job_details = auto_subclient.get_file_indexing_job_details(file_indexing_job_id)
                for guid in file_indexing_job_details.keys():
                    child_backup_job_id = file_indexing_job_details[guid][0]
                    vm_guid = guid
                    self.log.info("Validate Archive Files")
                    auto_subclient.validate_file_indexing_archive_files(child_backup_job_id)
            except Exception as exp:
                self.log.error("Exception when validating File Indexing: %s" % str(exp))

            try:
                VirtualServerUtils.decorative_log("Performing file level restore to guest agent from VM level")
                self.vsa_obj.ci_enabled = False
                self.vsa_obj.snap_restore = False
                self.vsa_obj.file_level_restore(vm_level=True)
            except Exception as exp:
                self.test_individual_failure_message = str(exp)

            try:
                for vm in auto_subclient.vm_list:
                    self.log.info("Got the VM GUID: %s", guid)
                    self.log.info("Got the Child Backup Job ID: %s", child_backup_job_id)
                    self.log.info("Verify if SOLR is used during Browse.")
                    auto_subclient.verify_if_solr_used(child_backup_job_id, vm_guid)

            except Exception as exp:
                self.test_individual_failure_message = str(exp)

            try:
                self.log.info("Now Search and Download a file")
                self.vsa_obj.ci_enabled = True
                self.vsa_obj.ci_download_file()
            except Exception as exp:
                self.log.error('Failed with error: %s', str(exp))

        except Exception as exp:
            self.log.error('Failed with error: %s', str(exp))
            self.result_string = str(exp)
            self.status = constants.FAILED

        finally:
            self.vsa_obj.cleanup_testdata()
            if not self.test_individual_status:
                self.result_string = self.test_individual_failure_message
                self.status = constants.FAILED
            self.browser.close()
