# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
TestCase is the only class in this file
"""

from cvpysdk.subclients.exchange.usermailbox_subclient import UsermailboxSubclient

from Application.Exchange.ExchangeMailbox.exchange_mailbox import ExchangeMailbox
from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils.options_selector import OptionsSelector
from Web.Common.page_object import TestStep, handle_testcase_exception
from Web.Common.exceptions import CVTestStepFailure


class TestCase(CVTestCase):
    """Class to perform check for O365 license consumption for active users only"""
    test_step = TestStep()

    def __init__(self):
        super(TestCase, self).__init__()
        self.name = 'Check O365 license consumption for active users only'
        self.client = None
        self.exchange_subclient = None
        self.agent = None
        self.instance = None
        self.backupset = None

        self._utility = None

        self.license_sp_query = """
            DECLARE @isMasterSPRunning INT=0
            SELECT @isMasterSPRunning = CONVERT(INT, value) FROM GXGlobalParam WITH (NOLOCK) WHERE name = 'LicCalcUsageMasterSPRunning'
            IF @isMasterSPRunning = 1
            BEGIN
                WHILE @isMasterSPRunning = 1
                BEGIN     
                    WAITFOR DELAY '00:00:05'
                    SET @isMasterSPRunning = 0
                    SELECT @isMasterSPRunning = CONVERT(INT, value) FROM GXGlobalParam WITH (NOLOCK) WHERE name = 'LicCalcUsageMasterSPRunning'
                END
            END
            ELSE
            BEGIN
                EXEC LicCalcUsageMaster @nCallerType = 2
            END
        """
        self.query = "select distinct ObjectName from Lic_CurrentUsage where UsageType =18" \
                     " and AppTypeId =137 and lictype = 200011"

        self.initial_consumed_licenses = None
        self.new_association_user_mailboxes = None
        self.new_user_smtp_aliases = None
        self.new_association_other_mailboxes = None
        self.delete_count = None

        self.tcinputs = {
            "ExchangePlan": None
        }

    def get_mailboxes_for_associations(self, discovered_users, associated_users,
                                       user_count=15, other_count=5):
        """
        Args:
            discovered_users (list): All discovered users from the client
            associated_users (list): All associated users of the client
            user_count (int): Required new user mailboxes
            other_count (int): Required new other mailboxes
        Returns:
            new_association_user_mailboxes (list): New user mailboxes for Association
            new_association_other_mailboxes (list): New shared and group mailboxes for Association
        """
        associated_users_alias_names = set(user['alias_name'] for user in associated_users)

        new_association_user_mailboxes = []
        new_association_other_mailboxes = []

        shared_count = group_count = other_count

        for user in discovered_users:
            if not user_count and not shared_count and not group_count:
                break

            if user['aliasName'] not in associated_users_alias_names:
                if user_count and user['msExchRecipientTypeDetails'] == 1:
                    new_association_user_mailboxes.append(user['aliasName'])
                    self.new_user_smtp_aliases[user['smtpAdrress']] = user['aliasName']
                    user_count -= 1
                elif group_count and user['msExchRecipientTypeDetails'] == 3:
                    new_association_other_mailboxes.append(user['aliasName'])
                    group_count -= 1
                elif shared_count and user['msExchRecipientTypeDetails'] == 36:
                    new_association_other_mailboxes.append(user['aliasName'])
                    shared_count -= 1

        return new_association_user_mailboxes, new_association_other_mailboxes

    @test_step
    def set_new_associations(self):
        """Set new associations to the client"""
        discovered_users = self.exchange_subclient.discover_users
        self.log.info('Discovered %d users', len(discovered_users))

        associated_users = self.exchange_subclient.users + self.exchange_subclient.o365groups
        self.log.info('Associated Users: %d', len(associated_users))

        self.new_association_user_mailboxes, self.new_association_other_mailboxes = \
            self.get_mailboxes_for_associations(discovered_users, associated_users)

        subclient_content = {
            'mailboxNames': self.new_association_user_mailboxes + self.new_association_other_mailboxes,
            'plan_name': self.tcinputs['ExchangePlan']
        }

        self.log.info('Associating %d new user mailboxes', len(self.new_association_user_mailboxes))
        self.log.info('Associating %d new other mailboxes', len(self.new_association_other_mailboxes))
        self.exchange_subclient.set_user_assocaition(subclient_content, use_policies=False)

    @test_step
    def filter_already_associated_mailboxes(self):
        """Filter already associated mailboxes with other clients"""
        prev_len = len(self.new_association_user_mailboxes)

        if not self.initial_consumed_licenses:
            self.log.info('No licenses consumed yet.')
            return

        # check if initial consumed licenses are populated with smtp addresses
        if '@' in self.initial_consumed_licenses[0]:
            # filter consumed licenses using SMTP Addresses
            new_association_user_stmp_addresses = set(self.new_user_smtp_aliases.keys())
            self.initial_consumed_licenses = set(self.initial_consumed_licenses)

            new_license_consumed_smtps = list(new_association_user_stmp_addresses
                                              .difference(self.initial_consumed_licenses))
            self.new_association_user_mailboxes = [
                self.new_user_smtp_aliases[val] for val in new_license_consumed_smtps
            ]
        else:  # filter consumed licenses using alias names
            self.initial_consumed_licenses = set(self.initial_consumed_licenses)
            self.new_association_user_mailboxes = list(set(self.new_association_user_mailboxes)
                                                       .difference(self.initial_consumed_licenses))

        if not prev_len == len(self.new_association_user_mailboxes):
            self.log.info("%d User Mailboxes have already consumed licences from other client"
                          , prev_len - len(self.new_association_user_mailboxes))
        else:
            self.log.info('None of the mailboxes consumed licenses from other clients.')
 
    @test_step
    def check_test_case_requirement(self):
        """Check basic requirement for the test case"""
        if not len(self.new_association_user_mailboxes) >= 4:
            raise CVTestStepFailure('Only %d User mailbox(es) have not consumed licences. '
                                    'Add new user mailboxes and try again' %
                                    len(self.new_association_user_mailboxes))
        self.log.info('Requirement satisfied.')

    @test_step
    def execute_sp(self):
        """Execute required SP to update licenses"""
        self.log.info('Executing required storage procedure')
        self._utility.update_commserve_db(self.license_sp_query)

    @test_step
    def license_check_after_new_associations(self):
        """License check after new associations to the client"""
        new_consumed_licenses = self._utility.exec_commserv_query(self.query)[0]

        self.log.info('Total new consumed licenses: %d', len(new_consumed_licenses))
        self.log.info('Initial + Added new user associations: %d'
                      , len(self.initial_consumed_licenses)
                      + len(self.new_association_user_mailboxes))

        if not (len(new_consumed_licenses) ==
                len(self.initial_consumed_licenses) + len(self.new_association_user_mailboxes)):
            raise CVTestStepFailure("License check failed after adding new mailboxes")

        self.log.info('License check after adding Mailboxes in successful')

    @test_step
    def delete_user_associations(self):
        """Delete half of the new associated user mailboxes"""
        if len(self.new_association_user_mailboxes) in {0, 1}:
            self.delete_count = len(self.new_association_user_mailboxes)
        else:
            self.delete_count = len(self.new_association_user_mailboxes) // 2

        delete_association_user_mailboxes = self.new_association_user_mailboxes[:self.delete_count]
        subclient_content = {
            'mailboxNames': delete_association_user_mailboxes,
            'plan_name': self.tcinputs['ExchangePlan']
        }

        self.log.info('Deleting %d User Mailboxes', len(delete_association_user_mailboxes))
        self.exchange_subclient.delete_user_assocaition(subclient_content, use_policies=False)

    @test_step
    def license_check_after_deleting_user_associations(self):
        """License check after deleting user associations"""
        new_consumed_licenses = self._utility.exec_commserv_query(self.query)[0]
        self.log.info('Consumed licenses after deleting User Mailboxes: %d',
                      len(new_consumed_licenses))

        if not (len(new_consumed_licenses) ==
                len(self.initial_consumed_licenses) +
                len(self.new_association_user_mailboxes) - self.delete_count):
            raise CVTestStepFailure("License check failed after deleting user mailboxes")

        self.log.info('License check after deleting user Mailbox(es) in successful')

    def setup(self):
        self.log.info('Creating Exchange Mailbox client object.')
        self.exmbclient_object = ExchangeMailbox(self)

        self._client = self.exmbclient_object.cvoperations.add_exchange_client()
        self.log.info("Exchange Client has been created")

        self.exchange_subclient = self.exmbclient_object.cvoperations.subclient
        self.log.info("Exchange Subclient is created")

        self.new_user_smtp_aliases = dict()
        self._utility = OptionsSelector(self.commcell)

        self.execute_sp()  # update Lic_CurrentUsage table
        self.initial_consumed_licenses = list(self._utility.exec_commserv_query(self.query)[0])
        self.log.info('Initial Consumed Licenses: %d', len(self.initial_consumed_licenses))

    def run(self):
        try:
            self.set_new_associations()
            self.filter_already_associated_mailboxes()
            self.check_test_case_requirement()
            self.execute_sp()
            self.license_check_after_new_associations()
            self.delete_user_associations()
            self.execute_sp()
            self.license_check_after_deleting_user_associations()
        except Exception as ex:
            handle_testcase_exception(self, ex)
