# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------


""""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    run()           --  run function of this test case
"""

from AutomationUtils.cvtestcase import CVTestCase
from Web.Common.cvbrowser import BrowserFactory, Browser
from Web.AdminConsole.adminconsole import AdminConsole
from Web.AdminConsole.AdminConsolePages.Events import Events
from Web.AdminConsole.Components.table import Table
from Reports.utils import TestCaseUtils
from datetime import datetime


class TestCase(CVTestCase):
    """Class for executing Basic acceptance Test for Events test case"""

    def __init__(self):
        """Initializes test case class object"""
        super(TestCase, self).__init__()
        self.name = "Basic Test Case to verify proper working of Events page filters"
        self.utils = TestCaseUtils(self)
        self.browser = None
        self.admin_console = None
        self.table = None
        self.event_obj = None
        self.tcinputs = {}

    def verify_event_table_displayed(self):
        """ To verify that the events table is displayed """

        event_table_data = self.table.get_table_data()
        if event_table_data:
            self.log.info('Event table displayed correctly')
        else:
            exp = "'Event table not displayed correctly'"
            raise Exception(exp)

    def verify_filters(self):
        """ To apply the filters and verify the data displayed one by one"""

        self.event_obj.show_critical_events()
        self.verify_events_list('Critical')
        self.event_obj.show_major_events()
        self.verify_events_list('Major')
        self.event_obj.show_minor_events()
        self.verify_events_list('Minor')
        self.event_obj.show_info_events()
        self.verify_events_list('Info')
        self.event_obj.show_critical_events(day=True)
        self.verify_events_list('Critical', day=True)

    def verify_events_list(self, event_type, day=False):
        """
        Method to verify lists all the events for a job
        Args:
                event_type - event severity selected
        """

        self.log.info('Validating displayed columns on {} filter applied'.format(event_type))
        if day:
            self.verify_date_for_critical()

        event_severity_list = self.table.get_column_data('Severity')
        similar_visible_list_count = event_severity_list.count(event_type)
        total_column_element_count = len(event_severity_list)
        if similar_visible_list_count == total_column_element_count:
            self.log.info('All the values match, filter "{}" working correctly'.format(event_type))
        else:
            exp = "'Values do not match, filter - {} not working correctly'".format(event_type)
            raise Exception(exp)

    def verify_date_for_critical(self):
        """ Method to check if all the values for filter Critical (last 24 hours) are of """

        event_date_list = self.table.get_column_data('Date')
        current_timestamp = datetime.now().timestamp()
        for i in range(len(event_date_list)):
            event_date_list[i] = datetime.timestamp(datetime.strptime(event_date_list[i], '%b %d, %Y %I:%M:%S %p'))
            time_diff = current_timestamp - event_date_list[i]
            if time_diff > 86400:
                exp = "'All events are not from the last 24 hrs'"
                raise Exception(exp)

        self.log.info('All events are of last 24 hrs')

    def setup(self):
        """Setup function of this test case"""
        factory = BrowserFactory()
        self.browser = factory.create_browser_object()
        self.browser.open()

        self.admin_console = AdminConsole(self.browser, self.commcell.webconsole_hostname)
        self.admin_console.login(self.inputJSONnode['commcell']['commcellUsername'],
                                 self.inputJSONnode['commcell']['commcellPassword'])
        self.table = Table(self.admin_console)
        self.event_obj = Events(self.admin_console)

    def run(self):
        """Main function for test case execution"""
        try:
            self.admin_console.navigator.navigate_to_events()
            self.verify_event_table_displayed()
            self.verify_filters()

        except Exception as exp:
            self.utils.handle_testcase_exception(exp)

    def tear_down(self):
        """ To clean-up the test case environment created """
        AdminConsole.logout_silently(self.admin_console)
        Browser.close_silently(self.browser)
