# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""
Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    run()           --  run function of this test case
"""
import sys
from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils import constants as cv_constants
from Database.SplunkApplication.splunk import Splunk

class TestCase(CVTestCase):
    """
    Class for executing advanced test case for creating new splunk client and performing
    backup and recovery and verifying the same.
    """

    def __init__(self):
        """
        init method of the test-case class which defines test-case arguments
        """
        super(TestCase, self).__init__()
        self.name = "Splunk iDA, Advanced Backup and Restore with unconditional overwrite"
        self.show_to_user = False
        self.splunk_object = None
        self.tcinputs = {
            "NewClientName": None,
            "MasterNode": None,
            "MasterUri": None,
            "UserName": None,
            "Password": None,
            "SplunkHomePath": None,
            "Plan": None,
            "Nodes": None,  # list of slave nodes:[slave1, slave2]
            "Slave1Ip": None,
            "Slave1Port": None,
            "Slave1SplunkUsername": None,
            "Slave1SplunkPassword": None
        }

    def run(self):
        """
        Run function of this test case
        """
        try:
            self.log.info("Creating Splunk Object")
            self.splunk_object = Splunk(self)
            self.log.info("Splunk Object Creation Successful")
            self.log.info("Starting Splunk Client Creation")
            client_obj = self.splunk_object.cvoperations.add_splunk_client()
            if client_obj is None:
                raise Exception("New Splunk Client Creation Failed")
            self.log.info("Splunk Client Creation Successful")
            index_obj = self.splunk_object.add_splunk_index()
            index_name = index_obj["name"]
            total_eventcount = index_obj["totalEventCount"]
            self.log.info("Starting Backup Job")
            nodes = self.tcinputs.get("Nodes")
            self.splunk_object.cvoperations.update_client_nodes(client_obj, nodes)
            index_list = [index_name]
            self.splunk_object.cvoperations.update_client_content(client_obj, index_list)
            self.splunk_object.cvoperations.run_backup(client_obj)
            self.log.info("Backup job Successful")
            self.splunk_object.edit_splunk_index(index_name)
            self.splunk_object.cvoperations.run_restore(client_obj, index_list)
            self.splunk_object.splunk_rolling_restart()
            self.splunk_object.cvoperations.verify_restore(total_eventcount, index_name)
            self.log.info("Starting Cleanup Job")
            self.splunk_object.cvoperations.cleanup(client_obj)
            self.splunk_object.cleanup_index(index_name)
            self.log.info("Cleanup Job Successful")
            self.log.info("Test CASE SUCCESSFUL")

        except Exception as ex:
            self.log.error("Exception in the test case")
            self.log.error(
                'Error %s on line %s. Error %s', type(ex).__name__,
                sys.exc_info()[-1].tb_lineno, ex
            )
            self.result_string = str(ex)
            self.status = cv_constants.FAILED
