# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
""" Verify End-user security """

from time import sleep, time
from cvpysdk.schedules import Schedules
from Web.Common.cvbrowser import BrowserFactory
from Web.Common.cvbrowser import Browser
from Web.Common.exceptions import CVTestCaseInitFailure, CVTestStepFailure
from Web.WebConsole.Reports import cte
from Web.Common.page_object import TestStep
from Web.WebConsole.Reports.Custom import viewer
from Web.WebConsole.webconsole import WebConsole
from Web.AdminConsole.adminconsole import AdminConsole
from Web.AdminConsole.Reports.manage_reports import ManageReport
from Web.AdminConsole.Reports.report import Report
from Web.AdminConsole.Reports.manage_schedules import ManageSchedules

from AutomationUtils.mail_box import MailBox
from AutomationUtils import config
from AutomationUtils.cvtestcase import CVTestCase

from Reports.utils import TestCaseUtils
from Reports.Custom.utils import CustomReportUtils


_CONSTANTS = config.get_config()
export_type = cte.ConfigureSchedules.Format


class TestCase(CVTestCase):
    """
        TestCase class used to execute the test case from here.
        """
    test_step = TestStep()

    def __init__(self):
        super(TestCase, self).__init__()
        self.name = "Verify End User Security check"
        self.browser = None
        self.webconsole = None
        self.admin_console = None
        self.reports = "Backup job summary"
        self.format = export_type.HTML
        self.manage_report = None
        self.manage_schedules_api = None
        self.recipient_id = _CONSTANTS.email.email_id
        self.schedule_window = None
        self.navigator = None
        self.mail = None
        self.non_admin_user = "auto_non_admin_user_56526"
        self.table = None
        self.viewer = None
        self.schedule_name = "Automation_tc_56526_%s_%s" % (self.format, str(int(time())))
        self.utils = CustomReportUtils(self)
        self.utils = TestCaseUtils(self)

    def init_tc(self):
        """
        Initial configuration for the test case
        """
        try:
            self.mail = MailBox()
            self.mail.connect()
            self.manage_schedules_api = Schedules(self.commcell)

            # open browser
            self.browser = BrowserFactory().create_browser_object()
            self.browser.open()

            self.admin_console = AdminConsole(self.browser, self.commcell.webconsole_hostname)
            self.admin_console.login()
            self.navigator = self.admin_console.navigator
            self.manage_report = ManageReport(self.admin_console)
            self.navigator.navigate_to_reports()
            self.manage_report.access_report(self.reports)
            self.webconsole = WebConsole(self.browser, self.commcell.webconsole_hostname)
            self.viewer = viewer.CustomReportViewer(self.webconsole)
            self.table = viewer.DataTable("Job Details")
            self.viewer.associate_component(self.table)

        except Exception as exception:
            raise CVTestCaseInitFailure(exception) from exception

    @test_step
    def create_non_admin_user(self):
        """Create a non admin user"""
        non_admin_password = self.inputJSONnode['commcell']['commcellPassword']
        role_name = "Report_Management_56526"
        # if user exists no need to create user/role.
        if not self.commcell.users.has_user(self.non_admin_user):
            self.log.info("Creating user [%s]", self.non_admin_user)
            self.commcell.users.add(
                user_name=self.non_admin_user,
                email='reportautomation@commvault.com',
                password=non_admin_password
            )
        else:
            self.log.info("Non admin user [%s] already exists", self.non_admin_user)
            return
        # create role
        if not self.commcell.roles.has_role(role_name):
            self.commcell.roles.add(rolename=role_name, permission_list=["Report Management"])
            entity_dictionary = {
                'assoc1': {
                    'clientName': [self.commcell.commserv_name],
                    'role': [role_name]
                }
            }
            non_admin_user = self.commcell.users.get(self.non_admin_user)
            non_admin_user.update_security_associations(entity_dictionary=entity_dictionary,
                                                        request_type='UPDATE')
            self.log.info("Non admin user [%s] is created", self.non_admin_user)

    @test_step
    def get_report_server_data(self):
        """Get the list of servers from the report"""
        servers = self.table.get_column_data('Server')
        return servers

    @test_step
    def create_schedules(self):
        """create schedule with end-user security enabled"""

        self.log.info("Creating [%s]  schedule with the [%s] export_type for the [%s] user ",
                      self.schedule_name, self.format, self.non_admin_user)
        report = Report(self.admin_console)
        schedule = report.schedule()
        schedule.set_schedule_name(self.schedule_name)
        schedule.select_format(self.format)
        schedule.enable_end_user_security()
        schedule.select_user_to_notify(self.non_admin_user)
        schedule.save()

    def verify_schedule_exists(self):
        """Verify schedules are present in commcell"""
        self.manage_schedules_api.refresh()
        self.log.info("Checking [%s] schedule is created", self.schedule_name)
        if not self.manage_schedules_api.has_schedule(self.schedule_name):
            err_str = "[%s] schedule does not exists in commcell" % self.schedule_name
            raise CVTestStepFailure(err_str)
        else:
            self.log.info("verified [%s] schedule is present in commcell",
                          self.schedule_name)

    def run_schedules(self):
        """Run schedule"""
        _schedule = self.manage_schedules_api.get(self.schedule_name)
        self.log.info("Running [%s] schedule", self.schedule_name)
        job_id = _schedule.run_now()
        job = self.commcell.job_controller.get(job_id)
        self.log.info("Wait for [%s] job to complete", str(job))
        if job.wait_for_completion(300):  # wait for max 5 minutes
            self.log.info("Schedule job completed with job id:[%s] for [%s] schedule",
                          job_id, self.schedule_name)
        else:
            err_str = "[%s] Schedule job failed with job id [%s]" % (self.schedule_name, job)
            raise CVTestStepFailure(err_str)
        self.log.info("All schedules are completed successfully")

    def validate_schedule_mails(self):
        """Validate schedule mails"""

        self.utils.reset_temp_dir()
        self.log.info("verifying [%s] schedule email for [%s] report with [%s] file extension",
                      self.schedule_name, self.reports, self.format)
        self.utils.download_mail(self.mail, self.schedule_name)

        # To navigate to the downloaded mail
        file_path = self.utils.get_attachment_files(ends_with="HTML")
        self.browser.open_new_tab()
        self.browser.switch_to_latest_tab()
        self.browser.driver.get(file_path[0])
        sleep(3)
        servers = self.table.get_column_data('Server')
        server_name = list(set(servers))
        if server_name[0] == self.commcell.commserv_name and len(server_name) == 1:
            self.log.info(" The email content has only visible client")
        else:
            self.log.error("Expected server name is [%s] but received [%s]",
                           self.get_report_server_data(), server_name[0])
        self.browser.close_current_tab()
        self.browser.driver.switch_to.window(self.browser.driver.window_handles[0])

    def delete_schedules(self):
        """Delete the schedules"""
        manage_schedules_web = ManageSchedules(self.admin_console)
        self.navigator.navigate_to_reports()
        self.manage_report.view_schedules()
        manage_schedules_web.delete_schedules([self.schedule_name])
        self.log.info("Schedules are deleted")

    @test_step
    def verify_schedule_data(self):
        """ create schedule and validate it"""
        self.create_schedules()
        self.verify_schedule_exists()
        self.run_schedules()
        self.log.info("Wait for 2 minutes for mail to be received")
        sleep(120)
        self.validate_schedule_mails()
        self.delete_schedules()

    def run(self):
        """Main function for test case execution"""
        try:
            self.init_tc()
            self.create_non_admin_user()
            self.get_report_server_data()
            self.verify_schedule_data()

        except Exception as err:
            self.utils.handle_testcase_exception(err)
        finally:
            self.mail.disconnect()
            AdminConsole.logout_silently(self.admin_console)
            Browser.close_silently(self.browser)
