# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

TestCase is the only class definied in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    run()           --  run function of this test case
"""

from AutomationUtils import constants
from AutomationUtils.cvtestcase import CVTestCase
from Server.SmartClientGroups.smartclient_helper import SmartClientHelper
from Server.Security.securityhelper import OrganizationHelper
from AutomationUtils.options_selector import OptionsSelector

# Class of Testcase is named as TestCase which inherits from CVTestCase


class TestCase(CVTestCase):
    """ Class for executing basic test case of File System Backup using user defined parameters"""

    # Constructor for the testcase
    def __init__(self):
        """Initializes the testcase object"""
        super(TestCase, self).__init__()
        self.name = 'Testcase for Creation of Smart Client Group with name SCG2'
        self.organization_helper = None
        self.options_selector = None
        self.company_name = "COMP_TEST_NAME55"
        self.company_email = "TestAutomation3@commvault.com"
        self.company_contact = "CONTACT_COMP_55"
        self.company_alias = "COMP_TEST_NAME55"

    def setup_entities(self):
        """Sets up the Entities required for this testcase"""
        # Creating Users, User1 and User2 and Adding them to a user Group
        self.log.info("=============================")
        self.log.info("Settting up Testcase Entities")
        self.organization_helper = OrganizationHelper(self.commcell)
        self.options_selector = OptionsSelector(self.commcell)
        clients_list = self.options_selector.get_ready_clients(list(self.commcell.clients.all_clients), num=3)[0]
        self.tenant_admin_clients = {"assoc1": {"clientName": clients_list, "role": ["Client Admins"]}}
        self.organization_helper.create(name=self.company_name,
                                        email=self.company_email,
                                        contact_name=self.company_contact,
                                        company_alias=self.company_alias)
        # Associate clients to Company User
        company_user = self.commcell.users.get(self.company_alias + "\\" + self.company_email.split("@")[0])
        company_user.update_security_associations(entity_dictionary=self.tenant_admin_clients,
                                                  request_type="UPDATE")
        self.log.info("Entities Setup Completed")
        self.log.info("=============================")

    def cleanup_entities(self):
        """Cleans up the entities created for the testcase"""
        self.log.info("Cleaning Up Testcase Entities")
        self.commcell.organizations.delete(self.company_name)
        self.log.info("Testcase Entities Cleaned")

    def run(self):
        """Main function for test case execution"""
        try:
            # Setup testcase Entities
            self.setup_entities()
            # Initialize client groups object
            self.log.info("Initializing Client Groups")
            smartclient_helper = SmartClientHelper(commcell_object=self.commcell,
                                                   group_name='SCG2',
                                                   description='Test Group',
                                                   client_scope='Clients of Companies',
                                                   value=self.company_alias)

            # Get a preview of the clients that will be a part of this group
            preview_clients_list = smartclient_helper.preview_clients(value=smartclient_helper.value)
            rule_list = []
            self.log.info("""
                            ====================================================
                            Step1:
                            Creating Automatic Client Group with Client installed
                            ====================================================
                            """)
            self.log.info("Creating Rule for Client equal to Installed")
            rule1 = smartclient_helper.create_smart_rule(filter_rule='Client',
                                                         filter_condition='equal to',
                                                         filter_value='Installed')

            rule_list.append(rule1)

            self.log.info('Creating Client Group %s for testcase %s', smartclient_helper.group_name, self.id)
            # Posting Client Group Make request
            smartclient_helper.create_smart_client(smart_rule_list=rule_list)

            # Get Generated clients list
            created_clients_list = smartclient_helper.get_clients_list(smartclient_helper.group_name)
            # Validation of Preview clients and Created group clients
            smartclient_helper.validate_clients_list(preview_clients_list, created_clients_list)

        except Exception as excp:
            self.log.error('Failed with error %s', str(excp))
            # Set the Test-Case params : result_string, status
            self.result_string = str(excp)
            self.status = constants.FAILED
        finally:
            if smartclient_helper is not None:
                smartclient_helper.smart_client_cleanup()
            self.cleanup_entities()
