# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    setup()         --  setup function of this test case

    run()           --  run function of this test case

    tear_down()     --  tear down function of this test case

"""
import random
import string
import os
from deepdiff import DeepDiff
from office365.runtime.auth.authentication_context import AuthenticationContext
from Application.Sharepoint.sharepoint_online import SharePointOnline
from AutomationUtils import constants
from AutomationUtils.cvtestcase import CVTestCase
from Web.Common.exceptions import CVTestCaseInitFailure, CVTestStepFailure
from Web.Common.page_object import TestStep


def generate_random_file():
    """ generates random file and populates data
    """
    size = random.randint(50, 100)
    chars = ''.join([random.choice(string.ascii_letters) for i in range(size)])
    file_name = ''.join([random.choice(string.ascii_letters) for i in range(10)]) + '.txt'
    with open(file_name, 'w') as f:
        f.write(chars)
    return file_name


def delete_file(file_name):
    """Deletes the specified file"""
    if os.path.exists(file_name):
        os.remove(file_name)
    else:
        raise Exception("The file does not exist")


def process_metadata(dictionary):
    """Returns only required fields in metadata and  removes all unwanted fields"""
    return {key: dictionary.get(key) for key in dictionary
            if not isinstance(dictionary.get(key), dict)}


class TestCase(CVTestCase):
    """Class for executing the test case of -------------

    For now, executing testcase with  manual configuration.
    All comments will be uncommented once the backup issue is resolved

    """
    test_step = TestStep()

    def __init__(self):
        """Initializes test case class object

            Properties to be initialized:

                name            (str)       --  name of this test case

                tcinputs        (dict)      --  test case inputs with input name as dict key
                                                and value as input type

        """
        super(TestCase, self).__init__()
        self.name = "Office365- SharePoint Online-Data protection-Incremental Backup and Restore-General"
        self.sp_client_object = None
        self.tcinputs = {
            "PseudoClientName": None,
            "ServerPlanName": None,
            "IndexServer": None,
            "AccessNodes": None,
            # ------to be uncommented after backup issue is fixed ------
            # "GlobalAdministrator": None,
            # "GlobalAdministrator Password": None,
            # "AzureAppId": None,
            # "AzureAppKeyValue": None,
            # "AzureDirectoryId": None,
            # ----------------------------------------------------------
            "TenantUrl": None,
            "Username": None,
            "Password": None,
            "AzureUserName": None,
            "AzureSecret": None,
            "SiteUrl": None,
            "ClientId": None,
            "ClientSecret": None
        }
        self.site_user_account_info = None
        self.test_folder = {}
        self.test_list = {}
        self.browse_response = None
        self.share_point_data_flag = False

    def init_tc(self):
        """ Initialization function for the test case. """
        try:
            self.log.info('Creating SharePoint client object.')
            self.sp_client_object = SharePointOnline(self)

            self.sp_client_object.pseudo_client_name = self.tcinputs.get("PseudoClientName")
            self.sp_client_object.server_plan = self.tcinputs.get('ServerPlanName')

            # Infrastructure details
            self.sp_client_object.index_server = self.tcinputs.get('IndexServer')
            self.sp_client_object.access_nodes_list = self.tcinputs.get('AccessNodes')

            # ------to be uncommented after backup issue is fixed ------
            # SharePoint details
            # self.sp_client_object.global_administrator = self.tcinputs.get('GlobalAdministrator')
            # self.sp_client_object.global_administrator_password = \
            #     self.tcinputs.get('GlobalAdministrator Password')

            # Azure details
            # self.sp_client_object.azure_app_id = self.tcinputs.get('AzureAppId', "")
            # self.sp_client_object.azure_app_key_id = self.tcinputs.get('AzureAppKeyValue', "")
            # self.sp_client_object.azure_directory_id = self.tcinputs.get('AzureDirectoryId', "")
            # -----------------------------------------------------
            self.sp_client_object.azure_username = self.tcinputs.get('AzureUserName', "")
            self.sp_client_object.azure_secret = self.tcinputs.get('AzureSecret', "")

            # ------to be removed after backup issue is fixed ------
            # SharePoint Details
            self.sp_client_object.tenant_url = self.tcinputs.get("TenantUrl")
            self.sp_client_object.user_username = self.tcinputs.get("Username")
            self.sp_client_object.user_password = self.tcinputs.get("Password")
            # --------------------------------------------------------

            # SharePoint Site details
            self.sp_client_object.site_url = self.tcinputs.get("SiteUrl", "")
            self.sp_client_object.api_client_id = self.tcinputs.get("ClientId", "")
            self.sp_client_object.api_client_secret = self.tcinputs.get("ClientSecret", "")
            self.log.info('SharePoint client object created.')
        except Exception as exception:
            raise CVTestCaseInitFailure(exception)from exception

    def create_list(self, list_title, request_body):
        """Creates a list in sharepoint site"""
        try:
            self.sp_client_object.create_sp_list(request_body)
            self.log.info("{0} list created successfully".format(list_title))
            self.log.info("Adding list item")
            item_title = ''.join([random.choice(string.ascii_letters) for i in range(10)])
            request_body = {
                "Title": item_title
            }
            response = self.sp_client_object.create_sp_list_item(list_title, request_body)
            return response
        except Exception as exception:
            self.log.exception("Exception while creating list on SharePoint Site: %s", str(exception))
            raise exception

    def create_site_structure_for_backup(self):
        """Creates the sample site structure for backup
            1. Create a folder under shared documents library
                - Create a folder with random name
                - Upload a file with some random content
            2. Create a list
        """
        try:
            if self.sp_client_object.generate_token():
                folder_name = ''.join([random.choice(string.ascii_letters) for i in range(10)])
                site_name = self.sp_client_object.site_url.split("/")[-1]
                folder_url = "/sites/" + site_name + "/shared documents/" + folder_name
                request_body = {
                    "ServerRelativeUrl": folder_url
                }
                self.sp_client_object.create_sp_folder(request_body)
                self.log.info("{0} folder created successfully".format(folder_url))
                self.test_folder = {
                    "folder_name": folder_name,
                    "folder_url": folder_url
                }
                file = generate_random_file()
                response = self.sp_client_object.upload_sp_binary_file(folder_url, file)
                self.log.info("{0} file uploaded successfully".format(file))
                file_metadata = process_metadata(response)
                self.test_folder["file"] = {
                    "file_name": file,
                    "file_metadata": file_metadata
                }
                delete_file(file)
                list_title = ''.join([random.choice(string.ascii_letters) for i in range(10)])
                request_body = {
                    "AllowContentTypes": True,
                    "BaseTemplate": 100,
                    "ContentTypesEnabled": True,
                    "Description": "This is a list created for test automation",
                    "Title": list_title
                }
                list_item_metadata = process_metadata(self.create_list(list_title, request_body))
                self.test_list = {
                    "list_title": list_title,
                    "list_item_metadata": list_item_metadata
                }
                self.share_point_data_flag = True
        except Exception as exception:
            self.log.exception("Exception while creating site structure on SharePoint Site: %s", str(exception))
            raise exception

    def modify_backup_content(self):
        """Modifies SharePoint backup content"""
        try:
            list_item_title = ''.join([random.choice(string.ascii_letters) for i in range(10)])
            request_body = {
                "Title": list_item_title
            }
            self.sp_client_object.update_sp_list_item(self.test_list.get("list_title"),
                                                               request_body)
            self.log.info("List item title is updated successfully from {0} to {1}".
                          format(self.test_list.get("list_item_title"), list_item_title))
            self.test_list["list_item_title"] = list_item_title
            list_item_metadata = self.sp_client_object.get_sp_list_item_metadata(self.test_list.get("list_title"))
            self.test_list["list_item_metadata"] = process_metadata(list_item_metadata)

            updated_file_name = self.test_folder.get("file").get("file_name")[:-4] + '_update.txt'
            self.sp_client_object.rename_sp_file(self.test_folder.get("folder_url"),
                                                          self.test_folder.get("file").get("file_name"),
                                                          updated_file_name)

            self.log.info("File name is successfully changed from {2}/{0} to {2}/{1}".
                          format(self.test_folder.get("file").get("file_name"), updated_file_name,
                                 self.test_folder.get("folder_url", "")))

            file_metadata = self.sp_client_object.get_sp_file_metadata(updated_file_name,
                                                                       self.test_folder.get("folder_url", ""))
            self.test_folder["file"] = {
                "file_name": updated_file_name,
                "file_metadata": process_metadata(file_metadata)
            }

            self.sp_client_object.cvoperations.wait_time(10,
                                                         "Waiting for file properties to get updated")

        except Exception as exception:
            self.log.exception("Exception while modifying backup content: %s", str(exception))
            raise exception

    def delete_backup_content(self):
        """Deletes the backed up content"""
        try:
            if self.sp_client_object.generate_token():
                self.log.info("Deleting folder from sharepoint site")
                folder_url = self.test_folder.get("folder_url", "")
                self.sp_client_object.delete_sp_file_or_folder(folder_url)
                self.log.info("{0} is deleted from sharepoint site".format(folder_url))

                self.log.info("Deleting list from sharepoint site")
                list_title = self.test_list.get("list_title")
                self.sp_client_object.delete_sp_list(list_title)

                self.log.info("Lists/{0} is deleted successfully from sharepoint site".format(list_title))

                self.log.info("Deleted all generated files successfully")
                self.share_point_data_flag = False
        except Exception as exception:
            self.log.exception("Exception while deleting backup content: %s", str(exception))
            raise exception

    def browse_restore_content_and_verify_browse_response(self):
        """Browses content for restore and verifies the response of restore operation"""
        try:
            self.log.info("Browsing content for restore")
            paths, dictionary = self.sp_client_object.cvoperations.subclient.browse(path=r"\MB")
            self.browse_response = paths
            if self.browse_response:
                if self.sp_client_object.site_url in self.browse_response[0]:
                    self.log.info("Browse for restore is successful")
            else:
                self.log.error("No content available for restore")
                raise CVTestStepFailure("Content not available for restore")
        except Exception as exception:
            self.log.exception("Exception occurred during browse: %s", str(exception))
            raise exception

    def restore_sharepoint_content(self):
        """Restores the selected content"""
        try:
            folder_url = None
            list_url = None
            for url in self.browse_response:
                if url.split("\\")[-1] == self.test_folder.get("folder_name"):
                    folder_url = url
                    break
            for url in self.browse_response:
                if url.split("\\")[-1] == self.test_list.get("list_title"):
                    list_url = url
                    break
            self.log.info("Restoring folder and list")
            self.log.info("Path for restore\nfolder url : {0}\nlist_url : {1}".format(folder_url, list_url))
            # ------to be uncommented after backup issue is fixed ------
            # self.sp_client_object.cvoperations.update_azure_storage_details()
            # --------------------------------------------------------
            self.sp_client_object.cvoperations.run_restore(paths=[folder_url, list_url])
        except Exception as exception:
            self.log.exception("Exception while triggering restore job: %s", str(exception))
            raise exception

    def collect_metadata_of_restore_content(self):
        """Collects the metadata of restored content"""
        try:
            if self.sp_client_object.generate_token():
                self.log.info("Collecting and validating metadata of file")
                filename = self.test_folder.get("file").get("file_name")
                folder_url = self.test_folder.get("folder_url", "")
                restore_metadata = process_metadata(self.sp_client_object.get_sp_file_metadata(filename, folder_url))
                exclude_paths = {"root['ContentTag']", "root['UniqueId']", "root['ETag']",
                                 "root['TimeCreated']", "root['TimeLastModified']"}
                restore_metadata['ServerRelativeUrl'] = restore_metadata.get('ServerRelativeUrl').lower()
                server_relative_url = self.test_folder.get("file").get("file_metadata", {}).get('ServerRelativeUrl')
                self.test_folder.get("file").get("file_metadata", {})['ServerRelativeUrl'] = server_relative_url.lower()
                self.compare_backup_and_restore_metadata(self.test_folder.get("file").get("file_metadata", {}),
                                                         restore_metadata, exclude_paths)

                self.log.info("Collecting and validating metadata of list")
                list_title = self.test_list.get("list_title")
                restore_metadata = process_metadata(self.sp_client_object.get_sp_list_item_metadata
                                                    (list_title))
                exclude_paths = {"root['ContentTypeId']", "root['GUID']"}
                self.compare_backup_and_restore_metadata(self.test_list.get("list_item_metadata", {}),
                                                         restore_metadata, exclude_paths)
                self.share_point_data_flag = True
        except Exception as exception:
            self.log.exception("Exception while collecting metadata from SharePoint Site: %s", str(exception))
            raise exception

    def compare_backup_and_restore_metadata(self,
                                            backup_metadata,
                                            restore_metadata,
                                            exclude_paths=None):
        """Compares backup metadata and restore metadata"""
        try:
            diff = DeepDiff(backup_metadata, restore_metadata, ignore_order=True,
                            exclude_paths=exclude_paths)
            if diff:
                if 'dictionary_item_removed' in diff.keys():
                    # By default title field is not enabled after restore. An MR is created for this. Will remove the
                    # following code once it is resolved
                    if "root['Title']" in diff.get('dictionary_item_removed')[0]:
                        list_title = self.test_list.get("list_title")
                        item_title = self.sp_client_object.get_sp_list_item_title(list_title)
                        if backup_metadata.get("Title") == item_title:
                            self.log.info("Backup and restore metadata are validated")
                else:
                    self.log.error("Difference in metadata {0}".format(diff))
                    self.log.exception("Backup and restore metadata are not same")
                    raise CVTestStepFailure("Backup and restore metadata are not same")
            else:
                self.log.info("Backup and restore metadata are validated")
        except Exception as exception:
            self.log.exception("Exception while comparing metadata: %s", str(exception))
            raise exception

    def delete_backup_site_structure(self):
        """Deletes the SharePoint site structure created for running testcase"""
        if self.share_point_data_flag:
            self.delete_backup_content()
            self.log.info("SharePoint site structure created for running testcase is deleted")
        else:
            self.log.info("Testcase got failed in between. Can't delete SharePoint site structure")

    def setup(self):
        """Setup function of this test case"""
        self.init_tc()
        self.create_site_structure_for_backup()
        self.sp_client_object.cvoperations.add_share_point_pseudo_client()

    def run(self):
        """Run function of this test case"""
        try:
            self.sp_client_object.cvoperations.browse_for_sp_sites()
            self.sp_client_object.cvoperations.associate_content_for_backup()
            self.sp_client_object.cvoperations.run_backup()
            self.modify_backup_content()
            self.sp_client_object.cvoperations.run_backup()
            self.delete_backup_content()
            self.browse_restore_content_and_verify_browse_response()
            self.restore_sharepoint_content()
            self.collect_metadata_of_restore_content()
        except Exception as exp:
            self.log.error('Failed to execute test case with error: %s', exp)
            self.result_string = str(exp)
            self.status = constants.FAILED

    def tear_down(self):
        """Tear down function of this test case"""
        if self.status != constants.FAILED:
            self.delete_backup_site_structure()
            self.sp_client_object.cvoperations.delete_share_point_pseudo_client\
                (self.sp_client_object.pseudo_client_name)
