# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    setup()         --  setup function of this test case

    run()           --  run function of this test case

    tear_down()     --  tear down function of this test case

"""

import json
from AutomationUtils import constants
from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils.machine import Machine
from AutomationUtils.options_selector import OptionsSelector
from dynamicindex.Datacube.crawl_job_helper import CrawlJobHelper
from Application.Exchange.ExchangeMailbox.solr_helper import SolrHelper
from Application.Exchange.ExchangeMailbox.exchange_mailbox import ExchangeMailbox
from Application.Exchange.ExchangeMailbox.constants import RETENTION_POLICY_DEFAULT
from Application.Exchange.ExchangeMailbox.constants import ARCHIVE_POLICY_DEFAULT
from Application.Exchange.ExchangeMailbox.constants import CLEANUP_POLICY_DEFAULT


class TestCase(CVTestCase):
    """Class for executing this test case"""

    def __init__(self):
        """Initializes test case class object

            Properties to be initialized:

                name            (str)       --  name of this test case

                tcinputs        (dict)      --  test case inputs with input name as dict key
                                                and value as input type

        """
        super(TestCase, self).__init__()
        self.name = "Standalone Index server - Validation of exchange role"
        self.tcinputs = {
            "IndexServerNodeName": None,
            "JobResultDirectory": None,
            "ProxyServers": None,
            "MediaAgentName": None,
            "ServiceAccountDetails": None,
            "ExchangeServerName": None,
            "DomainName": None,
            "DomainUserName": None,
            "DomainUserPassword": None,
            "UserMailBox": None
        }
        self.storage_policy = None
        self.archive_policy = None
        self.retention_policy = None
        self.cleanup_policy = None
        self.exchange_mailbox_client = None
        self.solr_helper = None
        self.crawl_job_helper = None
        self.total_count = None
        self.node_machine_obj = None
        self.media_agent_machine_obj = None
        self.index_server_roles = ['Exchange Index']

    def setup(self):
        """Setup function of this test case"""
        self.node_machine_obj = Machine(
            machine_name=self.tcinputs['IndexServerNodeName'],
            commcell_object=self.commcell)
        self.media_agent_machine_obj = Machine(
            machine_name=self.tcinputs['MediaAgentName'],
            commcell_object=self.commcell
        )
        option_selector_obj = OptionsSelector(self.commcell)
        index_directory_drive_letter = option_selector_obj.get_drive(self.node_machine_obj)
        restore_directory_drive_letter = option_selector_obj.get_drive(self.media_agent_machine_obj)
        self.tcinputs['EnvironmentType'] = 1
        self.tcinputs['IndexLocation'] = "%sindex_directory_%s" % (index_directory_drive_letter, self.id)
        self.tcinputs['DiskRestorePath'] = "%srestore_mail_%s" % (restore_directory_drive_letter, self.id)
        self.tcinputs['MountPath'] = "%sLibrary_%s" % (restore_directory_drive_letter, self.id)
        self.tcinputs['DiskRestoreClient'] = self.tcinputs['MediaAgentName']
        self.tcinputs['IndexServer'] = "Index_server_%s" % self.id
        self.tcinputs['SubclientName'] = "usermailbox"
        self.tcinputs['BackupsetName'] = "User Mailbox"
        self.tcinputs['StoragePolicyName'] = "storagePolicy%s" % self.id
        self.tcinputs['LibraryName'] = "Library_%s" % self.id
        self.tcinputs['RecallService'] = f'http://{self.commcell.webconsole_hostname}/webconsole'
        if not self.commcell.disk_libraries.has_library(self.tcinputs['LibraryName']):
            self.log.info("Creating %s on mount path %s" % (self.tcinputs['LibraryName'],
                                                            self.tcinputs['MountPath']))
            self.commcell.disk_libraries.add(self.tcinputs['LibraryName'],
                                             self.tcinputs['MediaAgentName'],
                                             self.tcinputs['MountPath'])
            self.log.info("Library created")
        archive_policy_default = ARCHIVE_POLICY_DEFAULT % self.id
        retention_policy_default = RETENTION_POLICY_DEFAULT % self.id
        cleanup_policy_default = CLEANUP_POLICY_DEFAULT % self.id
        if not self.commcell.storage_policies.has_policy(self.tcinputs[
                'StoragePolicyName']):
            self.log.info("Creating storage policy")
            self.commcell.storage_policies.add(
                self.tcinputs['StoragePolicyName'],
                self.tcinputs['LibraryName'],
                self.tcinputs['MediaAgentName']
            )
            self.log.info("Storage policy added successfully")
        if not self.commcell.index_servers.has(self.tcinputs['IndexServer']):
            self.node_machine_obj.remove_directory(self.tcinputs['IndexLocation'], 0)
            self.log.info("Creating index server")
            self.commcell.index_servers.create(
                self.tcinputs['IndexServer'],
                [self.tcinputs['IndexServerNodeName']],
                self.tcinputs['IndexLocation'],
                self.index_server_roles
            )
            self.log.info("Index server created")
            self.commcell.clients.refresh()
        self.storage_policy = self.commcell.storage_policies.get(
            self.tcinputs['StoragePolicyName']
        )
        self.exchange_mailbox_client = ExchangeMailbox(self)
        self.crawl_job_helper = CrawlJobHelper(self)
        self.log.info('Creating a exchange client')
        self._client = self.exchange_mailbox_client.cvoperations.add_exchange_client()
        self.log.info('Exchange client created with id: %s',
                      self._client.client_id)
        self.log.info("Creating an instance of subclient")
        self._subclient = self.exchange_mailbox_client.cvoperations.subclient
        self.log.info("Subclient instance created.")
        self.log.info("Creating policies.")
        self.cleanup_policy = self.exchange_mailbox_client.cvoperations.add_exchange_policy(
            self.exchange_mailbox_client.cvoperations.get_policy_object(
                cleanup_policy_default, "Cleanup"
            )
        )
        self.archive_policy = self.exchange_mailbox_client.cvoperations.add_exchange_policy(
            self.exchange_mailbox_client.cvoperations.get_policy_object(
                archive_policy_default, "Archive"
            )
        )
        self.retention_policy = self.exchange_mailbox_client.cvoperations.add_exchange_policy(
            self.exchange_mailbox_client.cvoperations.get_policy_object(
                retention_policy_default, "Retention"
            )
        )
        self.log.info("Policy generation completed.")

    def run(self):
        """Run function of this test case"""
        try:
            subclient_content = {
                "mailboxNames": self.tcinputs['UserMailBox'],
                "archive_policy": self.archive_policy,
                "retention_policy": self.retention_policy,
                "cleanup_policy": self.cleanup_policy
            }
            self.log.info("Now setting up the user email.")
            self.exchange_mailbox_client.cvoperations.subclient.\
                set_user_assocaition(subclient_content)
            self.exchange_mailbox_client.cvoperations.run_backup()
            self.media_agent_machine_obj.remove_directory(self.tcinputs['DiskRestorePath'], 0)
            self.log.info("Now starting the restore process")
            job = self.exchange_mailbox_client.cvoperations.subclient.\
                disk_restore(paths=[r"\MB"],
                             destination_client=self.tcinputs['DiskRestoreClient'],
                             destination_path=self.tcinputs['DiskRestorePath'],
                             overwrite=True,
                             journal_report=False)
            self.exchange_mailbox_client.cvoperations.check_job_status(job)
            restore_mails_count = self.crawl_job_helper.get_docs_count(
                folder_path=self.tcinputs['DiskRestorePath'],
                machine_name=self.tcinputs['DiskRestoreClient'],
                include_folders=False
            )
            self.solr_helper = SolrHelper(self.exchange_mailbox_client)
            select_dict = {
                "datatype": 2,
                "apid": self._subclient.subclient_id
            }
            solr_resp = self.solr_helper.create_url_and_get_response(
                select_dict=select_dict
            )
            self.total_count = self.solr_helper.get_count_from_json(solr_resp.content)
            self.log.info("Expected restore mails count : %s", self.total_count)
            if not restore_mails_count == self.total_count:
                self.log.error("Restored files count doesn't match with expected value")
                self.log.error("Expected value: %s", self.total_count)
                self.log.error("Actual value: %s", restore_mails_count)
                raise Exception("Restored mails count doesn't match with expected value")
            self.log.info("All mails Restored successfully")
            files_list = self.media_agent_machine_obj.get_files_in_path(self.tcinputs['DiskRestorePath'])
            file_name_map = {}
            for file in files_list:
                file_name_map["_".join((file.split('\\')[-1]).split('_')[:-3])] = file
            attr_params = {'szkb', 'conv'}
            op_params = {
                "rows": restore_mails_count
            }
            solr_resp = json.loads(self.solr_helper.create_url_and_get_response(
                select_dict=select_dict,
                attr_list=attr_params,
                op_params=op_params
            ).content)
            for doc in solr_resp['response']['docs']:
                disk_size = self.media_agent_machine_obj.get_file_size(file_name_map[doc['conv']], True)
                if int(disk_size) < int(doc['szkb']):
                    self.log.error("Invalid file size found for %s.msg" % doc['conv'])
                    self.log.error("Expected file size >=%s" % doc['szkb'])
                    raise Exception("Invalid file size found")
            self.log.info("No invalid file size found for any email")

        except Exception as exp:
            self.log.error('Failed to execute test case with error: %s', exp)
            self.result_string = str(exp)
            self.status = constants.FAILED

    def tear_down(self):
        """Tear down function of this test case"""
        if self.status == constants.PASSED:
            self.log.info("Deleting exchange pseudo client")
            self.commcell.clients.delete(self.exchange_mailbox_client.cvoperations.client.name)
            self.log.info("Exchange client deleted")
            self.log.info("Deleting exchange configuration policies")
            self.commcell.policies.configuration_policies.delete(
                ARCHIVE_POLICY_DEFAULT % self.id
            )
            self.commcell.policies.configuration_policies.delete(
                CLEANUP_POLICY_DEFAULT % self.id
            )
            self.commcell.policies.configuration_policies.delete(
                RETENTION_POLICY_DEFAULT % self.id
            )
            self.log.info("Exchange configuration policies deleted")
            self.log.info("Deleting storage policy")
            self.commcell.storage_policies.delete(self.tcinputs['StoragePolicyName'])
            self.log.info("Storage policy deleted")
            self.log.info("Deleting the index server")
            self.commcell.index_servers.delete(self.tcinputs['IndexServer'])
            self.log.info("Index server deleted")
            self.media_agent_machine_obj.remove_directory(self.tcinputs['DiskRestorePath'], 0)
            self.node_machine_obj.remove_directory(self.tcinputs['IndexLocation'], 0)
            self.log.info("Deleting the library")
            self.commcell.disk_libraries.delete(self.tcinputs["LibraryName"])
            self.log.info("Library deleted")
