# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()                   --  initialize TestCase class

    init_tc()                    --  initialize TestCase class

view_live_logs()                 --  Navigate to view logs for the client and choose the log provided

    thread_task_client_log()     --  Threaded task to get log data from client

    thread_task_live_log()       --  Threaded task to get log data from Admin console

    validation_view_logs()       --  validation logic for the opened log

    run()                        --  Main function for test case execution
"""
import threading, time
from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils.machine import Machine
from Reports.utils import TestCaseUtils
from Web.Common.page_object import TestStep
from Web.Common.exceptions import CVTestCaseInitFailure, CVTestStepFailure
from Web.Common.cvbrowser import BrowserFactory, Browser
from Web.AdminConsole.Helper.file_servers_helper import FileServersMain
from Web.AdminConsole.adminconsole import AdminConsole
from Web.AdminConsole.AdminConsolePages.view_logs import ViewLogs


class TestCase(CVTestCase):
    """Test case to verify view log file feature acceptance"""
    test_step = TestStep()

    def __init__(self):
        """Init method for test case class"""
        super(TestCase, self).__init__()
        self.name = "Admin console : View Logs Acceptance"
        self.browser = None
        self.admin_console = None
        self._utils = None
        self._file_servers = None
        self._view_logs = None
        self.log_data_from_client = ''
        self.log_data = ''
        self.client_machine = None
        self.tcinputs = {
            "log_name": None
        }

    def init_tc(self):
        """Initialize browser and redirect to page"""
        try:
            self._utils = TestCaseUtils(self)
            self.client_machine = Machine(self.client)
            self.browser = BrowserFactory().create_browser_object()
            self.browser.open()
            self.admin_console = AdminConsole(self.browser, self.commcell.webconsole_hostname)
            self.admin_console.login(
                self.inputJSONnode['commcell']['commcellUsername'],
                self.inputJSONnode['commcell']['commcellPassword']
            )
            self._file_servers = FileServersMain(self.admin_console, self.commcell)
            self._view_logs = ViewLogs(self.admin_console)
        except Exception as _exception:
            raise CVTestCaseInitFailure(_exception) from _exception

    @test_step
    def view_live_logs(self):
        """Navigate to view logs for the client and choose the log provided"""
        self._file_servers.view_live_logs(self.client.client_name, self.tcinputs['log_name'])

    def thread_task_client_log(self):
        """Threaded task to get log data from client"""
        self.log_data_from_client = self.client_machine.read_file(
            self.client_machine.join_path(self.client.install_directory, "Log Files", self.tcinputs['log_name']),
            last_n_lines=10).strip()

    def thread_task_live_log(self):
        """Threaded task to get log data from Admin console"""
        self._view_logs.pause_or_resume_logs()
        self.log_data = self._view_logs.get_log_data().strip()
        self._view_logs.pause_or_resume_logs()

    @test_step
    def validation_view_logs(self):
        """validation logic for the opened log"""
        client_thread = threading.Thread(target=self.thread_task_client_log())
        ac_thread = threading.Thread(target=self.thread_task_live_log())
        client_thread.start()
        ac_thread.start()
        client_thread.join()
        ac_thread.join()
        self.log_data_from_client = self.log_data_from_client.splitlines()
        self.log_data = self.log_data.splitlines()[-10:]
        if len(set(self.log_data_from_client) ^ set(self.log_data)) >= 10:
            raise CVTestStepFailure("Validation of log files from the client and Admin console is not matching")

    def run(self):
        """Main function for test case execution"""
        try:
            self.init_tc()
            self.view_live_logs()
            for validation_count in range(2):
                self.validation_view_logs()
                time.sleep(15)
        except Exception as err:
            self._utils.handle_testcase_exception(err)
        finally:
            AdminConsole.logout_silently(self.admin_console)
            Browser.close_silently(self.browser)
