# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
""" Validate HTML export of Custom report and  HTML file content validation"""
import os

from Web.API.webconsole import Reports
from Web.Common.cvbrowser import BrowserFactory
from Web.Common.cvbrowser import Browser
from Web.Common.exceptions import CVTestCaseInitFailure, CVTestStepFailure
from Web.Common.page_object import TestStep
from Web.WebConsole.Reports.Custom import viewer
from Web.WebConsole.Reports.navigator import Navigator
from Web.WebConsole.webconsole import WebConsole
from Web.WebConsole.Reports.Metrics.report import MetricsReport
from AutomationUtils.cvtestcase import CVTestCase
from Reports.utils import TestCaseUtils
from AutomationUtils.constants import AUTOMATION_DIRECTORY


class TestCase(CVTestCase):
    """
    TestCase class used to execute the test case from here.
    """
    test_step = TestStep()

    def __init__(self):
        super(TestCase, self).__init__()
        self.name = "HTML export validation"
        self.browser = None
        self.webconsole = None
        self.navigator = None
        self.utils = TestCaseUtils(self)
        self.html_browser = None
        self.table = None
        self.rpt_api = None
        self.import_report_name = "Formatters"

    def _init_tc(self):
        """
        Initial configuration for the test case
        """
        try:
            self.utils.reset_temp_dir()
            download_directory = self.utils.get_temp_dir()
            self.log.info("Download directory:%s", download_directory)
            self.browser = BrowserFactory().create_browser_object(name="ClientBrowser")
            self.browser.set_downloads_dir(download_directory)
            self.browser.open()
            self.webconsole = WebConsole(self.browser, self.commcell.webconsole_hostname)
            self.webconsole.login()
            self.webconsole.goto_reports()
            self.utils.webconsole = self.webconsole
            self.rpt_api = Reports(
                self.commcell.webconsole_hostname
            )
            self.navigator = Navigator(self.webconsole)
        except Exception as exception:
            raise CVTestCaseInitFailure(exception) from exception

    def access_report(self):
        """Access custom report"""
        self.rpt_api.import_custom_report_xml(
            os.path.join(AUTOMATION_DIRECTORY, "Reports", "Templates", f"{self.import_report_name}.xml"))
        self.navigator.goto_worldwide_report(self.import_report_name)
        self.webconsole.wait_till_load_complete()
        viewer_obj = viewer.CustomReportViewer(self.webconsole)
        self.table = viewer.DataTable(self.import_report_name)
        viewer_obj.associate_component(self.table)

    def get_html_content(self):
        """
        Read rows from table present in HTML
        """
        html_webconsole = WebConsole(self.html_browser, self.commcell.webconsole_hostname)
        html_viewer = viewer.CustomReportViewer(html_webconsole)
        html_table = viewer.DataTable(self.import_report_name)
        html_viewer.associate_component(html_table)
        html_data = html_table.get_rows_from_table_data()
        return html_data

    def access_html_file(self):
        """Access downloaded html file"""
        html_path = self.utils.poll_for_tmp_files(ends_with='html')[0]
        self.html_browser = BrowserFactory().create_browser_object(name="ClientBrowser")
        self.html_browser.open()
        self.html_browser.goto_file(file_path=html_path)

    @test_step
    def verify_export_to_html(self):
        """
        Verify export to html is working fine
        """
        report = MetricsReport(self.webconsole)
        export = report.export_handler()
        export.to_html()
        self.utils.wait_for_file_to_download('html')
        self.utils.validate_tmp_files("html")
        self.log.info("HTML export completed successfully")

    @test_step
    def verify_html_content(self):
        """
        Verify html file contents are matching with report table content
        """
        self.log.info("Verifying html content for the report [%s]", self.name)
        web_report_table_data = self.table.get_rows_from_table_data()
        self.access_html_file()
        html_report_table_data = self.get_html_content()
        if web_report_table_data != html_report_table_data:
            self.log.error("HTML contents are not matching with report table content")
            self.log.error("HTML content:%s", str(html_report_table_data))
            self.log.error("web report content:%s", str(web_report_table_data))
            raise CVTestStepFailure("HTML contents are not matching with report table content")
        self.log.info("HTML contents are verified successfully")

    def run(self):
        try:
            self._init_tc()
            self.access_report()
            self.verify_export_to_html()
            self.verify_html_content()
        except Exception as err:
            self.utils.handle_testcase_exception(err)
        finally:
            WebConsole.logout_silently(self.webconsole)
            Browser.close_silently(self.browser)
            self.html_browser.close()
