# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""
Testcase: This test case verifies whether Azure replication group creation succeeds with default options.
Then it waits for the replication and backup jobs to complete successfully, and finally verifies the details on the
replication monitor

Main file for executing this test case(verify the creation and updation of Azure recovery target
of replication type)

TestCase: Class for executing this test case
Sample JSON: {
        "ClientName": "idc",
        "source_vm": "vm1",
        "recovery_target": "target",
        "storage_name" : "storage",
        "source_ip": <source machine ip>",
        "source_subnet": "<source machine subnet>",
        "source_gateway": "<source machine gateway>",
        "destination_ip": "<destination machine ip>",
        "destination_subnet": "<destination machine subnet>",
        "destination_gateway": "<destination machine gateway>",
        "dns_1": "<destination machine dns primary>",
        "dns_2": "<destination machine dns secondary>",
        "wins_1": "<destination machine wins primary>",
        "wins_2": "<destination machine wins secondary>",
        "guest_username": "Admin",
        "guest_password": "<VM password>"
}
"""
from time import sleep

from AutomationUtils.cvtestcase import CVTestCase
from Reports.utils import TestCaseUtils
from VirtualServer.VSAUtils.VMHelpers.VmwareVM import VmwareVM
from VirtualServer.VSAUtils import VirtualServerUtils, OptionsHelper
from VirtualServer.VSAUtils.LiveSyncUtils import LiveSyncUtils
from Web.AdminConsole.DR.group_details import ReplicationDetails
from Web.AdminConsole.DR.replication import ReplicationGroup
from Web.AdminConsole.DR.virtualization_replication import SOURCE_HYPERVISOR_VMWARE
from Web.AdminConsole.adminconsole import AdminConsole
from Web.Common.cvbrowser import BrowserFactory
from Web.Common.exceptions import CVTestStepFailure, CVTestCaseInitFailure
from Web.Common.page_object import TestStep


class TestCase(CVTestCase):
    """This class is used to automate the replication of Defect #288874"""
    test_step = TestStep()
    _AGENT_NAME = 'virtual server'
    _INSTANCE_NAME = 'vmware'
    _BACKUPSET_NAME = 'defaultBackupSet'
    _HOSTNAME = 'DestVMHost'

    def __init__(self):
        """Initialises the objects and TC inputs"""
        super(TestCase, self).__init__()
        self.name = "Azure Replication group: Verify Creation and edit operation"

        self.tcinputs = {
            "ClientName": None,
            "source_vm": None,
            "recovery_target": None,
            "storage_name": None,
            "source_ip": None,
            "source_subnet": None,
            "source_gateway": None,
            "destination_ip": None,
            "destination_subnet": None,
            "destination_gateway": None,
            "dns_1": None,
            "dns_2": None,
            "wins_1": None,
            "wins_2": None,
            "guest_username": None,
            "guest_password": None,
        }

        self.client = None
        self.agent = None
        self.instance = None
        self.backupset = None
        self.subclient = None

        self.utils = None

        self._source_subclient = None
        self._vm = None
        self.destination_ip = None

        self.browser = None
        self.admin_console = None
        self.replication_group = None
        self.group_details = None

    @property
    def group_name(self):
        """Returns the virtualization group name"""
        return f"Group_TC_{self.id}"

    @property
    def schedule(self):
        """Returns the name of the schedule"""
        return self.replication_group.get_schedule_name_by_replication_group(self.group_name)

    @property
    def source_subclient(self):
        """Returns the VSA Auto subclient after initialising all the testcase objects"""
        try:
            hasattr(self, 'subclient')
        except:
            self.agent = self.client.agents.get(self._AGENT_NAME)
            self.instance = self.agent.instances.get(self._INSTANCE_NAME)
            self.backupset = self.agent.backupsets.get(self._BACKUPSET_NAME)
            self.subclient = self.backupset.subclients.get(self.group_name)
        if not self._source_subclient:
            self._source_subclient = VirtualServerUtils.subclient_initialize(self)
        return self._source_subclient

    @property
    def destination_vm_name(self):
        """Gets the destination VM name from the source subclient"""
        return self.source_subclient.get_live_sync_destination_vms(self.schedule)[0]

    @property
    def vm(self):
        """Returns the vmware vm after fetching it from the hypervisor object"""
        if not self._vm:
            self._vm = VmwareVM(self.source_subclient.hvobj, self.destination_vm_name)
        return self._vm

    @property
    def backup_job_id(self):
        """Get the latest backup job obj"""
        try:
            hasattr(self, 'subclient')
        except:
            self.agent = self.client.agents.get(self._AGENT_NAME)
            self.instance = self.agent.instances.get(self._INSTANCE_NAME)
            self.backupset = self.agent.backupsets.get(self._BACKUPSET_NAME)
            self.subclient = self.backupset.subclients.get(self.group_name)
        return self.subclient.find_latest_job().job_id

    @staticmethod
    def verify_ipv4(ip_address):
        """Verifies that the IPv4 are valid"""
        return not (ip_address is None
                    or ip_address.count('.') != 3
                    or not ''.join(ip_address.split('.')).isnumeric()
                    or not any([0 < int(ip_part) < 256 for ip_part in ip_address.split('.')])
                    )

    def login(self):
        """Logs in to admin console"""
        self.browser = BrowserFactory().create_browser_object()
        self.browser.open()
        self.admin_console = AdminConsole(self.browser, machine=self.inputJSONnode['commcell']['webconsoleHostname'])
        self.admin_console.login(self.inputJSONnode['commcell']['commcellUsername'],
                                 self.inputJSONnode['commcell']['commcellPassword'])

        self.replication_group = ReplicationGroup(self.admin_console)
        self.group_details = ReplicationDetails(self.admin_console)

    def logout(self):
        """Logs out of the admin console and closes the browser"""
        self.admin_console.logout_silently(self.admin_console)
        self.browser.close_silently(self.browser)

    def setup(self):
        """Sets up the Testcase"""
        try:
            self.utils = TestCaseUtils(self)
            self.login()
        except Exception as exp:
            raise CVTestCaseInitFailure(f'Failed to initialise testcase {str(exp)}')

    def wait_for_sync(self, num_minutes: int):
        """Waits for sync to happen for number of minutes"""
        for _ in range(num_minutes // 2):
            sleep(120)
            self.admin_console.refresh_page()
        if num_minutes % 2:
            sleep(60)

    def power_destination_vm_on(self):
        """Powers on the destination VM"""
        self.vm.power_on()
        self.vm.user_name = self.tcinputs["guest_username"]
        self.vm.password = self.tcinputs["guest_password"]
        self.vm.vm_guest_os = "Windows"
        self.log.info("VM object created with the powered on state")

    def power_destination_vm_off(self):
        """Powers off the destination VM"""
        self.vm.power_off()
        self.log.info("VM machine powered off")

    def verify_backup_job_completion(self):
        """Waits for backup job to complete and then verify its details"""
        self.log.info('Waiting to 3 minutes to let the jobs trigger')
        sleep(180)
        self.log.info('Waiting for backup job %s to finish', self.backup_job_id)
        job_obj = self.commcell.job_controller.get(self.backup_job_id)
        job_obj.wait_for_completion()
        self.utils.assert_comparison(job_obj.status, 'Completed')

    def verify_replication_job_completion(self):
        """Waits for replication job to complete and then verify its details"""
        self.log.info("Waiting 2 minutes to let live sync update")
        sleep(120)
        live_sync_utils = LiveSyncUtils(self.source_subclient, self.schedule)
        job_obj = live_sync_utils.get_recent_replication_job(self.backup_job_id)
        self.log.info("Waiting for replication job id %s to finish", job_obj.job_id)
        job_obj.wait_for_completion()
        self.utils.assert_comparison(job_obj.status, 'Completed')

    @test_step
    def delete_virtualization_group(self, wait_for_sync: bool = False):
        """Deletes the virtualization group if it exists already"""
        self.admin_console.navigator.navigate_to_replication_groups()
        if self.replication_group.has_group(self.group_name):
            self.log.info('The replication group %s exists, deleting it now', self.group_name)
            self.replication_group.delete_group(self.group_name)
            if wait_for_sync:
                self.wait_for_sync(5)
            else:
                self.admin_console.refresh_page()
            if self.replication_group.has_group(self.group_name):
                raise CVTestStepFailure(f"Deletion of replication group {self.group_name} failed")
        self.log.info('The replication group %s does not exist', self.group_name)

    @test_step
    def configure_replication_group(self):
        """Configures the replication group with default options"""
        vmware_configure = self.replication_group.configure_vmware()
        vmware_configure.content.set_name(self.group_name)
        vmware_configure.content.select_vm_from_browse_tree(self.tcinputs['ClientName'],
                                                            {"VMs and templates": [self.tcinputs['source_vm']]})
        sleep(5)
        vmware_configure.next()

        vmware_configure.target.select_recovery_target(self.tcinputs['recovery_target'])
        vmware_configure.target.unconditionally_overwrite_vm(True)
        vmware_configure.next()

        vmware_configure.storage_cache.select_storage(self.tcinputs['storage_name'])
        vmware_configure.next()

        override_vm = vmware_configure.override_options.override_vms(self.tcinputs['source_vm'])

        ip_settings = override_vm.add_ip()
        # The source IP address must be set static in the operating system
        ip_settings.set_source_ip(self.tcinputs['source_ip'], self.tcinputs['source_subnet'],
                                  self.tcinputs['source_gateway'])
        ip_settings.toggle_dhcp(enable=True)
        ip_settings.set_hostname(self._HOSTNAME)
        ip_settings.save()

        override_vm.save()
        vmware_configure.next()

        sleep(5)
        vmware_configure.finish()

    @test_step
    def verify_group_creation(self):
        """Verifies that the group has been configured with the correct settings in the command center"""
        if not self.replication_group.has_group(self.group_name):
            raise CVTestStepFailure(f"Group [{self.group_name}] does not exist on the replication groups page")
        self.replication_group.access_group(self.group_name)
        self.group_details.access_configuration_tab()

        override_vm = self.group_details.configuration.edit_virtual_machines(self.tcinputs['source_vm'],
                                                                             vm_type=SOURCE_HYPERVISOR_VMWARE)
        ip_settings = override_vm.edit_ip()
        self.utils.assert_comparison(list(ip_settings.source_ip.values()), [
            self.tcinputs['source_ip'],
            self.tcinputs['source_subnet'],
            self.tcinputs['source_gateway']
        ])
        self.utils.assert_comparison(ip_settings.hostname, self._HOSTNAME)
        self.utils.assert_comparison(ip_settings.dhcp_enabled, True)
        ip_settings.cancel()
        override_vm.cancel()

    @test_step
    def verify_hostname_and_dhcp(self):
        """Verifies that the destination VM hostname is set"""
        self.logout()
        self.verify_backup_job_completion()
        self.verify_replication_job_completion()

        self.power_destination_vm_on()
        self.utils.assert_comparison(self.vm.machine.get_hardware_info()['MachineName'], self._HOSTNAME)
        self.utils.assert_comparison(self.vm.machine.is_dhcp_enabled(), True)
        self.destination_ip = [self.vm.machine.ip_address, self.vm.machine.get_subnet().strip(),
                               self.vm.machine.get_default_gateway().strip()]

        self.power_destination_vm_off()

    @test_step
    def edit_group_static(self):
        """Sets static IP for the group with source and destination values"""
        self.login()
        self.admin_console.navigator.navigate_to_replication_groups()
        self.replication_group.access_group(self.group_name)
        self.group_details.access_configuration_tab()

        edit_vm = self.group_details.configuration.edit_virtual_machines(self.tcinputs['source_vm'],
                                                                         vm_type=SOURCE_HYPERVISOR_VMWARE)

        ip_settings = edit_vm.edit_ip(0)
        ip_settings.toggle_dhcp(enable=False)
        ip_settings.set_destination_ip(*self.destination_ip,
                                       self.tcinputs['dns_1'],
                                       self.tcinputs['dns_2'],
                                       self.tcinputs['wins_1'],
                                       self.tcinputs['wins_2'])

        ip_settings.save()
        edit_vm.save()

    @test_step
    def verify_static_ip(self):
        """Verifies that the correct VM ip settings are set on the destination VM"""
        edit_vm = self.group_details.configuration.edit_virtual_machines(self.tcinputs['source_vm'],
                                                                         vm_type=SOURCE_HYPERVISOR_VMWARE)
        ip_settings = edit_vm.edit_ip(0)

        self.utils.assert_comparison(list(ip_settings.destination_ip.values()), [
            *self.destination_ip,
            self.tcinputs['dns_1'],
            self.tcinputs['dns_2'],
            self.tcinputs['wins_1'],
            self.tcinputs['wins_2']
        ])

        ip_settings.cancel()
        edit_vm.cancel()
        self.logout()

        self.log.info('Performing incremental backup of the group')
        backup_options = OptionsHelper.BackupOptions(self.source_subclient)
        backup_options.backup_type = "INCREMENTAL"
        self.source_subclient.backup(backup_options, skip_discovery=True)
        self.verify_backup_job_completion()
        self.verify_replication_job_completion()

        self.power_destination_vm_on()
        self.utils.assert_comparison(self.vm.machine.get_hardware_info()['MachineName'], self._HOSTNAME)
        if not self.verify_ipv4(self.vm.machine.ip_address):
            raise CVTestStepFailure(f"The IP address seems to be incorrect [{self.vm.machine.ip_address}]")
        if not self.verify_ipv4(self.vm.machine.get_subnet().strip()):
            raise CVTestStepFailure(f"The subnet mask seems to be incorrect [{self.vm.machine.ip_address}]")
        if not self.verify_ipv4(self.vm.machine.get_default_gateway().strip()):
            raise CVTestStepFailure(f"The default gateway seems to be incorrect [{self.vm.machine.ip_address}]")

        destination_ip = [self.vm.machine.ip_address, self.vm.machine.get_subnet().strip(),
                          self.vm.machine.get_default_gateway().strip()]
        self.utils.assert_comparison(destination_ip, self.destination_ip)

        dns = self.vm.machine.get_dns_servers()
        self.utils.assert_includes(self.tcinputs['dns_1'], dns)
        self.utils.assert_includes(self.tcinputs['dns_2'], dns)

        # TODO: Find a way to find WINS and verify it
        self.power_destination_vm_off()

    def run(self):
        """Runs the testcase in order"""
        try:
            self.delete_virtualization_group(wait_for_sync=True)
            self.configure_replication_group()

            self.verify_group_creation()
            self.verify_hostname_and_dhcp()

            self.edit_group_static()
            self.verify_static_ip()

            self.login()
            self.delete_virtualization_group(wait_for_sync=False)

        except Exception as exp:
            self.utils.handle_testcase_exception(exp)

    def tear_down(self):
        """Tears down the testcase on terminate"""
        self.logout()
