# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Test case to verify vmware replication group using secondary group

Sample input:
"56282": {
            "ClientName": "idcvc61",
            "vm_names_to_backup": ["backup1"],
            "recovery_target":"test-3",
            "storage_name" : "FSvaultStorage",
            "secondary_storage_name": "Storage1"
       }

"""
from time import sleep

from Server.Scheduler.schedulerhelper import SchedulerHelper
from Web.AdminConsole.DR.replication import ReplicationGroup
from Web.AdminConsole.DR.monitor import ReplicationMonitor
from Web.AdminConsole.DR.group_details import ReplicationDetails
from Web.AdminConsole.VSAPages.vm_groups import VMGroups
from Web.AdminConsole.adminconsole import AdminConsole
from Web.AdminConsole.DR.virtualization_replication import _Target
from Web.Common.cvbrowser import BrowserFactory
from Web.Common.cvbrowser import Browser
from Web.Common.exceptions import CVTestCaseInitFailure, CVTestStepFailure
from Web.Common.page_object import TestStep
from AutomationUtils.cvtestcase import CVTestCase
from Reports.utils import TestCaseUtils


class TestCase(CVTestCase):
    """Test case to verify configuring vmware replication group using secondary copy."""
    test_step = TestStep()

    # strings repeatedly used, making them constants
    _SLA_STATUS = 'SLA status'
    _REPLICATION_GROUP = 'Replication group'
    _STATUS = 'Status'
    _GROUP_NAME = 'Group name'
    _SOURCE = 'Source'
    _DESTINATION = 'Destination'
    _TYPE = 'Type'
    _STATE = 'State'
    _VENDOR_NAME = 'VMware'

    def setup(self):
        """Initialize required veriables/inputs"""
        self.source_hypervisor = self.tcinputs["ClientName"]
        self.vm_names_to_backup = self.tcinputs["vm_names_to_backup"]
        self.recovery_target = self.tcinputs["recovery_target"]
        self.storage_name = self.tcinputs["storage_name"]
        self.secondary_storage_name = self.tcinputs["secondary_storage_name"]
        self.vitualization_group_name = "Auto_vitualization_group_56282"
        self.target_freq_number = 4
        self.target_freq_period = _Target.FREQUENCY_HOURS

        self.expected_replication_group_table_data = {
            self._GROUP_NAME: [self.vitualization_group_name],
            self._SOURCE: [self.source_hypervisor],
            self._DESTINATION: [self.recovery_target],
            self._TYPE: ['Virtual Server'],
            self._STATE: ['Enabled'],
            }

        self.expected_replication_monitor_table_data = {
            'Source': self.vm_names_to_backup,
            self._SLA_STATUS: ['Met'],
            self._REPLICATION_GROUP: [self.vitualization_group_name]
        }

    def __init__(self):
        super(TestCase, self).__init__()
        self.name = "CommandCenter: VMWare Replication group from Auxcopy"
        self.browser = None
        self.utils = None
        self.admin_console = None
        self.navigator = None
        self.replication_groups = None
        self.source_hypervisor = None
        self.vm_names_to_backup = None
        self.recovery_target = None
        self.storage_name = None
        self.secondary_storage_name = None
        self.vitualization_group_name = None
        self.expected_replication_monitor_table_data = None
        self.expected_replication_group_table_data = None
        self.replication_monitor = None
        self.vm_group = None
        self.target_freq_number = None
        self.target_freq_period = None
        self.replication_details = None
        self.schedule_name = None
        self.utils = TestCaseUtils(self)

    def init_commandcenter(self):
        """Initialize browser and redirect to page"""
        try:
            self.browser = BrowserFactory().create_browser_object()
            self.browser.open()
            self.admin_console = AdminConsole(self.browser, self.commcell.webconsole_hostname)
            self.admin_console.login()
            self.navigator = self.admin_console.navigator
            self.replication_groups = ReplicationGroup(self.admin_console)
            self.replication_monitor = ReplicationMonitor(self.admin_console)
            self.vm_group = VMGroups(self.admin_console)
            self.replication_details = ReplicationDetails(self.admin_console)
        except Exception as _exception:
            raise CVTestCaseInitFailure(_exception) from _exception

    @test_step
    def configure_vitualization_group(self):
        """Create a vitualization replication group"""
        try:
            self.navigator.navigate_to_replication_groups()
            vitualization_grp = self.replication_groups.configure_vmware()
            sleep(5)
            vitualization_grp.add_default_group(self.vitualization_group_name,
                                                self.source_hypervisor,
                                                self.vm_names_to_backup,
                                                self.recovery_target,
                                                self.storage_name,
                                                self.secondary_storage_name)

            self.log.info('Replication group created successfully')
        except Exception as err:
            raise CVTestStepFailure(err)

    @test_step
    def verify_vitualization_group_exists(self):
        """Verify vitualization replication group exists in Replication Groups page"""
        self.log.info("Verify Group name column is having configured vitualization group")
        self.admin_console.refresh_page()
        _table_contnet = self.replication_groups.get_replication_group_details_by_name(
            self.vitualization_group_name)
        actual_table_contnet = {
            self._GROUP_NAME: _table_contnet[self._GROUP_NAME],
            self._SOURCE: _table_contnet[self._SOURCE],
            self._DESTINATION: _table_contnet[self._DESTINATION],
            self._TYPE: _table_contnet[self._TYPE],
            self._STATE: _table_contnet[self._STATE]
        }
        if actual_table_contnet != self.expected_replication_group_table_data:
            raise CVTestStepFailure("Replication groups table contets are not matching;"
                                    "expected:%s; actual:%s" %
                                    (self.expected_replication_group_table_data,
                                     actual_table_contnet))
        self.log.info("Verified replication group[%s] created successfully!",
                      self.vitualization_group_name)

    @test_step
    def verify_frequency_and_storage(self):
        """Verify expected frequency is set"""
        self.log.info("Verifying replication frequency")
        self.replication_groups.access_group(self.vitualization_group_name)
        self.replication_details.access_configuration_tab()
        rpo_details = self.replication_details.configuration.get_rpo_details()
        actual_frequency = rpo_details['Replication frequency']
        expected_frequency = str(self.target_freq_number) + ' ' + self.target_freq_period.lower()
        if expected_frequency not in actual_frequency:
            raise CVTestStepFailure("Replication frequency must be [%s] but actual frequency is "
                                    "[%s]"
                                    % (expected_frequency, actual_frequency))
        self.log.info("Verified replication frequency")

        storage_details = self.replication_details.configuration.get_storage_details()
        expected_storage = [self.storage_name, self.secondary_storage_name]
        if storage_details != expected_storage:
            raise CVTestStepFailure(f"The replication storage information {storage_details} does not match "
                                    f"the expected value {expected_storage}")
        self.log.info("Verified storage details")

    @test_step
    def verify_vm_group(self):
        """Verify configured replication group exists in Protect->Virtualization->VM groups"""
        self.log.info("Wait for sync up for 20 seconds...")
        sleep(20)
        self.navigator.navigate_to_vm_groups()
        self.admin_console.refresh_page()
        actual_table_content = self.vm_group.get_details_by_vm_group(
            self.vitualization_group_name)
        if len(actual_table_content['Name']) > 1:
            raise CVTestStepFailure(
                "The get details returned multiple rows. This is most likely due to search fail")
        self.utils.assert_includes(self.vitualization_group_name, actual_table_content['Name'][0])
        self.utils.assert_comparison(actual_table_content['Vendor'][0], self._VENDOR_NAME)
        self.utils.assert_comparison(actual_table_content['Hypervisor'][0], self.source_hypervisor)
        self.utils.assert_comparison(
            actual_table_content['Plan'][0], f"{self.vitualization_group_name}_ReplicationPlan"
        )
        self.log.info("Verified replication group[%s] exists in vm groups",
                      self.vitualization_group_name)

    @test_step
    def verify_replication_monitor(self):
        """Verify replication monitor table contains configured replication group"""
        self.navigator.navigate_to_replication_monitor()
        table_content = self.replication_monitor.get_replication_group_details(
            self.vm_names_to_backup[0], self.vitualization_group_name)
        actual_table_content = {'Source': table_content['Source'],
                                self._SLA_STATUS: table_content[self._SLA_STATUS],
                                self._REPLICATION_GROUP: table_content[self._REPLICATION_GROUP]
                                }
        if actual_table_content != self.expected_replication_monitor_table_data:
            raise CVTestStepFailure("Replication monitor table content not equal; expected:%s, "
                                    "actual:%s" % (self.expected_replication_monitor_table_data,
                                                   actual_table_content))
        self.log.info("Verified replication monitor table content!")

    @test_step
    def delete_vitualization_group(self):
        """Delete replication group"""
        self.navigator.navigate_to_replication_groups()
        if self.replication_groups.has_group(self.vitualization_group_name):
            self.replication_groups.delete_group(self.vitualization_group_name)
            self.log.info("Replication[%s] group deleted!", self.vitualization_group_name)
            return True
        self.log.info("Replication group [%s] does not exist!",
                      self.vitualization_group_name)
        return False

    @test_step
    def wait_for_backup_job(self):
        """Wait for backup to complete"""
        _client = self.commcell.clients.get(self.source_hypervisor)
        _agent = _client.agents.get('virtual server')
        _backupset = _agent.backupsets.get('defaultBackupSet')
        _subclient = _backupset.subclients.get(self.vitualization_group_name)
        self.log.info("Wait for 3 minutes for jobs to trigger")
        sleep(180)
        job_id = _subclient.find_latest_job().job_id
        self.log.info(f"Waiting for Backup job id [{job_id} to complete")
        job_obj = self.commcell.job_controller.get(job_id)
        job_obj.wait_for_completion()
        if job_obj.status != 'Completed':
            raise CVTestStepFailure("Job[%s] failed" % job_id)

    @test_step
    def wait_for_replication_job(self):
        """wait for replication job to complete"""
        self.log.info("Wait for 16 minutes")
        sleep(960)
        self.schedule_name = self.replication_groups.get_schedule_name_by_replication_group(
            self.vitualization_group_name)
        self.client.schedules.refresh()
        schedule_obj = self.client.schedules.get(self.schedule_name)
        schedule_helper = SchedulerHelper(schedule_obj, self.commcell)
        job_obj = schedule_helper.get_jobid_from_taskid()
        self.log.info(f"Waiting for replication job id [{job_obj.job_id} to complete")
        job_obj.wait_for_completion()
        sleep(5)

    def logout(self):
        """Logout commandcenter"""
        AdminConsole.logout_silently(self.admin_console)
        Browser.close_silently(self.browser)

    @test_step
    def verify_deletion(self):
        """Verify replication group, vm group and schedules are deleted"""
        self.delete_vitualization_group()

        self.wait_for_sync()

        # verify vm group is deleted
        self.navigator.navigate_to_vm_groups()
        self.admin_console.refresh_page()
        if self.vm_group.has_vm_group(self.vitualization_group_name):
            raise CVTestStepFailure("VM group [%s] exists though replication group is deleted"
                                    % self.vitualization_group_name)
        # verify schedule is deleted
        self.commcell.schedules.refresh()
        if self.commcell.schedules.has_schedule(schedule_name=self.schedule_name):
            raise CVTestStepFailure("Schedule [%s] exists though replication group is deleted"
                                    % self.schedule_name)

    def wait_for_sync(self):
        """Wait for sync up, during waiting period page should not timeout, so refresh the page
        every 2 minutes"""
        self.log.info("Wait for sync up for 5 mins")
        sleep(120)
        self.admin_console.refresh_page()
        sleep(120)
        self.admin_console.refresh_page()
        sleep(60)

    def run(self):
        """Starting test case steps"""
        try:
            self.init_commandcenter()
            if self.delete_vitualization_group():
                # if group exists and deleted, wait for schedules/plans to sync once deleted
                self.wait_for_sync()
            self.configure_vitualization_group()
            self.verify_vitualization_group_exists()
            self.verify_frequency_and_storage()
            self.verify_vm_group()
            # till jobs to complete command-center may timeout, so logout and login back
            self.logout()
            self.wait_for_backup_job()
            self.log.info("wait for secondary copy job to be triggered [10 mins]")
            sleep(600)
            self.wait_for_replication_job()
            self.init_commandcenter()
            self.verify_replication_monitor()
            self.verify_deletion()
        except Exception as err:
            self.utils.handle_testcase_exception(err)
        finally:
            self.logout()
