# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    run()           --  run function of this test case
"""

from AutomationUtils.cvtestcase import CVTestCase

from Web.Common.cvbrowser import BrowserFactory, Browser

from Web.AdminConsole.adminconsole import AdminConsole
from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.Components.dialog import ModalDialog
from Web.AdminConsole.Components.panel import ModalPanel
from Web.AdminConsole.Reports.manage_reports import ManageReport
from Web.AdminConsole.Reports.manage_schedules import ManageSchedules
from Web.AdminConsole.AdminConsolePages.Jobs import Jobs

from Web.Common.exceptions import CVTestCaseInitFailure, CVTestStepFailure
from Web.Common.page_object import TestStep, handle_testcase_exception


class TestCase(CVTestCase):
    """Command Center: Table Component integration testcase"""

    test_step = TestStep()

    def __init__(self):
        """Initializes test case class object"""
        super(TestCase, self).__init__()
        self.name = "Command Center Table component integration"
        self.browser = None
        self.admin_console = None
        self.navigator = None
        self.table = None
        self.modal_dialog = None
        self.modal_panel = None

    def init_tc(self):
        """ Initial configuration for the test case. """
        try:
            self.browser = BrowserFactory().create_browser_object()
            self.browser.open()
            self.admin_console = AdminConsole(self.browser, self.commcell.webconsole_hostname)
            self.admin_console.login()
            self.navigator = self.admin_console.navigator
            self.navigator.navigate_to_servers()
            self.table = Table(self.admin_console)
            self.modal_dialog = ModalDialog(self.admin_console)
            self.modal_panel = ModalPanel(self.admin_console)
            self.client_name = self.commcell.clients.get(self.commcell.commserv_name).display_name
        except Exception as exception:
            raise CVTestCaseInitFailure(exception)from exception

    @test_step
    def get_number_of_columns(self):
        """gets number of columns present in table"""
        columns = self.table.get_number_of_columns()
        if not columns:
            raise CVTestStepFailure('unable to get columns')

    @test_step
    def get_visible_column_names(self):
        """Get visible Column names"""
        columns = self.table.get_visible_column_names()
        if 'Name' not in columns:
            raise CVTestStepFailure('Name not found in column names')

    @test_step
    def get_column_data(self):
        """Get column data"""
        column_data = self.table.get_column_data('Name')
        if not column_data:
            raise CVTestStepFailure('unable to get data for Name column')

    @test_step
    def access_action_item(self):
        """Selects the action item in table"""

        self.table.access_action_item(self.client_name, 'View jobs')
        if self.browser.driver.title != 'Jobs':
            raise CVTestStepFailure(
                "access action item failed, view jobs didn't access Jobs page"
            )

    @test_step
    def view_by_title(self):
        """View by type in grid"""
        self.navigator.navigate_to_jobs()
        self.table.view_by_title('Laptop jobs')
        if 'laptopJobs' not in self.browser.driver.current_url:
            raise CVTestStepFailure(
                "View by title drop down failed, "
                "click on laptop job in dropdown didn't access laptop jobs page"
            )

    @test_step
    def access_toolbar_menu(self):
        """Tool bar menu in table"""
        self.navigator.navigate_to_servers()
        self.table.access_toolbar_menu('addProxy')
        if self.modal_dialog.title() != 'Add gateway':
            raise CVTestStepFailure("Access tool bar menu didn't open gateway dialog")
        self.modal_dialog.click_cancel()

    @test_step
    def access_link(self):
        """Access the link from the table"""
        self.table.access_link(self.client_name)
        if self.browser.driver.title != self.client_name:
            raise CVTestStepFailure(
                "access link failed, click on Commserv name didn't access client detail page"
            )

    @test_step
    def access_link_by_column(self):
        """search by entity_name and access by link_text"""
        self.navigator.navigate_to_servers()
        self.table.access_link_by_column(
            self.client_name,
            self.client_name
        )
        if self.browser.driver.title != self.client_name:
            raise CVTestStepFailure(
                "access link by column failed, click on Commserv name "
                "didn't access client details page"
            )

    @test_step
    def access_context_action_item(self):
        """Selects the action item in table right click menu"""
        self.navigator.navigate_to_jobs()
        jobs = Jobs(self.admin_console)
        data = self.table.get_column_data('Job Id')
        if not data:
            jobs.access_job_history()
            data = self.table.get_column_data('Job Id')
            if not data:
                raise CVTestStepFailure(
                    "This setup doesn't have 1 active job or 1 job in last 24 hours to "
                    "validate table context menu"
                )
        self.table.access_context_action_item(
            data[0],
            self.admin_console.props['label.viewJobDetails']
        )
        if data[0] not in self.browser.driver.current_url:
            raise CVTestStepFailure(
                f"Context menu in jobs page didn't open Jobs details page of job {data[0]}")

    @test_step
    def select_rows(self):
        """Select rows in table"""
        self.navigator.navigate_to_reports()
        ManageReport(self.admin_console).view_schedules()
        schedule = ManageSchedules(self.admin_console)
        schedule_name = self.table.get_column_data('Name')
        schedule.enable_entity([schedule_name[0]])  # enable works only when select works

    @test_step
    def apply_filter_over_column(self):
        """apply filter on given column"""
        self.table.apply_filter_over_column('Name', self.client_name)
        server_name = self.table.get_column_data('Name')
        if not server_name or server_name[0] != self.client_name:
            raise CVTestStepFailure(
                f"Filter on column failed, expected [{ self.client_name}],"
                f"received [{server_name[0]}]"
            )

    @test_step
    def apply_filter_over_integer_column(self):
        """apply filter on integer type column"""
        self.admin_console.refresh_page()
        job_id = self.table.get_column_data('Job Id')
        self.table.apply_filter_over_integer_column('Job Id', job_id[0])
        job_id = self.table.get_column_data('Job Id')
        if not job_id or len(job_id) !=1 :
            raise CVTestStepFailure(
                f"Filter on column failed, expected only job [{job_id[0]}],"
                f"but received [{job_id}]"
            )

    @test_step
    def clear_column_filter(self):
        """clear filter from column"""
        self.table.clear_column_filter('Name')
        server_name = self.table.get_column_data('Name')
        if len(server_name) == 1:
            raise CVTestStepFailure(
                f"Clear filter on column failed, expected length of filter to be more than 1"
            )

    @test_step
    def get_table_data(self):
        """data in the table"""
        data = self.table.get_table_data()
        if not data:
            raise CVTestStepFailure(
                f"unable to get data from table"
            )

    @test_step
    def is_entity_present_in_column(self):
        """Check entity present"""
        if not self.table.is_entity_present_in_column('Name', self.client_name):
            raise CVTestStepFailure('unable to check entity present in column')

    @test_step
    def apply_filter_over_column_selection(self):
        """Apply filter over column list"""
        self.navigator.navigate_to_jobs()
        jobs = Jobs(self.admin_console)
        jobs.access_job_history()
        operation = self.table.get_column_data('Agent type')
        self.table.apply_filter_over_column_selection('Agent type', operation[0])
        operation1 = self.table.get_column_data('Agent type')
        if len(list(set(operation1))) != 1:
            self.log.error(
                f'Expected data [{operation[0]}] , received after filter {list(set(operation1))}'
            )
            raise CVTestStepFailure('unable to verify apply column over filter list functionality')

    def run(self):
        try:

            self.init_tc()
            self.get_number_of_columns()
            self.get_visible_column_names()
            self.get_column_data()
            self.access_action_item()
            self.view_by_title()
            self.access_toolbar_menu()
            self.apply_filter_over_column()
            self.clear_column_filter()
            self.get_table_data()
            self.is_entity_present_in_column()
            self.access_link()
            self.access_link_by_column()
            self.access_context_action_item()
            self.select_rows()
            self.apply_filter_over_column_selection()
            self.apply_filter_over_integer_column()

        except Exception as err:
            handle_testcase_exception(self, err)
        finally:
            AdminConsole.logout_silently(self.admin_console)
            Browser.close_silently(self.browser)
