# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    run()           --  run function of this test case

    setup()         -- provides access to destination cloud

    cleanup()       -- deletes pre existing contents from previous run

    wait_for_job_completion()    -- returns job id of the completed job

    create_ali_client()         -- Creates object store with provided inputs

    create_content_group()      -- Created content group for the Object store

    backup()             -- runs full backup for the specified content group

    in_place_restore()      --  runs inplace restore for specified content group

    out_place_restore()     --  runs out of place restore for specified content group

    disk_restore()          -- runs restore to disk for specified content group

    delete_client()         -- deletes client created

"""

from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils.idautils import CommonUtils

from Web.AdminConsole.adminconsole import AdminConsole
from Web.AdminConsole.ObjectStorage.object_storage import ObjectStorage
from Web.AdminConsole.ObjectStorage.object_details import ObjectDetails
from Web.Common.cvbrowser import BrowserFactory, Browser

from Web.Common.exceptions import CVTestCaseInitFailure, CVTestStepFailure
from Web.Common.page_object import TestStep, handle_testcase_exception

from AutomationUtils.machine import Machine
from Application.CloudStorage.s3_helper import S3Helper


class TestCase(CVTestCase):
    """Class for Command center Acceptance Test case for Ali Cloud """

    test_step = TestStep()

    def __init__(self):
        """Initializes test case class object"""
        super(TestCase, self).__init__()
        self.name = "Command center Acceptance - Test case for Ali Cloud"
        self.browser = None
        self.admin_console = None
        self.navigator = None
        self.obj_storage = None
        self.obj_details = None
        self.ali_client_name = 'AliClient'
        self.content_grp_name = 'group1'
        self.session = None
        self.machine = None
        self.s3_helper = None
        self.tcinputs = {
            "proxy": None,
            "plan": None,
            "access_key": None,
            "secret_key": None,
            "bucket_content": None,
            "dest_cloud": None,
            "dest_cloud_path": None,
            "dest_fs_client": None,
            "dest_fs_path": None,
            "access_key_s3": None,
            "secret_key_s3": None,
            "url": None
        }
        self.out_place_files = 0

    def setup(self):
        """Initializes pre-requisites for this test case"""
        self.machine = Machine(self.tcinputs['dest_fs_client'], self.commcell)
        self.s3_helper = S3Helper(self)
        self.session = self.s3_helper.create_session_s3(
            self.tcinputs["access_key_s3"],
            self.tcinputs["secret_key_s3"],
            'us-east-1'
        )

    def cleanup(self):
        """cleanup the test case pre requisites"""
        self.s3_helper.delete_contents_s3(self.session, [self.tcinputs['dest_cloud_path']])
        self.machine.remove_directory(self.tcinputs['dest_fs_path'])
        if self.commcell.clients.has_client(self.ali_client_name):
            self.commcell.clients.delete(self.ali_client_name)

    def init_tc(self):
        """ Initial configuration for the test case. """
        try:
            self.browser = BrowserFactory().create_browser_object()
            self.browser.open()
            self.admin_console = AdminConsole(self.browser, self.commcell.webconsole_hostname)
            self.admin_console.login(
                username=self.inputJSONnode['commcell']['commcellUsername'],
                password=self.inputJSONnode['commcell']['commcellPassword'],
                stay_logged_in=True)
            self.navigator = self.admin_console.navigator
            self.navigator.navigate_to_object_storage()
            self.obj_storage = ObjectStorage(self.admin_console)
            self.obj_details = ObjectDetails(self.admin_console)
        except Exception as exception:
            raise CVTestCaseInitFailure(exception)from exception

    def wait_for_job_completion(self, jobid):
        """Returns job id of completed job
        Args:
            jobid   (int)   id of completed job"""
        job_obj = self.commcell.job_controller.get(jobid)
        return job_obj.wait_for_completion()

    @test_step
    def create_ali_client(self):
        """Creates Ali Client    
        """
        self.obj_storage.add_ali_client(
            self.ali_client_name, self.tcinputs["proxy"],
            self.tcinputs["plan"],
            self.tcinputs["access_key"],
            self.tcinputs["secret_key"],
            self.tcinputs["url"],
        )
        if self.ali_client_name not in self.obj_storage.get_clients():
            raise CVTestStepFailure(
                f"Client [{self.ali_client_name}] doesn't exist after creation"
            )

    @test_step
    def create_content_group(self):
        """create content group"""
        self.obj_details.create_content_group(
            self.content_grp_name, self.tcinputs["plan"], self.tcinputs["bucket_content"]
        )
        if self.content_grp_name not in self.obj_details.get_content_groups():
            raise CVTestStepFailure(
                f"Content group [{self.content_grp_name}] doesn't exist after creation"
            )

    @test_step
    def backup(self):
        """submits backup job"""
        jobid = self.obj_details.submit_backup(self.content_grp_name)
        job_status = self.wait_for_job_completion(jobid)
        if not job_status:
            raise CVTestStepFailure(f"Backup job {jobid} didn't succeed")
        commonutils = CommonUtils(self.commcell)
        commonutils.backup_validation(jobid, 'Full')

    @test_step
    def in_place_restore(self):
        """Submits inplace restore"""
        self.browser.driver.refresh()
        self.admin_console.wait_for_completion()
        self.obj_details.access_restore(self.content_grp_name)
        rstr_obj = self.obj_details.submit_restore(
            self.tcinputs["bucket_content"].split('/')[1:]
        )
        jobid = rstr_obj.in_place()
        job_status = self.wait_for_job_completion(jobid)
        if not job_status:
            raise CVTestStepFailure(f"inplace restore job {jobid} didn't succeed")
        job_obj = self.commcell.job_controller.get(jobid)
        if not job_obj.num_of_files_transferred > 0:
            raise CVTestStepFailure(f"inplace restore job {jobid} success files is 0")

    @test_step
    def out_of_place_restore(self):
        """verify out of place restore"""

        rstr_obj = self.obj_details.submit_restore(
            self.tcinputs["bucket_content"].split('/')[1:]
        )
        jobid = rstr_obj.out_of_place(
            self.tcinputs["dest_cloud"], self.tcinputs["dest_cloud_path"]
        )
        job_status = self.wait_for_job_completion(jobid)
        if not job_status:
            raise CVTestStepFailure(f"out of place restore job {jobid} didn't succeed")
        job_obj = self.commcell.job_controller.get(jobid)
        if not job_obj.num_of_files_transferred > 0:
            raise CVTestStepFailure(f"out of place restore job {jobid} success files is 0")
        self.out_place_files = job_obj.num_of_files_transferred

    @test_step
    def disk_restore(self):
        """Verify disk restore"""
        rstr_obj = self.obj_details.submit_restore(
            self.tcinputs["bucket_content"].split('/')[1:]
        )
        jobid = rstr_obj.to_disk(self.tcinputs["dest_fs_client"], self.tcinputs["dest_fs_path"])
        job_status = self.wait_for_job_completion(jobid)
        if not job_status:
            raise CVTestStepFailure(f"Restore to disk job {jobid} didn't succeed")
        path = self.tcinputs["dest_fs_path"] + self.tcinputs["bucket_content"]
        if not self.machine.check_directory_exists(path):
            raise CVTestStepFailure(
                f"Disk Restore path [{path}] is empty after restore job [{jobid}"
            )
        job_obj = self.commcell.job_controller.get(jobid)
        if not job_obj.num_of_files_transferred > 0:
            raise CVTestStepFailure(f"Disk restore job {jobid} success files is 0")
        if self.out_place_files != job_obj.num_of_files_transferred:
            raise CVTestStepFailure(
                f"Disk restore file count {self.out_place_files} not matching with "
                f"disk restore files count {job_obj.num_of_files_transferred}"
            )

    @test_step
    def delete_client(self):
        """verifies client deletion"""
        self.navigator.navigate_to_object_storage()
        self.obj_storage.access_account(self.ali_client_name)
        self.obj_details.delete_account()
        if self.ali_client_name in self.obj_storage.get_clients():
            raise CVTestStepFailure(
                f"Client [{self.ali_client_name}] exist after deletion"
            )

    def run(self):
        try:
            self.cleanup()
            self.init_tc()
            self.create_ali_client()
            self.obj_storage.access_account(self.ali_client_name)
            self.create_content_group()
            self.backup()
            self.in_place_restore()
            self.out_of_place_restore()
            self.disk_restore()
            self.delete_client()
        except Exception as err:
            handle_testcase_exception(self, err)
        finally:
            AdminConsole.logout_silently(self.admin_console)
            Browser.close_silently(self.browser)
