# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()                                            --  initialize TestCase class
    init_tc()                                             --  Initialize pre-requisites
    run_backup()                                          --  Run backup job
    run_send_log()                                        -- run send log for job id return by backup job
    verify_cs_client_media_agent_in_send_log_bundle()     -- verify inside send log bundle log from client, comm server,
                                                             media agent is present or not
    verify_send_log_for_client()                          -- verify different log in side log collected from client
    verify_send_log_for_media_agent()                     -- verify different log in side log collected from media agent
    verify_send_log_for_cs()                              -- verify different log in side log collected from comm server
    run()                                                 --  run function of this test case

Input Example:

    "testCases":
            {
                "57821":
                        {
                           "ClientName": "client_test",
                           "client_display_name": "client_test_DN",
                           "AgentName": "File System",
                           "BackupsetName": "defaultBackupSet",
                           "SubclientName" : "mkp",
                           "BackupLevel": "FULL"
                        }
            }


"""
import time
import os
from AutomationUtils.machine import Machine
from Web.Common.page_object import handle_testcase_exception
from AutomationUtils.cvtestcase import CVTestCase
from Web.Common.cvbrowser import BrowserFactory, Browser
from Web.Common.page_object import TestStep
from Web.AdminConsole.AdminConsolePages.Jobs import Jobs
from Web.AdminConsole.adminconsole import AdminConsole
from Web.AdminConsole.AdminConsolePages.send_logs import SendLogs
from Web.Common.exceptions import CVTestCaseInitFailure, CVTestStepFailure
from Reports.SendLog.utils import SendLogUtils
from FileSystem.FSUtils.fshelper import FSHelper


class TestCase(CVTestCase):
    """send logs by JobID"""
    test_step = TestStep()

    def __init__(self):
        """Initializes test case class object

            Properties to be initialized:

                name            (str)       --  name of this test case

                tcinputs        (dict)      --  test case inputs with input name as dict key
                                                and value as input type


        """
        super(TestCase, self).__init__()
        self.browser = None
        self.admin_console = None
        self.send_log = None
        self.sjob_id = None
        self.machine = None
        self.path = None
        self.job = None
        self.media_agent = None
        self.commcell_name = None
        self.navigator = None
        self.send_log_utils = None
        self.commserv_client = None
        self.directory = None
        self.name = "send logs by JobID"
        self.tcinputs = {
            "ClientName": None,
        }

    def init_tc(self):
        """
            Initial configuration for the test case
            """
        try:
            self.log.info("Initializing pre-requisites")
            self.browser = BrowserFactory().create_browser_object()
            self.browser.open()
            self.admin_console = AdminConsole(self.browser, self.commcell.webconsole_hostname)
            self.admin_console.login(self.inputJSONnode['commcell']["commcellUsername"],
                                     self.inputJSONnode['commcell']["commcellPassword"])
            self.navigator = self.admin_console.navigator
            self.media_agent = self.subclient.storage_ma
            self.commcell_name = self.commcell.commserv_name
            self.commserv_client = self.commcell.commserv_client
            self.machine = Machine(self.commserv_client)
            self.send_log_utils = SendLogUtils(self, self.machine)
            self.directory = self.send_log_utils.create_directory_for_given_path("TC55121")
            self.send_log = SendLogs(self.admin_console)
        except Exception as exception:
            raise CVTestCaseInitFailure(exception) from exception

    @test_step
    def run_backup(self):
        """
        Run Backup
          """
        fs_helper = FSHelper(self)
        job = fs_helper.run_backup(backup_level=self.tcinputs['BackupLevel'])
        self.sjob_id = str(job[0].job_id)
        job_obj = self.commcell.job_controller.get(self.sjob_id)
        job_status = job_obj.wait_for_completion()
        if not job_status:
            raise CVTestStepFailure(
                f"Send log job id [{self.sjob_id}] failed"
            )

    @test_step
    def run_send_log(self):
        """Running SendLog job"""
        self.navigator.navigate_to_jobs()
        self.job = Jobs(self.admin_console)
        if not self.job.if_job_exists(self.sjob_id):
            self.job.access_job_history()
        self.job.access_send_log_job_by_id(self.sjob_id)
        self.send_log.select_local_output(local_path=self.directory)
        advanced_list = self.send_log.Advancedlist
        self.send_log.deselect_advanced(advanced_list=[advanced_list.SCRUB_LOGFILES])
        info_list = self.send_log.Informationlist
        self.send_log.select_information(information_list=[info_list.OS_LOGS,
                                                      info_list.LOGS])
        self.sjob_id = self.send_log.submit()
        job_obj = self.commcell.job_controller.get(self.sjob_id)
        job_status = job_obj.wait_for_completion()
        if not job_status:
            raise CVTestStepFailure(
                f"Send log job id [{self.sjob_id}] failed"
            )

    @test_step
    def verify_cs_client_media_agent_in_send_log_bundle(self):
        """ To unzip send log bundle and verifying cs, client, media agent bundle """
        self.path = self.send_log_utils.unzip_send_log_file(self.commserv_client, self.path)
        self.send_log_utils.log_file_verification_with_full_name(
            self.path, [self.tcinputs['ClientName'] + '.7z', self.commcell_name + '.7z',
                        self.media_agent + '.7z'])

    @test_step
    def verify_send_log_for_client(self):
        """Verifying send logs collected from client"""
        self.log.info("Unzipping client bundle and checking logs: clbackup.log ")
        path = os.path.join(self.path, self.tcinputs['ClientName'] + '.7z')
        path = self.send_log_utils.unzip_send_log_file(self.commserv_client, path)
        self.send_log_utils.log_file_verification_with_full_name(path, ['clBackup.log'])

    @test_step
    def verify_send_log_for_media_agent(self):
        """Verifying send logs collected from media agent """
        self.log.info("Unzipping media agent bundle and checking logs: archiveIndex.log ")
        path = os.path.join(self.path, self.media_agent + ".7z")
        path = self.send_log_utils.unzip_send_log_file(self.commserv_client, path)
        self.send_log_utils.log_file_verification_with_full_name(path, ['archiveIndex.log'])

    @test_step
    def verify_send_log_for_cs(self):
        """Verifying send logs collected from comm server"""
        self.log.info("Unzipping comm server bundle and checking logs: JobManager.log ")
        path = os.path.join(self.path, self.commcell_name + '.7z')
        path = self.send_log_utils.unzip_send_log_file(self.commserv_client, path)
        self.send_log_utils.log_file_verification_with_full_name(path, ['JobManager.log'])

    def run(self):
        try:
            self.run_backup()
            self.init_tc()
            self.run_send_log()

            self.log.info('Waiting for 20 sec to check file present at location ' + self.directory +
                          ' for send log job id ' + self.sjob_id)
            time.sleep(20)
            self.path = self.send_log_utils.is_send_log_bundle_exist(self.directory,
                                                                     self.sjob_id)
            self.verify_cs_client_media_agent_in_send_log_bundle()
            self.verify_send_log_for_client()
            self.verify_send_log_for_media_agent()
            self.verify_send_log_for_cs()
            self.machine.remove_directory(self.directory)
        except Exception as exp:
            handle_testcase_exception(self, exp)
        finally:
            AdminConsole.logout_silently(self.admin_console)
            Browser.close_silently(self.browser)
