# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    setup()         --  setup function of this test case

    run()           --  run function of this test case

    tear_down()     --  tear down function of this test case

"""
import calendar
import time
import datetime

from AutomationUtils import constants
from AutomationUtils.cvtestcase import CVTestCase

from dynamicindex.Datacube.dcube_sync_helper import SyncHelper
from dynamicindex.Datacube.dcube_solr_helper import SolrHelper
from dynamicindex.utils import constants as dynamic_constants


class TestCase(CVTestCase):
    """Class for executing this test case"""

    def __init__(self):
        """Initializes test case class object

            Properties to be initialized:

                name            (str)       --  name of this test case

                tcinputs    (dict)          --  test case inputs with input name as dict key
                                                and value as input type

        """
        super(TestCase, self).__init__()
        self.datasource_type = dynamic_constants.OPEN_DATASOURCE_DSTYPE
        self.datasource_id = None
        self.datasource_obj = None
        self.timestamp = None
        self.crawl_jobid = None
        self.properties = None
        self.schema = []
        self.coreid = None
        self.solr_core_name = None
        self.total_crawlcount = 0
        self.dcube_helper = None
        self.input_data = []
        self.open_ds_prefix = "DC_open_"
        self.sync_helper = None
        self.name = "DataCube open DS sync stats to CS database and cross verify max stats"
        self.tcinputs = {
            "DataSourcename": None,
            "IndexServer": None,
            "Column": None
        }

    def setup(self):
        """Setup function of this test case"""
        try:
            self.sync_helper = SyncHelper(self)
            self.dcube_helper = SolrHelper(self)
            self.timestamp = calendar.timegm(time.gmtime())
            self.tcinputs['DataSourcename'] = self.tcinputs['DataSourcename'] + \
                str(self.timestamp)
            self.log.info("Going to create Open Datasource via Rest API : %s",
                          str(self.tcinputs['DataSourcename']))

            prop_name = ['candelete', 'appname']
            prop_value = ["true", "DATACUBE"]
            self.properties = [{"propertyName": prop_name[x], "propertyValue": prop_value[x]}
                               for x in range(0, len(prop_name))]
            self.commcell.datacube.datasources.add(
                self.tcinputs['DataSourcename'],
                self.tcinputs['IndexServer'],
                self.datasource_type, self.properties)
            self.log.info("Open datasource created successfully")

            self.datasource_obj = self.commcell.datacube.datasources.get(
                self.tcinputs['DataSourcename'])

            self.datasource_id = self.datasource_obj.datasource_id
            self.log.info("Created DataSource id : %s", str(self.datasource_id))

            Fieldnames = self.tcinputs['Column'].split(",")
            fieldlist = {
                "fieldName": "",
                "type": "string",
                "indexed": True,
                "stored": True,
                "multiValued": False,
                "searchDefault": True,
                "autocomplete": False
            }
            for x in Fieldnames:
                fieldlist['fieldName'] = x
                self.schema.append((fieldlist))
                self.log.info("Schema formed : %s", str(self.schema))
                self.log.info("Calling update schema API to create column : %s", str(x))
                self.datasource_obj.update_datasource_schema(self.schema)
                self.schema.clear()

            self.log.info("Calling import data on this datasource")
            for x in range(10):
                fieldlist = {
                    Fieldnames[0]: str(x),
                    Fieldnames[1]: "DatacubeAutomationUser_" + str(x)
                }
                self.total_crawlcount = self.total_crawlcount + 1
                self.input_data.append((fieldlist))

            self.log.info("Import Data formed : %s", str(self.input_data))
            self.datasource_obj.import_data(self.input_data)
            self.log.info("Import Data done successfully")
            self.log.info("Total document count : %s", str(self.total_crawlcount))

        except Exception as ee:
            self.log.error('Setup for the test case failed. Error:')
            self.log.exception(ee)
            self.result_string = str(ee)
            self.status = constants.FAILED
            raise Exception("Test case setup(Creation of Datasource failed). Please check")

    def run(self):
        """Run function of this test case"""
        try:

            current_time = datetime.datetime.now()
            self.sync_helper.set_dcube_sync_registry(current_time, 2, 24, True)
            statusxml = self.sync_helper.get_syncxml_by_csrestart(self.datasource_id)
            self.coreid = self.dcube_helper.get_coreid_datasource(self.datasource_id)
            self.solr_core_name = f"{self.open_ds_prefix}{self.tcinputs['DataSourcename']}_{str(self.coreid)}"
            self.log.info("Core id for this Datasource is : %s", str(self.coreid))

            solr_url = self.dcube_helper.get_solr_baseurl(self.tcinputs['IndexServer'], 1)
            self.log.info("Base solr URl : %s", str(solr_url))

            stats_response = self.dcube_helper.get_corestats(solr_url, self.solr_core_name)
            self.log.info("Core stats response : %s", str(stats_response))

            # validation starts
            testresult = self.sync_helper.verify_sync_xml(statusxml, stats_response,
                                                          str(int(current_time.timestamp())),
                                                          None)
            if testresult:
                self.log.info("Sync xml validation passed")
            else:
                raise Exception("Sync xml validation failed")

            # Delete the content from datasource and push less rows than before
            old_statusxml = statusxml
            old_synctime = str(int(current_time.timestamp()))
            self.total_crawlcount = 0
            self.log.info("Call delete content on this datasource")
            self.datasource_obj.delete_content()
            self.input_data.clear()

            Field_names = self.tcinputs['Column'].split(",")
            for x in range(1):
                fieldlist = {
                    Field_names[0]: str(x),
                    Field_names[1]: "DatacubeAutomationUser_" + str(x)
                }
                self.total_crawlcount = self.total_crawlcount + 1
                self.input_data.append((fieldlist))

            self.log.info("Import Data formed : %s", str(self.input_data))
            self.datasource_obj.import_data(self.input_data)
            self.log.info("Import Data done successfully")
            self.log.info("Total document count : %s", str(self.total_crawlcount))

            self.log.info("Calling commit on datasource")
            core_solrurl = solr_url + "/" + self.solr_core_name
            self.dcube_helper.do_hard_commit(core_solrurl)

            current_time = datetime.datetime.now()
            self.sync_helper.set_dcube_sync_registry(current_time, 2, 24, True)
            statusxml = self.sync_helper.get_syncxml_by_csrestart(self.datasource_id)

            stats_response = self.dcube_helper.get_corestats(solr_url, self.solr_core_name)
            self.log.info("Core stats response : %s", str(stats_response))

            # validation starts for max stats
            maxstats = self.sync_helper.get_core_maxstats(
                old_statusxml, stats_response, old_synctime, str(int(current_time.timestamp())))

            testresult = self.sync_helper.verify_sync_xml(
                statusxml, stats_response, str(int(current_time.timestamp())), None, maxstats
            )
            if testresult:
                self.log.info("Sync xml validation passed")
            else:
                raise Exception("Sync xml validation failed")

        except Exception as exp:
            self.log.error('Failed to execute test case with error: ')
            self.result_string = str(exp)
            self.status = constants.FAILED
            self.log.exception(exp)

    def tear_down(self):
        """Tear down function of this test case"""
        if self.datasource_obj is not None and self.status == constants.PASSED:
            self.log.info("Delete the data source with id = %s", str(self.datasource_id))
            self.datasource_obj.delete_datasource()
