# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
""""Main file for executing this test case

TestCase is the only class definied in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  Initializes test case class object

    setup()         --  Setup function for this testcase

    tear_down()     --  Tear down function to delete automation generated data

    run()           --  Main function for test case executions

"""
from time import time
from AutomationUtils import constants
from AutomationUtils.cvtestcase import CVTestCase
from Application.AD.ms_azuread import AzureAd, CvAzureAd
from Application.AD.exceptions import ADException

class TestCase(CVTestCase):
    """ class to run agent basic testing for Azure ad agent"""

    def __init__(self):
        """ initial class
        Properties to be initialized:
        name            (str)        -- name of this test case
        applicable_os   (str)    -- applicable os for this test case
        product         (str)    -- applicable product for AD
        """
        super().__init__()
        self.name = "Azure AD agent basic backup/restore"
        self.applicable_os = self.os_list.WINDOWS
        self.product = self.products_list.ACTIVEDIRECTORY
        self.feature = self.features_list.DATAPROTECTION
        self.show_to_user = True
        self.tcinputs = {
            "ClientName" : None,
            "AgentName" : "Azure AD",
            }
        self.aad_ins = None
#       we can pick any objects type from ["user", "group", "reg_app", "ent_app"]
        self.aad_types = ['user']
        self.subclient = None
        self.aad_ins = None
        self.cv_aad_ins = None

    def setup(self):
        """ prepare the setup environment"""
        aad_credential = [self.tcinputs['TenantName'],
                          self.tcinputs['AdminUser'],
                          self.tcinputs['AdminPassword'],
                          self.tcinputs['ClientId'],
                          self.tcinputs['ClientPass']]
        if "types" in self.tcinputs:
            self.aad_types = self.tcinputs['types']
            self.log.info(f"will use obj types in answer file {self.aad_types}")
        self.aad_ins = AzureAd(*aad_credential, self.log)
        self.subclient = self._backupset.subclients.get("default")
        self.cv_aad_ins = CvAzureAd(self.aad_ins, self.subclient)
        self.log.info("get default subclient instance with id %s" % \
                      self.subclient.subclient_id)
        self.clean_up(phase="all")
        self.aad_ins.deleted_obj_clean(self.aad_types)

    def run(self):
        """ run test case steps"""
        try:
            # check full job bakcup/restore
            aad_objs_batch0 = self.aad_ins.group_objs_create(types=self.aad_types,
                                                             prestring="auto_init")
            aad_objs_ini = self.aad_ins.group_objs_check(types=self.aad_types)
            # create static objects , run initila backup job
            self.log.info("Start the first full backup job")
            b_f_job = self.subclient.backup(backup_level="full")
            self.log.debug("full backup job %s started" % b_f_job.job_id)
            jobresult = b_f_job.wait_for_completion()
            if not jobresult:
                raise ADException("testcase", self._id,
                                  f"full backup job result is {b_f_job.summary['pendingReason']}.")
            self.log.debug(f"""
backup job {b_f_job.job_id} completed. job result is {jobresult}""")

            self.cv_aad_ins.cv_obj_browse_check(aad_objs_ini, b_f_job)
            # delete objects and try to restore obj
            compare_result = self.cv_aad_ins.cv_obj_delete_restore(aad_objs_batch0,
                                                                   types=self.aad_types)
            self.log.info(f"here is the compare result {compare_result}")
            for type_ in self.aad_types:
                if compare_result[type_][1]:
                    self.log.debug("{} object with id {} was restored successuflly".\
                                   format(type_, aad_objs_batch0[type_].object_id))
                else:
                    self.log.info("{} delete and restore doesn't find match result".\
                                  format(type_))
                    raise ADException("testcase", self._id,
                                      "compare result after full is not correct")
            # check inc job backup/restore
            timestamp = int(time())
            aad_objs_batch1 = self.aad_ins.group_objs_create(types=self.aad_types,
                                                             prestring=f"auto_inc_{timestamp}")
            aad_objs_inc = self.aad_ins.group_objs_check(types=self.aad_types)
            self.log.info("Start an incrmeantal backup job ")
            b_in_job = self.subclient.backup()
            self.log.debug(f"incremental job {b_in_job.job_id} started")
            jobresult = b_in_job.wait_for_completion()
            if not jobresult:
                raise ADException("testcase", self._id,
                                  f"inc backup job result is {b_in_job.summary['pendingReason']}.")
            self.log.info(f"backup job {b_in_job.job_id} completed. job result is {jobresult}")
            self.cv_aad_ins.cv_obj_browse_check(aad_objs_inc, b_in_job)
            compare_result = self.cv_aad_ins.cv_obj_delete_restore(aad_objs_batch1,
                                                                   types=self.aad_types)
            self.log.debug(f"Here is the compare result after regular delete {compare_result}")
            for type_ in self.aad_types:
                if compare_result[type_][1]:
                    self.log.info("%s object with id %s was restored successuflly" % \
                                  (type_, aad_objs_batch1[type_]))
                else:
                    self.log.info("%s delete and restore doesn't find match result" % \
                                  type_)
                    raise ADException("testcase", self._id,
                                      "compare result after inc is not correct")
            compare_result = self.cv_aad_ins.cv_obj_delete_restore(aad_objs_batch1,
                                                                   types=self.aad_types,
                                                                   harddelete=True)
            self.log.debug(f"Here is the compare result after hard delete {compare_result}")
            for type_ in self.aad_types:
                if compare_result[type_][1]:
                    self.log.info("""
%s delete and restore doesn't find match result, it works in hard delete case""" % \
                                  type_)
                else:
                    self.log.info("%s object with id %s was restored successuflly" % \
                                  (type_, aad_objs_batch1[type_]))
                    raise ADException("testcase", self._id,
                                      "compare result after inc is not correct with hard delete")
            # Check inc job attribute change backup/restore
            timestamp = int(time())
            objs = self.aad_ins.deleted_obj_list("user")
            self.log.debug(f"here is the objects deletted : {objs}")
            aad_objs_batch2 = self.aad_ins.group_objs_create(types=self.aad_types,
                                                             prestring=f"auto_change_{timestamp}")
            changevalue = self.cv_aad_ins.cv_obj_change(aad_objs_batch2,
                                                        types=self.aad_types)
            self.log.info("Start an incrmeantal backup job Before make attribute chagne")
            b_change_base_job = self.subclient.backup()
            self.log.debug(f"incremental job {b_change_base_job.job_id} started")
            jobresult = b_change_base_job.wait_for_completion()
            if not jobresult:
                raise ADException("testcase", self._id,
                                  "inc backup job result is {}.".\
                                  format(b_change_base_job.summary['pendingReason']))
            self.log.info("backup job {} completed. job result is {}".\
                          format(b_change_base_job.job_id, jobresult))
            self.cv_aad_ins.cv_obj_change_browse_check(aad_objs_batch2,
                                                       changevalue,
                                                       b_change_base_job)
            self.cv_aad_ins.cv_obj_change(aad_objs_batch2,
                                          types=self.aad_types)
            b_change_new_job = self.subclient.backup()
            self.log.debug(f"incremental job {b_change_new_job.job_id} started")
            jobresult = b_change_new_job.wait_for_completion()
            if not jobresult:
                raise ADException("testcase", self._id,
                                  "inc backup job result is {}.".\
                                  format(b_change_new_job.summary['pendingReason']))
            self.log.info("backup job {} completed. job result is {}".\
                          format(b_change_new_job.job_id, jobresult))
            self.cv_aad_ins.cv_obj_change_restore(aad_objs_batch2,
                                                  b_change_base_job)
            self.status = constants.PASSED
            self.log.info(f"Test case completed without issue")
        except ADException as exp:
            self.status = constants.FAILED
            self.log.exception("There is excpetion happened, here is the detail %s" % \
                                exp.report)
            self.result_string = exp.report
        except Exception as exp:
            self.status = constants.FAILED
            self.log.exception("There is not AD excpetion happened, here is the detail %s" % \
                                exp)

    def tear_down(self):
        """tear down when the case is completed, include error handle"""
        # check existing aad objects
        if self.status == constants.PASSED:
            self.clean_up(phase="all")

    def clean_up(self, phase):
        """clean up all object created in the script"""
        # remove all static objects
        if phase == "all":
            for type_ in self.aad_types:
                for phase_ in ['init', "full", "change", "inc"]:
                    self.aad_ins.type_operation(type_,
                                                "delete",
                                                **{"displayname": [f"auto_{phase_}"]})
        else:
            for type_ in self.aad_types:
                self.aad_ins.type_operation(type_,
                                            "delete",
                                            **{"displayname": [f"auto_{phase}"]})
