# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:

    __init__()      --  initialize TestCase class

    _get_storage_dedupe_flags() --  To get default dedupe flags set on SIDB store of given disk storage

    _get_storage_flags_not_set()--  To know which all flags are not set that should be set by default on a storage

    _get_storage_flags_set()    --  To know which all dedupe flag are set that should not be set by default on a storage

    _get_copy_dedupe_flag()     --  To get default dedupe flags set on archGroupCopy table of given plan and copy

    _get_copy_flags_not_set()   --  To know which all dedupe flags are not set that should be set by default on a copy

    _get_copy_flags_set()       --  To know which all dedupe flags are set that should not be set by default on a copy

    _get_retention_period()     --  To get retention set on copy from archAgingRule table for given plan and copy

    _validate_auxcopy_association() --  To validate if  Auxiliary Copy schedule is associated to plan secondary copy

    setup()         --  setup function of this test case

    run()           --  run function of this test case

Sample Input:
"54620": {
            "ClientName": "skclient",
            "AgentName": "File System",
            "MediaAgent1": "skma"
    }
"""

from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils.options_selector import OptionsSelector
from Web.Common.cvbrowser import BrowserFactory, Browser
from Web.Common.page_object import TestStep, handle_testcase_exception
from Web.Common.exceptions import CVTestCaseInitFailure, CVTestStepFailure
from Web.AdminConsole.adminconsole import AdminConsole
from Web.AdminConsole.Helper.PlanHelper import PlanMain
from Web.AdminConsole.Helper.StorageHelper import StorageMain


class TestCase(CVTestCase):
    """ TestCase class used to execute the test case from here"""
    test_step = TestStep()

    def __init__(self):
        """Initializing the Test case file"""

        super(TestCase, self).__init__()
        self.name = "Admin Console: Validate default dedupe flags, retention rule, schedule policy set on plan creation"
        self.browser = None
        self.admin_console = None
        self.plan_helper = None
        self.storage_helper = None
        self.primary_storage_name = None
        self.secondary_storage_name = None
        self.primary_backup_location = None
        self.secondary_backup_location = None
        self.primary_ddb_location = None
        self.secondary_ddb_location = None
        self.plan_name = None
        self.tcinputs = {
            "MediaAgent": None
        }

    def _get_storage_dedupe_flags(self, disk_storage_name):
        """
        To get default dedupe flags set on SIDB store of given disk storage
            Args:
             disk_storage_name (str) --  Name of the disk storage

            Returns:
                list    --  contains storage dedupeflags, extended flags
        """

        query = f"""SELECT	DDB.flags, DDB.ExtendedFlags
                    FROM	IdxSIDBStore DDB
                    JOIN	archGroupCopy AGC
                            ON	DDB.SIDBStoreId = AGC.SIDBStoreId
                    JOIN	archGroup AG
                            ON	AGC.archGroupId = AG.id
                    WHERE	AG.name = '{disk_storage_name}'"""
        self.csdb.execute(query)

        return self.csdb.fetch_one_row()

    def _get_storage_flags_not_set(self, dedupe_flag, extended_flag):
        """
        To know which all flags are not set that should be set by default on a storage
            Args:
             dedupe flag (int) --  storage dedupe flag

             extended_flag (int) -- storage extended flag

            Returns:
                None
        """

        if dedupe_flag & 2 == 0:
            self.log.error('SW_COMPRESSION flag is not set')
        if dedupe_flag & 16 == 0:
            self.log.error('GLOBAL_DEDUPE flag is not set')
        if dedupe_flag & 65536 == 0:
            self.log.error('STORE_DEDUPFACTOR_ENABLED flag is not set')
        if dedupe_flag & 131072 == 0:
            self.log.error('SECONDARY_FOLLOW_SOURCE_DEDUP_BLOCK flag is not set')
        if dedupe_flag & 1048576 == 0:
            self.log.error('SILO_PREPARED flag is not set')
        if dedupe_flag & 8388608 == 0:
            self.log.error('ENABLE_DDB_VALIDATION flag is not set')
        if dedupe_flag & 536870912 == 0:
            self.log.error('PRUNING_ENABLED flag is not set')
        if extended_flag & 2 == 0:
            self.log.error('DEFAULT Extended flag is not set')
        if extended_flag & 4 == 0:
            self.log.error('MARK_AND_SWEEP_ENABLED Extended flag is not set')
        if extended_flag & 8 == 0:
            self.log.error('ZEROREF_LOGGING_ENABLED Extended flag is not set')

    def _get_storage_flags_set(self, dedupe_flag, extended_flag):
        """
        To know which all dedupe flags are set that should not be set by default on a storage
            Args:
             dedupe flag (int) --  storage dedupe flag

             extended_flag (int) -- storage extended flag

            Returns:
                None
        """

        if dedupe_flag & 1 != 0:
            self.log.error('MULTI_TAG_HEADER flag is set')
        if dedupe_flag & 4 != 0:
            self.log.error('NONTRANS_DB flag is set')
        if dedupe_flag & 8 != 0:
            self.log.error('NO_SECONDARY_TABLE flag is set')
        if dedupe_flag & 32 != 0:
            self.log.error('SINGLE_THREAD_DB flag is set')
        if dedupe_flag & 64 != 0:
            self.log.error('SIDB_SUSPENDED flag is set')
        if dedupe_flag & 128 != 0:
            self.log.error('RECYCLABLE flag is set')
        if dedupe_flag & 256 != 0:
            self.log.error('SILO_AGED flag is set')
        if dedupe_flag & 512 != 0:
            self.log.error('DDB_COPYOP_INPROGRESS flag is set')
        if dedupe_flag & 1024 != 0:
            self.log.error('DDB_MOVEOP_INPROGRESS flag is set')
        if dedupe_flag & 2048 != 0:
            self.log.error('DDB_ARCHIVE_STATUS flag is set')
        if dedupe_flag & 4096 != 0:
            self.log.error('SIDB_ENGINE_RUNNING flag is set')
        if dedupe_flag & 8192 != 0:
            self.log.error('MEMDB_DDB flag is set')
        if dedupe_flag & 16384 != 0:
            self.log.error('OPTIMIZE_DB flag is set')
        if dedupe_flag & 32768 != 0:
            self.log.error('STORE_SEALED flag is set')
        if dedupe_flag & 2097152 != 0:
            self.log.error('SILO_ENABLED flag is set')
        if dedupe_flag & 4194304 != 0:
            self.log.error('DDB_VALIDATION_FAILED flag is set')
        if dedupe_flag & 16777216 != 0:
            self.log.error('DDB_UNDER_MAINTENANCE flag is set')
        if dedupe_flag & 33554432 != 0:
            self.log.error('DDB_NEEDS_AUTO_RESYNC flag is set')
        if dedupe_flag & 67108864 != 0:
            self.log.error('DDB_VERIFICATION_INPROGRESS flag is set')
        if dedupe_flag & 134217728 != 0:
            self.log.error('TIMESTAMP_MISMATCH flag is set')
        if dedupe_flag & 268435456 != 0:
            self.log.error('DDB_VERIFICATION_INPROGRESS_ALLOW_BACKUPS flag is set')
        if dedupe_flag & 1073741824 != 0:
            self.log.error('DDB_RESYNC_IN_PROGRESS flag is set')
        if dedupe_flag & 214748368 != 0:
            self.log.error('DDB_PRUNING_IN_PROGRESS flag is set')
        if extended_flag & 1 != 0:
            self.log.error('IDX_SIDBSTORE_EX_FLAGS_FULL extended flag is set')

    def _get_copy_dedupe_flag(self, plan_name, copy_name):
        """
        To get default dedupe flags set on archGroupCopy table of given plan and copy
            Args:
             plan_name (str) --  Name of the plan
             copy_name (str) -- Name of the copy

            Returns:
                int    --  given copy related dedupe flags
        """

        query = f"""SELECT AGC.dedupeFlags
                    FROM   archGroupCopy AGC
                    JOIN   archGroup AG
                           ON     AGC.archGroupId = AG.id
                    WHERE	AG.name = '{plan_name}'
                    AND     AGC.name = '{copy_name}'"""
        self.csdb.execute(query)

        return int(self.csdb.fetch_one_row()[0])

    def _get_copy_flags_not_set(self, dedupe_flag, copy_name):
        """
        To know which all dedupe flags are not set that should be set by default on a copy
            Args:
                dedupe_flag (int) : flags set on the given copy
                copy_name   (str) : name of the copy
            Returns:
                None
        """

        if dedupe_flag & 262144 == 0:
            self.log.error('SIDB_STORE_ENABLED_FLAG flag is not set')
        if dedupe_flag & 524288 == 0 and copy_name == 'Primary':
            self.log.error('CLIENT_SIDE_DEDUP_FLAG flag is not set')
        if dedupe_flag & 8388608 == 0:
            self.log.error('USE_READLESS_MODE_DEDUP_FLAG flag is not set')
        if dedupe_flag & 33554432 == 0:
            self.log.error('RECONSTRUCT_SIDB_FROM_SNAPSHOT_FLAG flag is not set')
        if dedupe_flag & 67108864 == 0:
            self.log.error('AUTO_RECONSTRUCT_DEDUP_STORE_FLAG flag is not set')
        if dedupe_flag & 134217728 == 0:
            self.log.error('USE_GLOBAL_DEDUP_STORE_FLAG flag is not set')

    def _get_copy_flags_set(self, dedupe_flag, copy_name):
        """
        To know which all dedupe flags are set that should not be set by default on a copy
            Args:
                dedupe_flag (int) : flags set on the given copy
                copy_name   (str) : name of the copy
            Returns:
                None
        """

        if dedupe_flag & 2 != 0:
            self.log.error('ENABLE_SIGNATURE_BACKWARD_REF_FLAG flag is set')
        if dedupe_flag & 4 != 0:
            self.log.error('OPTIMIZE_HIGH_LATENCY_NETWORK_FLAG flag is set')
        if dedupe_flag & 8 != 0:
            self.log.error('SIDB_RESILIENCY_ENABLED_FLAG flag is set')
        if dedupe_flag & 131072 != 0:
            self.log.error('ENABLE_SIDB_ARCHIVE_FLAG flag is set')
        if dedupe_flag & 524288 == 0 and copy_name == 'Secondary':
            self.log.error('CLIENT_SIDE_DEDUP_FLAG flag  not set')
        if dedupe_flag & 1048576 != 0:
            self.log.error('DEDUP_INACTIVE_FLAG flag is set')
        if dedupe_flag & 2097152 != 0:
            self.log.error('BACKUP_SILO_ENABLED_FLAG flag is set')
        if dedupe_flag & 4194304 != 0:
            self.log.error('KEEP_ACTIVE_SILO_IN_CACHE_FLAG flag is set')
        if dedupe_flag & 16777216 != 0:
            self.log.error('ENABLE_SILO_DISK_SPACE_MANAGEMENT_FLAG flag is set')
        if dedupe_flag & 268435456 != 0:
            self.log.error('HOST_GLOBAL_DEDUP_STORE_FLAG flag is set')
        if dedupe_flag & 536870912 != 0:
            self.log.error('CLIENT_CACHE_DB_DIRTY_FLAG flag is set')
        if dedupe_flag & 1073741824 != 0:
            self.log.error('DASH_SOURCE_SIDE_DISK_CACHE_FLAG flag is set')

    def _get_retention_period(self, plan_name, copy_name):
        """
        To get retention set on copy from archAgingRule table for given plan and copy
        Args:
             plan_name (str) --  Name of the plan
             copy_name (str) -- Name of the copy

        Returns:
            int    --  retention days set on the copy
        """

        query = f"""SELECT	AGR.retentionDays
                    FROM	archAgingRule AGR
                    JOIN	archGroupCopy AGC
                            ON     AGR.copyId = AGC.id
                    JOIN	archGroup AG
                            ON     AGC.archGroupId = AG.id
                    WHERE	AG.name = '{plan_name}'
                    AND     AGC.name = '{copy_name}'"""
        self.csdb.execute(query)
        cur = self.csdb.fetch_one_row()
        self.log.info("RESULT: %s", cur[0])
        if cur[0] != '':
            return int(cur[0])
        raise Exception("Unable to fetch retention period")

    def _validate_auxcopy_association(self, plan_name):
        """
        To validate if System Created Autocopy schedule(Auxiliary Copy) is associated to plan secondary copy
        Args:
             plan_name (str) --  Name of the plan

        Returns:
            True  --    if schedule is associated
            False --    if schedule is not associated
        """

        query = f"""SELECT	COUNT(1)
                    FROM	TM_AssocEntity AE
                    JOIN	archGroupCopy AGC
                            ON     AE.copyId = AGC.id
                    JOIN	archGroup AG
                            ON     AGC.archGroupId = AG.id
                    WHERE	AG.name = '{plan_name}'
                    AND     AGC.name = 'Secondary'
                    AND		AE.taskId=9"""
        self.csdb.execute(query)

        if self.csdb.fetch_one_row()[0] == '0':
            return False
        return True

    def init_tc(self):
        """Initial configuration for the test case"""

        try:
            self.browser = BrowserFactory().create_browser_object()
            self.browser.open()
            self.admin_console = AdminConsole(self.browser, self.commcell.webconsole_hostname)
            self.admin_console.login(self.inputJSONnode['commcell']['commcellUsername'],
                                     self.inputJSONnode['commcell']['commcellPassword'])
        except Exception as exception:
            raise CVTestCaseInitFailure(exception) from exception

    @test_step
    def cleanup(self):
        """To perform cleanup operation"""

        try:
            # To delete plan if exists
            self.log.info('Check for plan %s', self.plan_name)
            if self.commcell.plans.has_plan(self.plan_name):
                self.log.info('Deletes plan %s', self.plan_name)
                self.plan_helper.delete_plans()
                self.log.info('Deleted plan %s', self.plan_name)
            else:
                self.log.info('No plan exists with name %s', self.plan_name)

            # To delete disk storage if exists
            self.log.info('Check for storage %s', self.primary_storage_name)
            if self.primary_storage_name in self.storage_helper.list_disk_storage():
                self.log.info('Deletes storage %s', self.primary_storage_name)
                self.storage_helper.delete_disk_storage(self.primary_storage_name)
            else:
                self.log.info('No storage exists with name %s', self.primary_storage_name)

            # To delete disk storage if exists
            self.log.info('Check for storage %s', self.secondary_storage_name)
            if self.secondary_storage_name in self.storage_helper.list_disk_storage():
                self.log.info('Deletes storage %s', self.secondary_storage_name)
                self.storage_helper.delete_disk_storage(self.secondary_storage_name)
            else:
                self.log.info('No storage exists with name %s', self.secondary_storage_name)

            self.commcell.refresh()
        except Exception as exp:
            raise CVTestStepFailure(f'Cleanup operation failed with error : {exp}')

    @test_step
    def create_entities(self):
        """To create required entities for test case"""

        ma_name = self.commcell.clients.get(self.tcinputs['MediaAgent']).display_name

        # To create a new disk storage
        self.log.info("Adding a new disk for primary storage: %s", self.primary_storage_name)
        self.storage_helper.add_disk_storage(
            self.primary_storage_name,
            ma_name,
            self.primary_backup_location,
            deduplication_db_location=self.primary_ddb_location)

        self.log.info("Adding a new disk for secondary storage: %s", self.secondary_storage_name)
        self.storage_helper.add_disk_storage(
            self.secondary_storage_name,
            ma_name,
            self.secondary_backup_location,
            deduplication_db_location=self.secondary_ddb_location)

        # To create a new plan
        self.log.info("Adding a new plan: %s", self.plan_name)
        self.plan_helper.add_plan()
        self.log.info("successfully created plan: %s", self.plan_name)

    @test_step
    def check_dedupe_flags(self):
        """ To validate all default dedupe flags on storage and plan"""

        # To validate if default dedupe flags are set or not on the primary disk storage.
        self.log.info("Validate if default dedupe flags are set or not on primary disk storage: %s",
                      self.primary_storage_name)
        dedupe_flag, extended_flag = map(int, self._get_storage_dedupe_flags(self.primary_storage_name))
        if dedupe_flag == 546504722 and extended_flag == 14:
            self.log.info('All the default dedupe flags are set on primary disk storage')
            self.log.info("Validation for default dedupe flags on primary disk storage: %s was successful",
                          self.primary_storage_name)
        else:
            self._get_storage_flags_not_set(dedupe_flag, extended_flag)
            self._get_storage_flags_set(dedupe_flag, extended_flag)
            raise CVTestStepFailure('The default dedupe flags are incorrectly set on  primary disk storage')

        # To validate if default dedupe flags are set or not on the secondary disk storage.
        self.log.info("Validate if default dedupe flags are set or not on secondary disk storage: %s",
                      self.secondary_storage_name)
        dedupe_flag, extended_flag = map(int, self._get_storage_dedupe_flags(self.secondary_storage_name))
        if dedupe_flag == 546504722 and extended_flag == 14:
            self.log.info('All the default dedupe flags are set on secondary disk storage')
            self.log.info("Validation for default dedupe flags on secondary disk storage: %s was successful",
                          self.secondary_storage_name)
        else:
            self._get_storage_flags_not_set(dedupe_flag, extended_flag)
            self._get_storage_flags_set(dedupe_flag, extended_flag)
            raise CVTestStepFailure('The default dedupe flags are incorrectly set on secondary disk storage')

        # To validate if default dedupe flags are set or not on dependent primary copy.
        self.log.info("Validate if default dedupe flags are set or not on dependent primary copy of %s",
                      self.plan_name)
        dedupe_flag = self._get_copy_dedupe_flag(self.plan_name, 'Primary')
        if dedupe_flag == 244056064:
            self.log.info('All the default dedupe flags are set on dependent primary copy')
            self.log.info("Validation for default dedupe flags on dependent primary copy of %s was successful",
                          self.plan_name)
        else:
            self._get_copy_flags_not_set(dedupe_flag, 'Primary')
            self._get_copy_flags_set(dedupe_flag, 'Primary')
            raise CVTestStepFailure('The default dedupe flags are incorrectly set on dependent primary copy')

        # To validate if default dedupe flags are set or not on dependent secondary copy.
        self.log.info("Validate if default dedupe flags are set or not on dependent secondary copy of %s",
                      self.plan_name)
        dedupe_flag = self._get_copy_dedupe_flag(self.plan_name, 'Secondary')
        if dedupe_flag == 243531776:
            self.log.info('All the default dedupe flags are set on dependent secondary copy')
            self.log.info("Validation for default dedupe flags on dependent secondary copy of %s was successful",
                          self.plan_name)
        else:
            self._get_copy_flags_not_set(dedupe_flag, 'Secondary')
            self._get_copy_flags_set(dedupe_flag, 'Secondary')
            raise CVTestStepFailure('The default dedupe flags are incorrectly set on dependent secondary copy')

    @test_step
    def check_retention(self):
        """To validate retention rule on plan"""

        # To validate retention days is correctly set to the plan primary copy
        self.log.info("Validate if retention days is set on dependent primary copy of %s", self.plan_name)
        retention_days = self._get_retention_period(self.plan_name, 'Primary')
        if retention_days == 10:
            self.log.info('Retention days is set on dependent primary copy')
        else:
            raise CVTestStepFailure('Retention days was not correctly set on dependent primary copy')

        # To validate retention days is correctly set to the plan secondary copy
        self.log.info("Validate if retention days is set on dependent secondary copy of %s", self.plan_name)
        retention_days = self._get_retention_period(self.plan_name, 'Secondary')
        if retention_days == 15:
            self.log.info('Retention days is set on dependent secondary copy')
        else:
            raise CVTestStepFailure('Retention days was not correctly set on dependent secondary copy')

    @test_step
    def check_auxcopy_schedule(self):
        """To validate System Created Autocopy schedule(Auxiliary Copy) is associated to plan secondary copy"""

        if self._validate_auxcopy_association(self.plan_name):
            self.log.info("System Created Autocopy schedule(Auxiliary Copy) is associated to dependent"
                          " secondary copy of %s was successful", self.plan_name)
        else:
            raise CVTestStepFailure('System Created Autocopy schedule(Auxiliary Copy) is not associated to plan '
                                    'secondary copy')

    def setup(self):
        """Initializes pre-requisites for this test case"""

        self.init_tc()
        self.plan_helper = PlanMain(self.admin_console)
        self.storage_helper = StorageMain(self.admin_console)
        options_selector = OptionsSelector(self.commcell)
        time_stamp = options_selector.get_custom_str()
        self.primary_storage_name = str(self.id) + '_Primary_Disk'
        self.secondary_storage_name = str(self.id) + '_Secondary_Disk'
        ma_machine = options_selector.get_machine_object(self.tcinputs['MediaAgent'])
        self.plan_helper.storage = {'pri_storage': self.primary_storage_name, 'pri_ret_period': '10',
                                    'sec_storage': self.secondary_storage_name, 'sec_ret_period': '15',
                                    'ret_unit': 'Day(s)'}
        self.plan_name = "%s_Plan" % str(self.id)
        self.plan_helper.plan_name = {"server_plan": self.plan_name}
        self.plan_helper.backup_data = None
        self.plan_helper.backup_day = None
        self.plan_helper.backup_duration = None
        self.plan_helper.rpo_hours = None
        self.plan_helper.allow_override = None
        self.plan_helper.snapshot_options = {'Enable_backup_copy': 'OFF'}
        self.plan_helper.database_options = None

        # To select drive with space available in MA machine
        self.log.info('Selecting drive in the MA machine based on space available')
        ma_drive = options_selector.get_drive(ma_machine, size=5 * 1024)
        if ma_drive is None:
            raise Exception("No free space for hosting ddb and mount paths")
        self.log.info('selected drive: %s', ma_drive)
        self.primary_backup_location = ma_machine.join_path(ma_drive, 'Automation', str(self.id), 'MP_Primary')
        self.secondary_backup_location = ma_machine.join_path(ma_drive, 'Automation', str(self.id), 'MP_Secondary')
        self.primary_ddb_location = ma_machine.join_path(ma_drive, 'Automation', str(self.id), 'DDB_Primary_%s' %
                                                         time_stamp)
        self.secondary_ddb_location = ma_machine.join_path(ma_drive, 'Automation', str(self.id), 'DDB_Secondary_%s'
                                                           % time_stamp)

    def run(self):
        """Main function for test case execution"""

        try:
            self.cleanup()
            self.create_entities()
            self.check_dedupe_flags()
            self.check_retention()
            self.check_auxcopy_schedule()
        except Exception as exp:
            handle_testcase_exception(self, exp)
        else:
            self.cleanup()
        finally:
            Browser.close_silently(self.browser)
