# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------


""""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    setup()         --  setup function of this test case

    run()           --  run function of this test case

    tear_down()     --  tear down function of this test case


"""
import  time
from cvpysdk.exception import SDKException
from AutomationUtils.database_helper import MSSQL
from AutomationUtils import constants
from AutomationUtils.options_selector import OptionsSelector
from AutomationUtils.machine import Machine
from AutomationUtils.cvtestcase import CVTestCase
from MediaAgents.MAUtils.mahelper import DedupeHelper, MMHelper


class TestCase(CVTestCase):
    """Class for executing this test case"""

    def __init__(self):
        """Initializes test case class object
        """
        super(TestCase, self).__init__()
        self.name = "zeroref_pruning_offline_mountpath"
        self.tcinputs = {
            "MediaAgentName": None,
            "sqlpassword": None,
            "sqluser": None,
            "MountPathList": None,
        }

        # Local variables
        self.backupset_obj = None
        self.mediaagentname = ""
        self.machineobj = None
        self.ma_machineobj = None
        self.subclient_obj = None
        self.is_time_moved = False
        self.dedup_path = None
        self.storage_policy_name = None
        self.library_name = None
        self.backupset_name = None
        self.subclient_name = None
        self.content_path = None
        self.client_system_drive = None
        self.mountpathlist = None

        self.result_string = ""

    def setup(self):
        """Setup function of this test case"""
        optionobj = OptionsSelector(self.commcell)
        self.mediaagentname = self.tcinputs["MediaAgentName"]
        self.mountpathlist = self.tcinputs['MountPathList'].split(',')

        self.machineobj = Machine(self.client)
        self.client_system_drive = optionobj.get_drive(self.machineobj)
        self.ma_machineobj = Machine(self.mediaagentname, self.commcell)

        timestamp_suffix = OptionsSelector.get_custom_str()
        self.dedup_path = self.machineobj.join_path(self.client_system_drive,
                                                    "DDBs\\tc_54604_{0}".format(timestamp_suffix))
        self.storage_policy_name = "{0}_{1}".format("tc_54604_sp", timestamp_suffix)
        self.library_name = "lib_{0}_{1}".format(self.name, timestamp_suffix)
        self.backupset_name = "bkpset_tc_54604"
        self.subclient_name = "subclient_tc_54604"
        self.content_path = self.machineobj.join_path(self.client_system_drive, "content_54604")
        self.backupset = "defaultBackupSet"
        self.subclient = "default"
        self.mountpathlist = ["{0}_{1}".format(x, timestamp_suffix) for x in self.mountpathlist]
        self.tcinputs['MountPath'] = self.mountpathlist[0]


    def run(self):
        """Run function of this test case"""
        try:

            # Local Variable initialization
            dedup_obj = DedupeHelper(self)
            mmhelper_obj = MMHelper(self)
            sqluser = self.tcinputs['sqluser']
            sqlpassword = self.tcinputs['sqlpassword']

            self.log.info("---Create new Library---")
            # Create new library
            disk_library_obj = mmhelper_obj.configure_disk_library()

            # Configure storage policy

            # Delete Backupset if it exists
            if self.agent.backupsets.has_backupset(self.backupset_name):
                self.log.info("backup set %s exists .. Deleting the same", self.backupset_name)
                try:
                    self.agent.backupsets.delete(self.backupset_name)
                    self.log.info("Successfully deleted backupset %s", self.backupset_name)
                except Exception:
                    self.log.error("Failed to delete backup set. Halting the test case.")
                    raise Exception("Failed to delete backup set. Halting the test case.")

            self.log.info("---Configuring Dedup Storage Policy---")

            self.log.info("Creating new storage policy - %s", self.storage_policy_name)


            sp_obj = self.commcell.storage_policies.add(
                self.storage_policy_name, self.library_name, self.mediaagentname, self.dedup_path)

            self.log.info("---Successfully configured storage policy - %s", self.storage_policy_name)

            sp_obj = self.commcell.storage_policies.get(self.storage_policy_name)
            sp_copy_obj = sp_obj.get_copy('Primary')

            # Configure backup set and subclients
            self.log.info("---Configuring backup set---")
            self.backupset_obj = MMHelper.configure_backupset(self)

            if self.machineobj.check_directory_exists(self.content_path):
                self.machineobj.remove_directory(self.content_path)
            self.machineobj.create_directory(self.content_path)

            self.log.info("---Configuring subclient---")
            self.subclient_obj = MMHelper.configure_subclient(self, content_path=self.content_path)

            self.subclient_obj.data_readers = 1
            self.log.info("----------TC environment configuration completed----------")

            self.log.info("----------Configuring TC environment-----------")
            current_content_dir = self.content_path
            if self.machineobj.check_directory_exists(current_content_dir):
                self.machineobj.remove_directory(current_content_dir)
            self.machineobj.create_directory(current_content_dir)

            sp_id = self.commcell.storage_policies.get(self.storage_policy_name).storage_policy_id
            copy_name = 'Primary'
            engine_id = dedup_obj.get_sidb_ids(sp_id, copy_name)[0]

            # STEP : Set MS Interval to 1 hour

            sqlobj = MSSQL(self.commcell.commserv_name + r'\commvault', sqluser, sqlpassword, "commserv")

            query = "update MMEntityProp set intval = 1 " \
                    "where entityid = %s and propertyname = 'DDBMarkAndSweepOpTime'"% engine_id

            self.log.info("QUERY: %s", query)
            sqlobj.execute(query)

            # STEP : Generate unique data - 500 MB on client and take 3 backups
            self.log.info("---Creating uncompressable unique data---")
            fulljobobj = None
            backup_jobs_list = []
            for i in range(0, 2):
                mmhelper_obj.create_uncompressable_data(self.client.client_name, current_content_dir, 0.1, 0)
                fulljobobj = self.run_backup_job()
                backup_jobs_list.append(fulljobobj)
                time.sleep(60)

            # STEP : Make sure that idxsidbusagehistory shows pending deletes as 0
            zerorefcount_before = self.get_zeroref_count_for_store(engine_id)
            self.log.info("Before deleting any backup jobs, zeroref count on store = %s", zerorefcount_before)

            # Get Mountpath ID
            self.log.info("Fetching MountPathID for the mount path in library %s", self.library_name)
            query = "select mountpathid,mountpathname from mmmountpath where libraryid = ( select LibraryId from " \
                    "mmlibrary where aliasname =  '%s')" % self.library_name
            self.log.info("QUERY: %s", query)
            self.csdb.execute(query)
            mountpath_id, mountpath_name = self.csdb.fetch_one_row()
            self.log.info("MountPathID = %s and MountPathName = %s", mountpath_id, mountpath_name)

            # STEP : Make a note of chunk for job being deleted and confirm its presence
            delete_job = backup_jobs_list[-1]
            query = """select id,volumeid from archchunk where id = (
                        select archchunkid from archchunkmapping where archfileid in (
                        select id from archfile where jobid=%s and filetype=1))""" % delete_job.job_id
            self.log.info("QUERY: %s", query)
            self.csdb.execute(query)
            chunk_id, volume_id = self.csdb.fetch_one_row()
            # write code to check if  chunk folder exists
            #chunk_to_validate = "{0}{1}{2}{1}CV_MAGNETIC{1}V_{3}{1}"{4}{1}SFILE_CONTAINER.idx".format(
            #    self.mountpathlist[0], self.ma_machineobj.os_sep, mountpath_name, volume_id, chunkd_id)
            os_sep = self.ma_machineobj.os_sep
            chunk_to_validate = "%s%s%s%sCV_MAGNETIC%sV_%s%sCHUNK_%s" \
                                "%sSFILE_CONTAINER.idx"%(self.mountpathlist[0], os_sep, mountpath_name,
                                                         os_sep, os_sep, volume_id, os_sep, chunk_id,
                                                         os_sep)
            """chunk_to_validate=os.path.join(self.mountpathlist[0] + self.ma_machineobj.os_sep,
                                           mountpath_name, 'CV_MAGNETIC', 'V_%s' % volume_id,
                                           'CHUNK_%s' % chunk_id, 'SFILE_CONTAINER.idx')
            """
            if self.ma_machineobj.check_file_exists(chunk_to_validate):
                self.log.info("Successfully validated presence of chunk - %s", chunk_to_validate)
            else:
                # Raise Exception and terminate TC as something is wrong.
                raise Exception("Failed to validate presence of chunk - %s", chunk_to_validate)

            # Update MMConfig to get pruning done faster
            query = "update mmconfigs set nmin=1 where name = 'MM_CONFIG_PRUNE_PROCESS_INTERVAL_MINS'"
            self.log.info("QUERY: %s", query)
            sqlobj.execute(query)


            query = "update MMConfigs set value=1 where name='MM_CONFIG_PRUNE_PROCESS_INTERVAL_MINS'"
            self.log.info("QUERY: %s", query)
            sqlobj.execute(query)

            # STEP : Delete Job and run data aging
            self.log.info('Deleting backup job [%s]', delete_job.job_id)
            sp_copy_obj.delete_job(delete_job.job_id)
            data_aging_job = self.commcell.run_data_aging('Primary', self.storage_policy_name)
            self.log.info("data aging job: %s", data_aging_job.job_id)
            if not data_aging_job.wait_for_completion():
                self.log.info("Failed to run data aging with error: %s", data_aging_job.delay_reason)

            self.log.info("Sleeping for 5 minutes")
            time.sleep(300)

            # Delete contents of content directory
            if self.machineobj.check_directory_exists(self.content_path):
                self.ma_machineobj.remove_directory(self.content_path)
            self.ma_machineobj.create_directory(self.content_path)

            # Run a backup
            mmhelper_obj.create_uncompressable_data(self.client.client_name, current_content_dir, 0.1, 0)
            fulljobobj = self.run_backup_job()
            query = "select archchunkid from mmdeletedaf where sidbstoreid=%s" % engine_id
            self.log.info("QUERY: %s", query)
            self.csdb.execute(query)
            chunk_id = self.csdb.fetch_one_row()[0]
            if chunk_id == '':
                self.log.info("MMDeletedAF does not have any entry for chunk %s", chunk_to_validate)
            else:
                self.log.error("MMDeletedAF still has entry for chunk %s", chunk_to_validate)
                raise Exception("MMDeletedAF entries not being picked up for pruning .. Exiting ..")

            # STEP : Rename the mount path folder and make it inaccessible
            self.ma_machineobj.rename_file_or_folder(self.mountpathlist[0],
                                                     "{0}_renamed".format(self.mountpathlist[0]))
            # STEP : Check that MMDeletedAF has no entries

            # STEP : Get current time and Move time forward by 90 minutes
            try:
                self.log.info("Advancing clock by 90 minutes")
                self.is_time_moved = True
                self.ma_machineobj.change_system_time(90*60)
            except Exception as ex:
                self.log.warning("Advancing the clock threw exception %s..", ex)

            if not self.is_time_moved:
                self.log.error("Unable to advance time on MA, no pruning will take place. Failing the test case.")

            self.log.info("Marking MountPath offline for library %s", self.library_name)
            self.mark_mountpath_state(sqlobj, mountpath_id, 'offline')
            self.log.info("Successfully marked MountPath offline")

            self.log.info("Adding a new mount path %s to library to enable backups", self.mountpathlist[1])
            # STEP : Add new MP
            disk_library_obj.add_mount_path(self.mountpathlist[1], self.mediaagentname)

            self.log.info("Successfully added new mountpath - %s", self.mountpathlist[1])

            mmhelper_obj.create_uncompressable_data(self.client.client_name, current_content_dir, 0.1, 0)
            fulljobobj = self.run_backup_job()
            zerorefcount_after = self.get_zeroref_count_for_store(engine_id)
            self.log.info("After deleting the backup job, zeroref count on store = %s", zerorefcount_after)

            # Keep running backups and checking the zeroref count is non-zero
            for i in range(0, 3):
                data_aging_job = self.commcell.run_data_aging('Primary', self.storage_policy_name)
                self.log.info("data aging job: %s", data_aging_job.job_id)
                if not data_aging_job.wait_for_completion():
                    self.log.info("Failed to run data aging with error: %s", data_aging_job.delay_reason)

                # STEP : Get current time and Move time forward by 90 minutes
                mmhelper_obj.create_uncompressable_data(self.client.client_name, current_content_dir, 0.1, 0)
                fulljobobj = self.run_backup_job()


                zerorefcount_current = self.get_zeroref_count_for_store(engine_id)
                self.log.info("After deleting the backup job, zeroref count on store = %s", zerorefcount_after)
                if zerorefcount_current < zerorefcount_after:
                    self.log.error("ERROR : Zeroref count has decreased even when MountPath is offline")
                    self.result_string = "{0}\n{1}".format(
                        self.result_string, "ERROR : Zeroref count has decreased even when MountPath is offline")
                else:
                    self.log.info("Successfully validated - Zeroref count has not gone down when MountPath is offline")
                time.sleep(60)

            # STEP : Now enable MountPath
            self.log.info("Renaming the mountpath folder back to its original name")
            self.ma_machineobj.rename_file_or_folder("{0}_renamed".format(self.mountpathlist[0]),
                                                     self.mountpathlist[0])

            self.log.info("Marking MountPath online for library %s", self.library_name)
            self.mark_mountpath_state(sqlobj,mountpath_id, 'online')
            try:
                self.log.info("QUERY: %s", query)
                sqlobj.execute(query)
            except SDKException as ex:
                self.log.error("Failed to run query ==> %s", ex.exception_message)
                raise Exception("Failed to mark MountPath online ... Exiting ...")
            self.log.info("Successfully marked MountPath online")

            # STEP : Run another couple of backups and check if pruning has caught up
            for i in range(0, 5):
                mmhelper_obj.create_uncompressable_data(self.client.client_name, current_content_dir, 0.1, 0)
                fulljobobj = self.run_backup_job()
                data_aging_job = self.commcell.run_data_aging('Primary', self.storage_policy_name)
                self.log.info("data aging job: %s", data_aging_job.job_id)
                if not data_aging_job.wait_for_completion():
                    self.log.info("Failed to run data aging with error: %s", data_aging_job.delay_reason)

                time.sleep(180)

            zerorefcount_final = self.get_zeroref_count_for_store(engine_id)
            self.log.info("After enabling mount path, zeroref count on store = %s", zerorefcount_final)
            if zerorefcount_final <= zerorefcount_before:
                self.log.info("Successfully verified that zeroref count has gone down after enabling mountpath")
            else:
                self.log.error("Zeroref count is not going down even when mountpath is online")

            # STEP : Validate successful physical pruning by checking existence of volume and chunk folder
            if self.ma_machineobj.check_file_exists(chunk_to_validate):
                self.log.error("FAILURE : Chunk is still present after pruning - %s", chunk_to_validate)
                self.result_string = "%s\n%s", \
                                     self.result_string, \
                                     "FAILURE : Chunk is still present after pruning - {0}".format(chunk_to_validate)
                raise Exception(self.result_string)
            else:
                self.log.info("SUCCESS : Chunk has been successfully deleted physically- %s", chunk_to_validate)

        except Exception as exp:
            self.log.error('Failed to execute test case with error: %s', exp)
            self.result_string = str(exp)
            self.status = constants.FAILED

    def tear_down(self):
        """Tear down function of this test case"""
        self.log.info("In tear down method ...")
        if self.is_time_moved:
            self.log.info("Restore system time on MA to original value")
            try:
                self.ma_machineobj.change_system_time(-90 * 60)
            except Exception as exp:
                self.log.warning("Failed to move clock back due to exception - %s", exp)


        if self.result_string == "":
            self.log.info("Testcase shows successful execution, cleaning up the test environment ...")
            self.log.info("Deleting backupset %s", self.backupset_name)
            self.agent.backupsets.delete(self.backupset_name)
            self.log.info("Deleting storage policy  %s", self.storage_policy_name)
            self.commcell.storage_policies.delete(self.storage_policy_name)
            self.log.info("Deleting library %s", self.library_name)
            self.commcell.disk_libraries.delete(self.library_name)
            if self.machineobj.check_directory_exists(self.content_path):
                self.machineobj.remove_directory(self.content_path)

        else:
            self.log.warning("Testcase shows failure in execution, not cleaning up the test environment ...")

    def run_backup_job(self, backuptype="Incremental"):
        """
        Run a backup job on subclient
        Args:
        backuptype (str) -- Backup type , Incremental by default.

        Return:
        job object of the successfully completed job.
        """
        try:
            fulljobobj = self.subclient_obj.backup(backuptype)
            self.log.info("Successfully initiated a backup job on subclient with jobid - %s", fulljobobj.job_id)
            try:
                if fulljobobj.wait_for_completion() is False:
                    raise Exception("Backup job %s did not complete in given timeout", fulljobobj.job_id)
            except:
                self.log.error("Failed to wait till job completes.")
                raise Exception("Wait for job completion did not work as expected. Exiting the test case.")
        except:
            self.log.error("Failed to run backup job on subclient")

        self.log.info("Successfully completed a backup job on subclient with jobid - %s", fulljobobj.job_id)
        return fulljobobj

    def mark_mountpath_state(self, sqlobj, mountpath_id, state):
        """
        Mark a mountpath as online of offline
        Args:
        sqlobj (object) -- SQL connection object of MSSQL class

        mountpath_id (int) -- Mount path id to be marked online or offline

        state (str) -- mountpath state to mark - online or offline

        Return:
        job object of the successfully completed job.
        """

        if state.lower() == "offline":
            query = "update mmmountpath set isoffline=1, isEnabled = 0 where mountpathid = %s " % mountpath_id
        else:
            query = "update mmmountpath set isoffline=0, isEnabled = 1 where mountpathid = %s " % mountpath_id
        self.log.info("QUERY: %s", query)
        sqlobj.execute(query)

    def get_zeroref_count_for_store(self, engine_id):
        """
        Get pending delete count for ddb store as present in idxsidbusagehistory table
        Args:
        engine_id (int) -- SIDB engine id.

        Return:
        An integer representing the pending delete count for the store
        """

        query = "select top 1 zerorefcount from IdxSIDBUsageHistory where sidbstoreid=%s and historytype=0 " \
                "order by ModifiedTime desc" % engine_id
        self.log.info("QUERY: %s", query)
        self.csdb.execute(query)
        return self.csdb.fetch_one_row()[0]




