# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    run()           --  run function of this test case
"""
from AutomationUtils import config
from AutomationUtils import constants
from AutomationUtils.machine import Machine
from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils.idautils import CommonUtils
from Server.Scheduler.schedulerhelper import ScheduleCreationHelper
from Install.softwarecache_helper import SoftwareCache
from Install.install_helper import InstallHelper


class TestCase(CVTestCase):
    """Class for verifying the Retire Client Option when data is associated with the client"""

    def __init__(self):
        """Initializes test case class object"""
        super(TestCase, self).__init__()
        self.name = "Retire Client - Retire client when data is associated with it."
        self.config_json = None
        self.commcell = None
        self.windows_machine = None

    def setup(self):
        """Initializes pre-requisites for this test case"""
        self.config_json = config.get_config()
        install_helper = InstallHelper(self.commcell)
        self.windows_machine = install_helper.get_machine_objects(1)[0]

    def run(self):
        """Main function for test case execution"""
        try:
            self.install_helper = InstallHelper(self.commcell, self.windows_machine)
            if not self.commcell.clients.has_client(self.install_helper.client_host):
                self.log.info("Creating {0} client".format(self.windows_machine.os_info))
                job = self.install_helper.install_software(features=['FILE_SYSTEM', 'MEDIA_AGENT'])

                if not job.wait_for_completion():
                    raise Exception("{0} Client installation Failed".format(self.windows_machine.os_info))

            self.commcell.clients.refresh()
            self.client = self.commcell.clients.get(self.config_json.Install.windows_client.machine_host)
            software_cache_obj = SoftwareCache(self.commcell, self.client)

            entities = ScheduleCreationHelper(self.commcell)
            entity_properties = entities.entities_setup(client_name=self.client.client_name)

            # Associate data with client.
            self.log.info("Associating back up data with the given client.")
            subclient_object = entity_properties['subclient']['object']
            Idautils = CommonUtils(self)
            Idautils.subclient_backup(subclient_object)

            # Perform the Retire client Operation and wait for the job to complete.
            self.log.info("Will perform Retire operation now on client %s.", self.client.client_name)
            job = self.client.retire()
            if not job.wait_for_completion():
                raise Exception("Uninstall job failed with error: {0}".format(job.delay_reason))

            # Refreshing the clients associated with the commcell Object
            self.commcell.clients.refresh()

            self.log.info("validating Retire operation")

            self.log.info("Checking if client is **not deleted from the commcell")
            if not self.commcell.clients.has_client(self.client.client_name):
                raise Exception("Client has been deleted. Check logs to make sure Retire Operation succeeded.")

            self.log.info("Try backup for retired client")
            try:
                Idautils.subclient_backup(subclient_object)
                raise Exception("Backup succeeded on a retired client."
                                "Client Retire failed. Please check logs")
            except Exception as exp:
                self.log.info("Backup failed for the retired client as expected with reason - %s", exp)

            self.log.info("Check if packages are removed from DB table")
            try:
                software_cache_obj.get_packages(client_id=self.client.client_id)
                raise Exception("Packages are not removed from SimInstalledPackages table."
                                "Client Retire failed. Please check logs")
            except Exception:
                self.log.info("No rows found in the table for this client."
                              "Packages are removed successfully from SimInstalledPackages table")

            self.log.info("Performing Out of place restore for the Retired client")
            client_machine = Machine(machine_name=self.client.client_hostname,
                                     username=self.config_json.Install.windows_client.machine_username,
                                     password=self.config_json.Install.windows_client.machine_password)
            restore_location = client_machine.join_path(self.client.install_directory, "AutomationTest")
            client_machine.create_directory(directory_name=restore_location, force_create=True)
            Idautils.subclient_restore_out_of_place(
                restore_location, subclient_object.content,
                self.commcell.commserv_client.client_name, subclient_object)

            self.log.info("Test case to retire a client when data is associated with it completed successfully.")

        except Exception as exp:
            self.log.error('Failed with error: %s', exp)
            self.result_string = str(exp)
            self.status = constants.FAILED

        finally:
            self.install_helper.uninstall_client(self.client)
