# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Triggering workflow from Custom Reports"""

from AutomationUtils.cvtestcase import CVTestCase

from Reports.Custom.report_templates import DefaultReport
from Reports.Custom.utils import CustomReportUtils

from Web.Common.cvbrowser import (
    BrowserFactory,
    Browser
)
from Web.Common.exceptions import CVTestStepFailure, CVTestCaseInitFailure
from Web.Common.page_object import TestStep

from Web.WebConsole.webconsole import WebConsole
from Web.WebConsole.Reports.Custom.viewer import CustomReportViewer
from Web.WebConsole.Reports.navigator import Navigator
from Web.WebConsole.Forms.forms import Forms
from Web.WebConsole.Reports.Custom import viewer


class TestCase(CVTestCase):
    """ TestCase class used to execute the test case from here."""
    test_step = TestStep()

    def __init__(self):
        super(TestCase, self).__init__()
        self.name = "Custom Reports: Button Panel in Custom Reports"
        self.show_to_user = True
        self.browser = None
        self.webconsole = None
        self.utils = CustomReportUtils(self)
        self.button = None
        self.report = None
        self.forms = None

    def init_tc(self):
        """ Initial configuration for the test case. """
        try:
            self.browser = BrowserFactory().create_browser_object()
            self.browser.open()
            self.webconsole = WebConsole(self.browser, self.commcell.webconsole_hostname)
            self.utils.webconsole = self.webconsole
            self.webconsole.login()
            self.webconsole.goto_reports()
            self.forms = Forms(self.webconsole)
            navigator = Navigator(self.webconsole)
            navigator.goto_report_builder()
            self.report = DefaultReport(self.utils)
            self.report.build_default_report(open_report=False)
        except Exception as exception:
            raise CVTestCaseInitFailure(exception)from exception

    @test_step
    def add_button_and_set_workflow(self):
        """Adds button to the table and runs workflow"""
        self.button = self.report.table.Button("Workflow button")
        self.report.table.toggle_button_panel()
        self.report.table.add_button(self.button)
        self.button.set_workflow("Demo_CheckReadiness")
        self.report.report_builder.save_and_deploy()
        self.browser.driver.refresh()

    @test_step
    def column_level_script(self):
        """Enter column level script"""
        script = """return `<a href="" onclick="return openForm({
        'workflowName':'Demo_CheckReadiness' ,'inputs':{'ClientGroupName':'temp'},success:clearCacheNow,
         _submit:true})">Run Workflow via openForms</a>`"""
        colscrpt = self.report.table.Column("id")
        self.report.table.associate_column_in_builder(colscrpt)
        colscrpt.format_as_custom(script)
        self.report.report_builder.save_and_deploy()

    @test_step
    def verify_openforms_api(self):
        """verifies if the form is submitted"""
        crv = CustomReportViewer(self.webconsole)
        table = viewer.DataTable("Automation Table")
        crv.associate_component(table)
        column = table.Column('id')
        table.associate_column(column)
        column.open_hyperlink_on_cell('Run Workflow via openForms')
        if self.forms.is_form_submitted():
            self.forms.close_form()
        else:
            raise CVTestStepFailure("Form is not submitted")

    @test_step
    def verify_workflow(self):
        """verifies if workflow panel is open"""
        self.report.report_builder.open_report()
        self.button.click_button()
        if self.forms.is_form_open("Demo_CheckReadiness"):
            self.forms.click_cancel()
        else:
            raise CVTestStepFailure("Workflow panel doesn't open")

    def run(self):
        try:
            self.init_tc()
            self.add_button_and_set_workflow()
            self.column_level_script()
            self.verify_workflow()
            self.verify_openforms_api()
        except Exception as err:
            self.utils.handle_testcase_exception(err)
        finally:
            WebConsole.logout_silently(self.webconsole)
            Browser.close_silently(self.browser)
