# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

from AutomationUtils.cvtestcase import CVTestCase
from Web.Common.cvbrowser import (
    Browser,
    BrowserFactory
)
from Web.Common.exceptions import (
    CVTestCaseInitFailure,
    CVTestStepFailure,
)
from Web.Common.page_object import TestStep
from Web.WebConsole.webconsole import WebConsole
from Web.WebConsole.Reports.navigator import Navigator
from Web.WebConsole.Reports.Custom import (
    builder, inputs, viewer
)
from Reports.Custom.utils import CustomReportUtils


class TestCase(CVTestCase):

    test_step = TestStep()

    def __init__(self):
        super(TestCase, self).__init__()
        self.name = "Custom Reports: Required, Optional and Defaults in Listbox"
        self.applicable_os = self.os_list.WINDOWS
        self.product = self.products_list.SOFTWARESTORE
        self.feature = self.features_list.WEBCONSOLE
        self.browser: Browser = None
        self.webconsole: WebConsole = None
        self.builder: builder.ReportBuilder = None
        self.util = CustomReportUtils(self)
        self.listbox: inputs.ListBoxController = None
        self.string = None

    def init_tc(self):
        try:
            self.browser = BrowserFactory().create_browser_object()
            self.browser.open()
            self.webconsole = WebConsole(
                self.browser, self.commcell.webconsole_hostname
            )
            self.webconsole.login()
            self.util.delete_custom_report(self.name)
            Navigator(self.webconsole).goto_report_builder()
            self.builder = builder.ReportBuilder(self.webconsole)
            self.builder.set_report_name(self.name)

            self.string = inputs.String("Listbox")
            self.builder.add_input(self.string)
            self.listbox = inputs.ListBoxController("Listbox")
            self.string.add_html_controller(self.listbox)
        except Exception as err:
            raise CVTestCaseInitFailure(err) from err

    def create_report(self):
        try:
            dataset = builder.Datasets.DatabaseDataset()
            self.builder.add_dataset(dataset)
            dataset.set_dataset_name("AutomationDataset")
            dataset.set_sql_query(
                """
                SELECT @val [Val]
                """
            )
            dataset.add_parameter("@val", self.string)
            dataset.save()

            table = builder.DataTable("AutomationTable")
            self.builder.add_component(table, dataset)
            table.add_column_from_dataset()
            self.builder.save(deploy=True)
            self.builder.open_report()
        except Exception as err:
            raise CVTestStepFailure(err) from err

    @test_step
    def select_manual_values(self):
        """Create a Listbox with values populated manually and marked optional"""
        self.listbox.set_labels_and_values(
            ["A", "C", "B", "D"],
            ["1", "3", "2", "4"]
        )

    @test_step
    def set_default_value(self):
        """Create a listbox having default value"""
        self.string.set_default_value("1")
        self.string.save()

    @test_step
    def validate_viewer_data(self):
        """Validate the data on the optional and default values on viewer"""
        viewer_obj = viewer.CustomReportViewer(self.webconsole)
        table = viewer.DataTable("AutomationTable")
        viewer_obj.associate_component(table)
        received_data = table.get_table_data()
        expected_data = {"Val": ["1"]}
        if received_data != expected_data:
            self.log.error(
                f"Expected: {expected_data}\nReceived: {received_data}"
            )
            raise CVTestStepFailure(
                "Default value not set, unexpected data"
            )

        self.listbox.select_value("C")
        self.listbox.apply()
        expected_data = {"Val": ["3"]}
        received_data = table.get_table_data()
        if received_data != expected_data:
            self.log.error(
                f"Expected: {expected_data}\nReceived: {received_data}"
            )
            raise CVTestStepFailure(
                "Selected value not set, unexpected data"
            )

    def run(self):
        try:
            self.init_tc()
            self.select_manual_values()
            self.set_default_value()
            self.create_report()
            self.validate_viewer_data()
        except Exception as err:
            self.util.handle_testcase_exception(err)
        finally:
            WebConsole.logout_silently(self.webconsole)
            Browser.close_silently(self.browser)
