# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
""" Validate Software Version report"""

from Web.Common.cvbrowser import BrowserFactory
from Web.Common.cvbrowser import Browser
from Web.Common.exceptions import CVTestCaseInitFailure, CVTestStepFailure
from Web.Common.page_object import TestStep

from Web.WebConsole.webconsole import WebConsole
from Web.WebConsole.Reports.navigator import Navigator
from Web.WebConsole.Reports.Metrics.health_tiles import GenericTile
from Web.WebConsole.Reports.Custom import viewer

from AutomationUtils.cvtestcase import CVTestCase

from Reports.Custom.utils import CustomReportUtils


class TestCase(CVTestCase):
    """
    Verify Version and Service pack report
    """
    test_step = TestStep()

    def __init__(self):
        super(TestCase, self).__init__()
        self.name = "Software Version report validation"
        self.tcinputs = {
            "ClientName": None}
        self.browser: Browser = None
        self.webconsole: WebConsole = None
        self.navigator: Navigator = None
        self.utils = CustomReportUtils(self)
        self.viewer = None
        self.table = None
        self.version_sp_tile = None

    def init_tc(self):
        """
        initialize testcase objects
        """
        try:
            self.browser = BrowserFactory().create_browser_object()
            self.browser.open()
            self.webconsole = WebConsole(self.browser, self.commcell.webconsole_hostname)
            self.webconsole.login()
            self.webconsole.goto_commcell_dashboard()

            self.navigator = Navigator(self.webconsole)
            self.navigator.goto_commcell_dashboard(self.commcell.commserv_name)
            self.navigator.goto_health_report()

            self.version_sp_tile = GenericTile(self.webconsole, 'Software Version')

        except Exception as exp:
            raise CVTestCaseInitFailure(exp) from exp

    @test_step
    def verify_tile_status(self):
        """
        verify the version and service pack tile status in health report
        """
        version_sp_status = self.version_sp_tile.get_health_status()
        self.version_sp_tile.access_view_details()
        self.table = viewer.DataTable("Servers")
        self.viewer = viewer.CustomReportViewer(self.webconsole)
        self.viewer.associate_component(self.table)
        service_pack = int(self.client.service_pack)
        difference = int(self.get_major_sp()) - service_pack
        if difference <= 4:
            raise CVTestCaseInitFailure(
                f"Client [{self.client}] is not older by 4 major SP "
                f"provide such a client in input"
            )
        client_status = self.get_client_status()
        if client_status != version_sp_status:
            raise CVTestStepFailure(f"The expected tile status is [{client_status}] "
                                    f"but received [{version_sp_status}]")

    @test_step
    def verify_commserver_status(self):
        """
        verify the commserv status from the report
        """
        html_comp = viewer.HtmlComponent("")
        self.viewer.associate_component(html_comp)
        status = html_comp.get_html_component_contents()
        cs_status = status.split('Status: ')[1]
        expected_status = 'Good'
        if expected_status not in cs_status:
            raise CVTestStepFailure(f"The expected commcell [{self.commcell.commserv_name}] "
                                    f"status is [Good] but received [{cs_status}]")

    @test_step
    def validate_critical_status(self):
        """
        validate the critical client status`
        """
        expected_status = 'Critical'
        self.table.set_filter("Name", self.client.client_name)
        client_status = self.table.get_column_data('Update Status')
        if expected_status != client_status[0]:
            raise CVTestStepFailure(
                f"The expected client status is [{expected_status}] "
                f"but received [{client_status[0]}]"
            )

    def get_client_status(self):
        """ get the client status from the report page
        Returns: status
        """
        client_status = self.table.get_column_data('Update Status')
        if "Critical" in client_status:
            return "Critical"
        if "Warning" in client_status:
            return "Warning"
        return "Good"

    def get_major_sp(self):
        """
        get the major service pack from CvCloud DB
        Returns: service pack number
        """
        query = """
                SELECT VALUE FROM CF_SURVEYCONFIG 
                WHERE NAME='11_0_LatestSPMajorNo'
                """
        major_version = self.utils.cre_api.execute_sql(
            query,
            database_name="CVCloud",
            desc="Getting major service pack info from CF_SURVEYCONFIG table")[0][0]
        return major_version

    def run(self):
        try:
            self.init_tc()
            self.verify_tile_status()
            self.verify_commserver_status()
            self.validate_critical_status()

        except Exception as exp:
            self.utils.handle_testcase_exception(exp)

        finally:
            WebConsole.logout_silently(self.webconsole)
            Browser.close_silently(self.browser)
