# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

from AutomationUtils.cvtestcase import CVTestCase
from Reports.Custom.utils import CustomReportUtils
from Web.Common.page_object import TestStep

from Web.Common.exceptions import (
    CVTestCaseInitFailure, CVTestStepFailure
)
from Web.Common.cvbrowser import (
    BrowserFactory, Browser
)
from Web.WebConsole.webconsole import WebConsole
from Web.WebConsole.Reports.navigator import Navigator
from Web.WebConsole.Reports.Custom import (
    builder, inputs, viewer
)
from Web.API.customreports import CustomReportsAPI


class TestCase(CVTestCase):

    test_step = TestStep()

    def __init__(self):
        super(TestCase, self).__init__()
        self.name = "Custom Reports: Commcell input"
        self.util = CustomReportUtils(self)
        self._browser = None
        self.webconsole: WebConsole = None
        self.tcinputs = {
            "multiCommcell": None,
            "singleCommcell": None
        }
        self.rpt_builder: builder.ReportBuilder = None
        self.input_controller: inputs.ListBoxController = None
        self.expected_multi_cc = None

    @property
    def browser(self):
        if self._browser is None:
            self._browser = BrowserFactory().create_browser_object()
            self._browser.open()
        return self._browser

    def init_multi_commcell_wc(self):
        try:
            with CustomReportsAPI(self.tcinputs["multiCommcell"]) as api:
                api.delete_custom_report_by_name(self.name, suppress=True)
                self.expected_multi_cc = list(map(
                    lambda c: c.split(".")[0].lower(),
                    self.util.get_commcell_datasources(api)
                ))
                self.expected_multi_cc.sort()
                err_msg = f"Not enough commcells on [{self.tcinputs['multiCommcell']}]"
                assert len(self.expected_multi_cc) > 1, err_msg

            self.webconsole = WebConsole(
                self.browser, self.tcinputs["multiCommcell"]
            )
            self.webconsole.login()
            Navigator(self.webconsole).goto_report_builder()
            self.rpt_builder = builder.ReportBuilder(self.webconsole)
            self.rpt_builder.set_report_name(self.name)
        except Exception as err:
            raise CVTestCaseInitFailure(err) from err

    def init_single_commcell_wc(self):
        try:
            self.log.info("-" * 10 + " Initializing Single Commcell " + str("-" * 10))
            with CustomReportsAPI(self.tcinputs["singleCommcell"]) as api:
                api.delete_custom_report_by_name(self.name, suppress=True)
                expected_single_cc = self.util.get_commcell_datasources(api)
                cc = self.tcinputs['singleCommcell']
                err_msg = (
                    f"More than one commcell found on [{cc}]\n." +
                    f"Expecting only one commcell."
                )
                assert len(expected_single_cc) == 1, err_msg

            self.webconsole = WebConsole(
                self.browser, self.tcinputs["singleCommcell"]
            )
            self.webconsole.login()
            Navigator(self.webconsole).goto_report_builder()
            self.rpt_builder = builder.ReportBuilder(self.webconsole)
            self.rpt_builder.set_report_name(self.name)
        except Exception as err:
            raise CVTestCaseInitFailure(err) from err

    def _create_report(self):
        ip_datatype = inputs.Commcell("CommcellInput")
        self.rpt_builder.add_input(ip_datatype)
        ip_datatype.enable_multi_selection()
        ip_datatype.set_required()
        self.input_controller = inputs.ListBoxController("CommcellInput")
        ip_datatype.add_html_controller(self.input_controller)
        ip_datatype.save()

        ds = builder.Datasets.DatabaseDataset()
        self.rpt_builder.add_dataset(ds)
        ds.set_all_commcell_datasource()
        ds.set_dataset_name("AutomationCommcellDataSet")
        ds.set_sql_query(
            """
            SELECT name [Commcell]
            FROM APP_Client
            WHERE id = 2
            """
        )
        ds.save()

        table = builder.DataTable("CommcellTable")
        self.rpt_builder.add_component(table, ds)
        table.add_column_from_dataset("Commcell")

        self.rpt_builder.save_and_deploy()

    @test_step
    def create_report(self):
        """Create a report with commcell input"""
        self._create_report()

    @test_step
    def list_all_commcells(self):
        """On machines with multiple commcells, all the commcells should be listed."""
        received_commcells = self.input_controller.get_available_options()
        received_commcells_ = list(map(
            lambda c: c.split(".")[0].lower(),
            self.input_controller.get_available_options()
        ))
        received_commcells_.sort()
        if self.expected_multi_cc != received_commcells_:
            self.log.error(
                f"\nExpected: {self.expected_multi_cc}"
                f"\nReceived: {received_commcells_}"
            )
            raise CVTestStepFailure(
                "Commcell list mismatch in multi commcell WebConsole"
            )
        self.rpt_builder.open_report()
        return received_commcells

    @test_step
    def select_commcell_from_dropdown(self, received_commcells):
        """After selecting a commcell, the dataset result should be filtered."""
        self.input_controller.select_value(received_commcells[0])
        table = builder.DataTable("CommcellTable")
        rpt_viewer = viewer.CustomReportViewer(self.webconsole)
        rpt_viewer.associate_component(table)
        data = table.get_column_data("Commcell")
        expected = received_commcells[0].split(".")[0].lower()
        received = (data if data else [""])[0].split(".")[0].lower()
        if expected != received:
            err = f"\nExpected: {expected}\nReceived: {received}"
            raise CVTestStepFailure(f"Commcell list not filtered.{err}")

    @test_step
    def input_should_not_be_visible(self):
        """In single commcell WC, commcell input should not be shown and report should auto select local commcell."""
        self._create_report()
        self.rpt_builder.open_report()
        rpt_viewer = viewer.CustomReportViewer(self.webconsole)
        if rpt_viewer.get_all_input_names():
            raise CVTestStepFailure("Commcell Input is visible on report viewer")
        table = viewer.DataTable("CommcellTable")
        rpt_viewer.associate_component(table)
        expected_data = table.get_column_data("Commcell")[0].split(".")[0].lower()
        received_data = self.tcinputs["singleCommcell"].split(".")[0].lower()
        if expected_data != received_data:
            raise CVTestStepFailure(
                f"Unexpected data; "
                f"\nExpected: {expected_data}"
                f"\nReceived: {received_data}"
            )

    @test_step
    def verify_required_input(self):
        """Components should not be visible till all the required inputs are selected"""
        viewer_obj = viewer.CustomReportViewer(self.webconsole)
        if viewer_obj.get_all_component_titles():
            raise CVTestStepFailure(
                "Components are visible without selecting required commcell input"
            )

    def run(self):
        try:
            self.init_multi_commcell_wc()
            self.create_report()
            commcells = self.list_all_commcells()
            self.verify_required_input()
            self.select_commcell_from_dropdown(commcells)
            self.init_single_commcell_wc()
            self.input_should_not_be_visible()
        except Exception as err:
            self.util.handle_testcase_exception(err)
        finally:
            WebConsole.logout_silently(self.webconsole)
            Browser.close_silently(self.browser)
