# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
""" verify download center anonymous access feature """

from time import sleep

from Web.Common.cvbrowser import BrowserFactory
from Web.Common.cvbrowser import Browser
from Web.Common.exceptions import CVTestCaseInitFailure, CVTestStepFailure
from Web.Common.page_object import TestStep
from Web.WebConsole.webconsole import WebConsole
from Web.WebConsole.DownloadCenter.downloadcenter import DownloadCenter
from Web.WebConsole.DownloadCenter.settings import ManageInformation

from AutomationUtils.cvtestcase import CVTestCase
from Reports.utils import TestCaseUtils


class TestCase(CVTestCase):
    """
    TestCase class used to execute the test case from here.
    """
    test_step = TestStep()

    def __init__(self):
        super(TestCase, self).__init__()
        self.name = "verify download center anonymous access feature"
        self.show_to_user = True
        self.browser = None
        self.web_console = None
        self.download_center = None
        self.manage_information = None
        self.test_web_console = None
        self.test_webconsole_dc = None
        self.test_webconsole_manage_information = None
        self.category = "AnonymousCategory"
        self.sub_category = "AnonymousSC"
        self.utils = TestCaseUtils(self)

    def access_download_center_url(self, webconsole):
        """Access download center url"""
        dc_url = self.test_webconsole_dc.download_center_url
        webconsole.browser.driver.get(dc_url)
        webconsole.wait_till_load_complete()

    def init_tc(self):
        """
        Initial configuration for the test case
        """
        try:
            self.browser = BrowserFactory().create_browser_object(
                browser_type= Browser.Types.FIREFOX, name="ClientBrowser")
            self.browser.open()

            # login to web console
            self.web_console = WebConsole(self.browser, self.commcell.webconsole_hostname)
            self.web_console.login()
            self.web_console.goto_download_center()
            self.download_center = DownloadCenter(self.web_console)
            self.manage_information = ManageInformation(self.web_console)
            self.utils.reset_temp_dir()
        except Exception as exception:
            raise CVTestCaseInitFailure(exception) from exception

    def verify_xml_downloaded(self):
        """Verify xml file is downloaded"""
        xml_path = self.utils.poll_for_tmp_files('xml')
        self.log.info("[%s] XML file downloaded successfully!", xml_path)

    def initial_configuration(self):
        """Check if subcategory already anonymous access is enabled, if yes disable it"""
        try:
            self.log.info("Check if subcategory already anonymous access is enabled, "
                          "if yes disable it")
            self.initiate_test_browser()
            self.test_webconsole_dc.access_category(self.category)
            if self.test_webconsole_dc.is_subcategory_exists(self.sub_category) is False:
                return
            self.test_web_console.login()
            self.test_web_console.goto_download_center()
            self.test_webconsole_dc.access_manage_information()
            self.test_webconsole_manage_information.search_category(self.category)
            self.test_webconsole_manage_information.select_category(self.category)
            self.test_web_console.wait_till_load_complete()
            sub_category = self.test_webconsole_manage_information.edit_sub_category(
                self.sub_category)
            sub_category.switch_anonymous_access(status=False)
            sub_category.save()
            sleep(3)
        except Exception:
            self.log.info("Configuration settings are as expected")
        finally:
            self.test_web_console.browser.close()

    @test_step
    def test_step_1(self):
        """Enable anonymous access and see packages can be seen without login"""
        try:
            self.log.info("Enabling anonymous access")
            self.download_center.access_manage_information()
            self.manage_information.search_category(self.category)
            self.manage_information.select_category(self.category)
            sub_category = self.manage_information.edit_sub_category(self.sub_category)
            sub_category.switch_anonymous_access(status=True)
            sub_category.save()
            self.log.info("Anonymous access is enabled!")

            self.log.info("Verify expected sub category is visible without login")
            self.log.info("Wait for 2 minutes for syncup")
            sleep(120)
            self.initiate_test_browser()
            self.test_webconsole_dc.access_category(self.category)
            if self.test_webconsole_dc.is_subcategory_exists(self.sub_category) is False:
                raise CVTestStepFailure("Subcategory [%s] is not found without login" %
                                        self.sub_category)
            self.log.info("Expected sub category [%s] exists in download center",
                          self.sub_category)

            self.log.info("click on download and then login, search for subcategory. "
                          "click on download package")
            package_name = self.test_webconsole_dc.get_package_list()[0]
            self.test_webconsole_dc.download_package(package_name)
            self.test_web_console.login(auto_open_login_page=False)
            self.test_webconsole_dc.access_category(self.category)
            self.test_webconsole_dc.access_sub_category(self.sub_category)
            self.test_webconsole_dc.download_package(package_name)

            # Verify xml is downloaded
            self.verify_xml_downloaded()
            WebConsole.logout_silently(self.test_web_console)
            Browser.close_silently(self.test_web_console)
        except Exception as _exception:
            raise CVTestStepFailure(_exception)

    @test_step
    def test_step_2(self):
        """Enable anonymous download, verify without login should be able to download the xml"""
        try:
            self.utils.reset_temp_dir()
            self.log.info("Enabling free download")
            sub_category = self.manage_information.edit_sub_category(self.sub_category)
            sub_category.switch_free_downloads(status=True)
            sub_category.save()

            self.log.info("access download center url, verify xml can be downloaded without"
                          " login")
            self.log.info("Wait for 2 minutes for syncup")
            sleep(120)
            self.initiate_test_browser()
            self.test_webconsole_dc.access_category(self.category)
            self.test_webconsole_dc.access_sub_category(self.sub_category)
            package_name = self.test_webconsole_dc.get_package_list()[0]
            self.test_webconsole_dc.download_package(package_name)

            # Verify xml is downloaded
            self.verify_xml_downloaded()
            Browser.close_silently(self.test_web_console.browser)
        except Exception as _exception:
            raise CVTestStepFailure(_exception)

    def initiate_test_browser(self):
        """Initiate test browser"""
        browser = BrowserFactory().create_browser_object(
            browser_type= Browser.Types.FIREFOX, name="TestDCBrowser")
        browser.set_downloads_dir(self.utils.get_temp_dir())
        browser.open()
        # login to web console
        self.test_web_console = WebConsole(browser, self.commcell.webconsole_hostname)
        self.test_webconsole_dc = DownloadCenter(self.test_web_console)
        self.access_download_center_url(self.test_web_console)
        self.test_webconsole_manage_information = ManageInformation(self.test_web_console)

    def revert_changes(self):
        """Disable anonymous access for sub category"""
        sub_category = self.manage_information.edit_sub_category(self.sub_category)
        sub_category.switch_anonymous_access(status=False)
        sub_category.save()
        self.log.info("Reverted back configuration changes")

    def run(self):
        try:
            self.initial_configuration()
            self.init_tc()
            self.test_step_1()
            self.test_step_2()
            self.revert_changes()
        except Exception as err:
            self.utils.handle_testcase_exception(err)
        finally:
            WebConsole.logout_silently(self.web_console)
            Browser.close_silently(self.web_console.browser)
            WebConsole.logout_silently(self.test_web_console)
            Browser.close_silently(self.test_web_console.browser)
