# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
# --------------------------------------------------------------------------

""""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    setup()         --  initial settings for the test case

    run()           --  run function of this test case

"""

from AutomationUtils.cvtestcase import CVTestCase
from Server.Network.networkhelper import NetworkHelper
from AutomationUtils import constants


class TestCase(CVTestCase):
    """Class for executing network case to validate Cascading gateways topology

        Setup requirements to run this test case:
        4 clients -- can be any clients in the commcell

    """

    def __init__(self):
        """Initializes test case class object"""

        super(TestCase, self).__init__()
        self.name = "[Network & Firewall] : Cascading gateways topology"
        self.product = self.products_list.COMMSERVER
        self.feature = self.features_list.NETWORK
        self.client_list = []
        self.tcinputs = {
            "NetworkClient1": None,
            "NetworkClient2": None,
            "NetworkClient3": None,
            "NetworkClient4": None,
        }
        self._client_group_name1 = 'Trusted Client Group1'
        self._client_group_name2 = 'Trusted Client Group2'
        self._client_group_name3 = 'DMZ group near Trusted Client Group1'
        self._client_group_name4 = 'DMZ group near Trusted Client Group2'
        self.commserv = None
        self._network = None
        self.entities = None

        self.topology_name = "Cascading_Gateway_Topology_54254"
        self.client_groups = None
        self.client_group_obj1 = None
        self.client_group_obj2 = None
        self.client_group_obj3 = None
        self.client_group_obj4 = None

    def setup(self):
        """Setup function of this test case"""
        self.commserv = self.commcell.commserv_name
        self._network = NetworkHelper(self)
        self.entities = self._network.entities

        self.client_list.extend([self.commserv,
                                 self.tcinputs['NetworkClient1'],
                                 self.tcinputs['NetworkClient2'],
                                 self.tcinputs['NetworkClient3'],
                                 self.tcinputs['NetworkClient4']
                                 ])

        self._network.remove_network_config([{'clientName': self.client_list[0]},
                                             {'clientName': self.client_list[1]},
                                             {'clientName': self.client_list[2]},
                                             {'clientName': self.client_list[3]},
                                             {'clientName': self.client_list[4]}
                                             ])

        self._network.push_config_client([self.client_list[0],
                                          self.client_list[1],
                                          self.client_list[2],
                                          self.client_list[3],
                                          self.client_list[4]
                                          ])

        self.client_groups = self._network.entities.create_client_groups([self._client_group_name1,
                                                                          self._client_group_name2,
                                                                          self._client_group_name3,
                                                                          self._client_group_name4])

        self.client_group_obj1 = self.client_groups[self._client_group_name1]['object']
        self.client_group_obj2 = self.client_groups[self._client_group_name2]['object']
        self.client_group_obj3 = self.client_groups[self._client_group_name3]['object']
        self.client_group_obj4 = self.client_groups[self._client_group_name4]['object']

    def run(self):
        """Run function"""

        try:

            self.log.info("Started executing testcase")

            self.client_group_obj1.add_clients([self.client_list[1]])

            self.client_group_obj2.add_clients([self.client_list[2], self.client_list[0]])

            self.client_group_obj3.add_clients([self.client_list[3]])

            self.client_group_obj4.add_clients([self.client_list[4]])

            self._network.set_one_way({'clientName': self.client_list[0]},
                                      {'clientName': self.client_list[4]})

            self._network.set_one_way({'clientName': self.client_list[0]},
                                      {'clientGroupName': self._client_group_name1})

            self._network.topology_pre_settings(self.client_list)

            self._network.serverbase.check_client_readiness(self.client_list)

            self.log.info("***Creating cascading gateways topology with display type as Servers***")

            self._network.cascading_gateways_topology(self._client_group_name1,
                                                      self._client_group_name2,
                                                      self._client_group_name3,
                                                      self._client_group_name4,
                                                      self.topology_name)

            self._validate()

            self._network.delete_topology(self.topology_name)

            self.log.info("***Creating cascading gateways topology again with display "
                          "type as Laptops***")

            self._network.cascading_gateways_topology(self._client_group_name1,
                                                      self._client_group_name2,
                                                      self._client_group_name3,
                                                      self._client_group_name4,
                                                      self.topology_name,
                                                      display_type=1)

            self._validate()

            self._network.outgoing_route_settings({'clientGroupName': self._client_group_name1},
                                                  remote_entity=self._client_group_name3,
                                                  streams=4,
                                                  is_client=False,
                                                  connection_protocol=2,
                                                  )

            self._network.outgoing_route_settings({'clientGroupName': self._client_group_name2},
                                                  remote_entity=self._client_group_name3,
                                                  streams=4,
                                                  is_client=False,
                                                  connection_protocol=2,
                                                  )

            self._network.outgoing_route_settings({'clientGroupName': self._client_group_name1},
                                                  remote_entity=self._client_group_name4,
                                                  streams=4,
                                                  is_client=False,
                                                  connection_protocol=2,
                                                  )

            self._network.outgoing_route_settings({'clientGroupName': self._client_group_name2},
                                                  remote_entity=self._client_group_name4,
                                                  streams=4,
                                                  is_client=False,
                                                  connection_protocol=2,
                                                  )

            self._network.outgoing_route_settings({'clientGroupName': self._client_group_name3},
                                                  remote_entity=self._client_group_name4,
                                                  streams=4,
                                                  is_client=False,
                                                  connection_protocol=2,
                                                  )

            self._network.validate([self.client_list[1]], self.client_list[0], test_data_path="C:\\testData_54254")

            self._network.modify_topology(self.topology_name,
                                          [{'group_type': 2, 'group_name': self._client_group_name1,
                                            'is_mnemonic': False},
                                           {'group_type': 4, 'group_name': self._client_group_name3,
                                            'is_mnemonic': False},
                                           {'group_type': 1, 'group_name': self._client_group_name2,
                                            'is_mnemonic': False},
                                           {'group_type': 3, 'group_name': self._client_group_name4,
                                            'is_mnemonic': False}],
                                          topology_type=4,
                                          is_smart_topology=False, topology_description="Updated topology")

            self._network.delete_topology(self.topology_name)

            self.log.info("*" * 10 + " TestCase {0} successfully completed! ".format(self.id) + "*" * 10)
            self.status = constants.PASSED

        except Exception as excp:
            self.status = constants.PASSED
            self._network.server.fail(excp)

        finally:
            self._network.delete_topology(self.topology_name)
            self._network.cleanup_network()
            self._network.entities.cleanup()

    def _validate(self):
        """validation of created topology routes"""
        self._network.push_topology(self.topology_name)

        self._network.options.sleep_time(10)

        self._network.serverbase.check_client_readiness(self.client_list)

        self._network.validate_cascading_gateways_topology(self.topology_name)

        self._network.topology_post_settings()

        self.client_group_obj3.refresh()
        self.client_group_obj4.refresh()

        if not self.client_group_obj3.network.proxy:
            raise Exception("isDMZ option is not set on proxy client group")

        if not self.client_group_obj4.network.proxy:
            raise Exception("isDMZ option is not set on proxy client group")

        if self.client_group_obj3.network.lockdown:
            raise Exception("Lockdown option is set to 1 on proxy client group")

        if self.client_group_obj4.network.lockdown:
            raise Exception("Lockdown option is set to 1 on proxy client group")

        self._network.validate_fwconfig_file(2, self.client_list[1],
                                             self.client_list[3])

        self._network.validate_fwconfig_file(2, self.client_list[3],
                                             self.client_list[4])

        self._network.validate_fwconfig_file(2, self.client_list[2],
                                             self.client_list[4])

        self._network.validate_fwconfig_file(1, self.client_list[3],
                                             self.client_list[4],
                                             self.client_list[2])

        self._network.validate_fwconfig_file(1, self.client_list[1],
                                             self.client_list[3],
                                             self.client_list[2])

        self._network.validate_fwconfig_file(1, self.client_list[2],
                                             self.client_list[4],
                                             self.client_list[1])
