# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

Basic acceptance test case for Credential Manager in AdminConsole

TestCase:
    __init__()                  --  Initializes the TestCase class

    setup()                     --  All testcase objects are initializes in this method

    run()                       --  Contains the core testcase logic and it is the one executed

"""

from AutomationUtils.cvtestcase import CVTestCase
from Web.Common.cvbrowser import BrowserFactory, Browser
from Web.AdminConsole.adminconsole import AdminConsole
from Web.AdminConsole.Helper.credential_manager_helper import CredentialManagerHelper
from Reports.utils import TestCaseUtils


class TestCase(CVTestCase):

    def __init__(self):
        """Initializes test case class object

            Testcase json example:

            "54147":
                {
                "account_type": "Windows Account",
                "owner": "master",
                "credential_name": "######",
                "credential_username": "######",
                "credential_password": "######",
                "security": "######",
                "new_credential_name":"######",
                "modified_credential_username": "######",
                "modified_credential_password": "abcd",
                "subclient_content_path": "\\\\######\\cred_test",

                "relogin_username": "domain\\administrator",
                "relogin_password": "abcd"

                }

        """

        super(TestCase, self).__init__()
        self.name = "Basic Acceptance Test Credential Manager in AdminConsole"
        self.product = self.products_list.COMMSERVER
        self.feature = self.features_list.ADMINCONSOLE
        self.utils = TestCaseUtils(self)
        self.browser = None
        self.admin_console = None
        self.credential_manager_helper = None
        self.tcinputs = {
            "owner": None,
            "credential_name": None,
            "credential_username": None,
            "credential_password": None,
            "security": None,
            "modified_credential_username": None,
            "modified_credential_password": None,
            "relogin_username": None,
            "relogin_password": None
        }

    def setup(self):

        self.browser = BrowserFactory().create_browser_object(name="User Browser")
        self.browser.open()
        self.admin_console = AdminConsole(self.browser, self.commcell.webconsole_hostname)
        self.admin_console.login(self.inputJSONnode['commcell']['commcellUsername'],
                                 self.inputJSONnode['commcell']['commcellPassword'])
        self.credential_manager_helper = CredentialManagerHelper(self.admin_console)

    def run(self):

        try:

            self.credential_manager_helper.account_type = self.tcinputs['account_type']
            self.credential_manager_helper.credential_name = self.tcinputs['credential_name']
            self.credential_manager_helper.new_credential_name = self.tcinputs['new_credential_name']
            self.credential_manager_helper.credential_username = self.tcinputs['credential_username']
            self.credential_manager_helper.credential_password = self.tcinputs['credential_password']
            self.credential_manager_helper.owner = self.tcinputs['owner']
            self.credential_manager_helper.user_or_group = self.tcinputs['security'].split(",")

            self.log.info("*********Adding a credential*********")
            self.credential_manager_helper.add_credential()

            self.log.info("*********Editing a credential*********")
            self.credential_manager_helper.credential_username = self.tcinputs['modified_credential_username']
            self.credential_manager_helper.credential_password = self.tcinputs['modified_credential_password']

            self.credential_manager_helper.edit_credential()

            self.admin_console.navigator.logout()
            self.admin_console.wait_for_completion()

            self.credential_manager_helper.credential_name = self.credential_manager_helper.new_credential_name
            self.log.info("*********Attempting edit/delete from non-owner account*********")
            self.admin_console.login(self.tcinputs['relogin_username'],
                                     self.tcinputs['relogin_password'])

            self.credential_manager_helper.owner = self.tcinputs['relogin_username']
            self.credential_manager_helper.user_or_group = self.tcinputs['relogin_username']

            self.credential_manager_helper.attempt_edit_delete_non_owner()
            self.admin_console.navigator.logout()

        except Exception as exp:
            self.utils.handle_testcase_exception(exp)

    def tear_down(self):
        """ To clean-up the test case environment created """
        try:

            self.admin_console.login(self.inputJSONnode['commcell']['commcellUsername'],
                                     self.inputJSONnode['commcell']['commcellPassword'])
            self.credential_manager_helper.credential_name = self.credential_manager_helper.new_credential_name
            self.credential_manager_helper.delete_credential()

        except Exception as exp:
            self.utils.handle_testcase_exception(exp)

        finally:
            AdminConsole.logout_silently(self.admin_console)
            Browser.close_silently(self.browser)
