# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

This testcase does indexing operations like playback, browse, restore, synthetic full and
compaction and prepares a report on the time taken between two service packs

TestCase:
    __init__()                  --  Initializes the TestCase class

    setup()                     --  All testcase objects are initializes in this method

    run()                       --  Contains the core testcase logic and it is the one executed

    test_playback()             --  Tests play

    tear_down()                 --  Cleans the data created for Indexing validation

"""

import traceback
import time
import re
import base64
import socket
import matplotlib.pyplot as plt

from io import BytesIO
from datetime import datetime

from AutomationUtils import commonutils
from AutomationUtils import constants
from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils.machine import Machine
from AutomationUtils.database_helper import MSSQL

from Indexing.database import index_db

from cvpysdk.exception import SDKException


class TestCase(CVTestCase):
    """This testcase does indexing operations like playback, browse, restore, synthetic full and
    compaction and prepares a report on the time taken between two service packs"""

    def __init__(self):
        """Initializes test case class object"""

        super(TestCase, self).__init__()
        self.name = 'Indexing - Performance test report'
        self.show_to_user = False

        self.tcinputs = {
            'IndexServer': None,
            'CSDBPassword': None
        }

        self.storage_policy = None
        self.cs_name = None
        self.cs_hostname = None
        self.cs_db = None
        self.history_db = None

        self.cl_machine = None
        self.cl_delim = None
        self.isc_obj = None
        self.isc_sp = None
        self.db = None

        self.tests = ['playback', 'browse', 'restore', 'synthetic_full', 'compaction']
        self.current_stats = []
        self.attempt = None
        self.counter_id = None
        self.low_sp = None
        self.high_sp = None
        self.test_process_id = {}
        self.send_report = None

        self.job_types = {
            '2': 'Full',
            '4': 'Incremental',
            '8': 'Synthetic Full',
            '16': 'Differential'
        }

    def setup(self):
        """All testcase objects are initializes in this method"""

        try:

            perf_tests = self.tcinputs.get('Tests', None)
            if perf_tests is not None:
                self.tests = perf_tests.split(',')
                self.tests = [test.strip().lower() for test in self.tests]

            self.cs_name = self.commcell.commserv_name
            self.cs_hostname = self.commcell.commserv_hostname

            db_server_name = self.cs_hostname + '\\Commvault'

            self.cs_db = MSSQL(
                db_server_name, 'sa', self.tcinputs.get('CSDBPassword'), 'CommServ'
            )

            self.history_db = MSSQL(
                db_server_name, 'sa', self.tcinputs.get('CSDBPassword'), 'HistoryDB'
            )

            self.create_stats_table()

            self.isc_name = self.get_index_server_change()
            self.isc_obj = self.commcell.clients.get(self.isc_name)
            self.isc_machine = Machine(self.isc_obj, self.commcell)
            self.isc_delim = self.isc_machine.os_sep
            self.isc_sp = self.isc_obj.service_pack

            self.attempt = self.tcinputs.get('Attempt', self.get_attempt())
            self.skip_delete_db = self.tcinputs.get('SkipDeleteDB', False)
            self.send_report = self.tcinputs.get('SendReport', 'always').lower()

            self.old_index_server_name = self.backupset.index_server.client_name
            self.oldest_cycle_etime = self.get_oldest_cycle_time()

            self.get_last_two_sps()

        except Exception as exp:
            self.log.error(str(traceback.format_exc()))
            raise Exception(exp)

    def run(self):
        """Contains the core testcase logic and it is the one executed"""

        try:
            self.log.info('Started executing {0} testcase'.format(self.id))
            self.log.info('Running following performance tests {0}'.format(self.tests))
            self.log.info('Current attempt ID: {0}'.format(self.attempt))
            self.log.info('Current Index server: {0}'.format(self.old_index_server_name))
            self.log.info('Index Server service pack: {0}'.format(self.isc_sp))

            if self.send_report == 'only':
                self.log.info('This run is to send report only')
                self.send_email()
                return

            if self.old_index_server_name.lower() != self.isc_obj.client_name.lower():

                self.log.info('Changing index server')
                self.backupset.index_server = self.isc_obj

                self.log.info('Restarting CS services')
                cs = self.commcell.clients.get(self.cs_name)
                cs.restart_services()
                self.log.info('Waiting for 5 minutes to get CS services ready')
                time.sleep(300)

                new_index_server_name = self.backupset.index_server.client_name
                self.log.info('New Index Server: ' + new_index_server_name)

            self.log.info('Getting Index DB object')
            self.db = index_db.get(self.backupset)

            self.log.info('DB Path: [{0}]'.format(self.db.db_path))

            self.start_perf_counter()

            for test in self.tests:
                test = test.lower()
                method_name = 'test_' + test

                self.log.info('Running test: ' + method_name)
                self.log.info('Killing CVODS processes before test')
                self.isc_machine.kill_process('CVODS')

                if hasattr(self, method_name):
                    getattr(self, method_name)()
                else:
                    self.log.error('Invalid test name')

                self.set_test_process_id(test)

            self.stop_perf_counter()

            if self.send_report == 'always':
                self.send_email()

        except Exception as exp:
            self.log.error('Test case failed with error: ' + str(exp))
            self.result_string = str(exp)
            self.status = constants.FAILED
            self.log.error(str(traceback.format_exc()))

            if self.counter_id is not None:
                self.stop_perf_counter()

    def test_playback(self):
        """Runs the playback test and collects the playback time taken"""

        try:
            self.log.info('***** Running playback test *****')

            if not self.skip_delete_db:
                self.log.info('Aging checkpoints')
                self.age_checkpoints(self.db.backupset_guid)

                self.log.info('Deleting DB')
                self.db.delete_db()
                time.sleep(30)

            self.log.info('***** Initiating playback *****')
            while not self.db.is_upto_date:
                self.log.info('DB is not upto date')
                time.sleep(60)

            self.log.info('***** DB is upto date *****')

            jbs_file = self.db.db_path + self.isc_delim + 'JobStats.csv'
            self.log.info('Reading job stats file: [{0}]'.format(jbs_file))

            lines = self.isc_machine.read_csv_file(jbs_file)
            total_added = 0
            total_time = 0

            for line in lines:
                job_type = self.job_types[line['Job Type']]
                job_id = line['Job Id']
                name = '{0} job - J{1}'.format(job_type, job_id)
                deleted_items = int(line['Deleted Folders']) + int(line['Deleted Files'])
                added_items = line['Added Files']
                deleted_items_fmt = self.format_number(deleted_items)
                added_items_fmt = self.format_number(added_items)
                comments = 'Added {0} Deleted {1}'.format(added_items_fmt, deleted_items_fmt)

                total_added += int(added_items) - int(deleted_items)
                total_time += float(line['Seconds To Playback'])

                self.update_stat('playback_time', name, line['Seconds To Playback'], comments)

            if total_added:
                pt_1m_items = (total_time * 1000000)/total_added
                self.update_stat(
                    'playback_time', 'Playback time for 1 million items', str(pt_1m_items), '')

            self.push_stats_database()
            self.log.info('***** Playback tests completed *****')

        except Exception as exp:
            self.log.error('Got exception while running test: [{0}]'.format(exp))
            self.log.error(str(traceback.format_exc()))

    def test_restore(self):
        """Runs the restore test and collects the restore vector creation time taken"""

        try:
            self.log.info('***** Running restore vector tests *****')
            self.log.info('Starting latest cycle restore job')

            rst_job = self.subclient.restore_out_of_place(self.cs_name, 'e:\\restores', ['\\'])
            rst_job_id = rst_job.job_id
            time_taken = self.get_restore_vector_time(rst_job_id, 'restore')

            self.update_stat('restore_rv_time', 'Latest cycle restore', time_taken, '')
            self.push_stats_database()

            self.log.info('Killing restore job')

            try:
                rst_job.kill()
            except:
                pass

            self.log.info('Starting oldest cycle restore job')

            rst_job = self.subclient.restore_out_of_place(
                self.cs_name, 'e:\\restores', ['\\'],
                to_time=commonutils.convert_to_formatted_time(self.oldest_cycle_etime)
            )

            rst_job_id = rst_job.job_id
            time_taken = self.get_restore_vector_time(rst_job_id, 'restore')

            self.update_stat('restore_rv_time', 'Oldest cycle restore', time_taken, '')
            self.push_stats_database()

            self.log.info('Killing restore job')

            try:
                rst_job.kill()
            except:
                pass

            self.log.info('***** Restore tests completed *****')

        except Exception as exp:
            self.log.error('Got exception while running test: [{0}]'.format(exp))
            self.log.error(str(traceback.format_exc()))

    def test_synthetic_full(self):
        """Runs the synthetic full job test and collects the time taken for RV creation"""

        sfull_job = None

        try:
            self.log.info('***** Running synthetic full job test *****')
            self.log.info('Starting synthetic full job')

            sfull_job = self.subclient.backup('Synthetic_full')
            sfull_job_id = sfull_job.job_id

            time_taken = self.get_restore_vector_time(sfull_job_id, 'synthetic_full')

            self.log.info('Time taken for SFULL restore vector creation: ' + time_taken)
            self.update_stat('sfull_rv_time', 'Restore vector creation time', time_taken, '')
            self.push_stats_database()

            self.log.info('Killing synthetic full job')
            self.log.info('***** Synthetic full job tests completed *****')

            try:
                sfull_job.kill()
            except:
                pass

        except Exception as exp:
            self.log.error('Got exception while running test: [{0}]'.format(exp))
            self.log.error(str(traceback.format_exc()))

        finally:
            if sfull_job is not None:
                try:
                    sfull_job.kill()
                except:
                    pass

    def test_compaction(self):
        """Runs the compaction test and collects the compaction  time taken"""

        try:
            self.log.info('***** Running Compaction test *****')
            self.log.info('Running index checkpoint job')
            self.db.checkpoint_db(by_all_index_backup_clients=False)

            self.log.info('Running compaction operation')
            compaction_job = self.db.compact_db()

            if compaction_job:
                job_id = compaction_job.job_id
                self.log.info('Compaction job: ' + job_id)

                time.sleep(10)
                log_lines = []
                attempts = 3

                while True:

                    if not attempts:
                        self.log.info('Compaction check attempts exhausted !')
                        break

                    self.log.info('Getting compaction time from IndexServer.log')

                    db_guid = self.db.backupset_guid
                    log_lines = self.read_log(
                        'IndexServer.log', [job_id, 'secs to compact DB', db_guid])

                    if log_lines:
                        break
                    else:
                        self.log.info('Cannot get compaction time trying again')
                        attempts -= 1
                        time.sleep(60)

                self.log.info('Got compaction time')
                line = log_lines[-1]
                matches = re.findall('\[([0-9.]+)\]', line)
                time_taken = matches[0]

                self.log.info('Time taken for compaction operation: ' + time_taken)
                self.update_stat('compaction_time', 'Compaction time', time_taken, '')
                self.push_stats_database()

                self.log.info('***** Compaction tests completed *****')

        except Exception as exp:
            self.log.error('Got exception while running test: [{0}]'.format(exp))
            self.log.error(str(traceback.format_exc()))

    def test_browse(self):
        """Runs the find test and collects the find time taken"""

        try:
            self.log.info('***** Running browse test *****')

            start_time = time.time()
            attempts = 1

            while attempts <= 5:
                self.log.info('Doing latest cycle filtered find operation. Attempt [{0}/5]'.format(
                    attempts))
                try:
                    self.backupset.find({
                        'filters': [('FileName', '*a*')],
                        'page_size': 1000
                    })
                    break
                except SDKException as e:
                    self.log.error('Browse request timed out. Error [{0}]'.format(e))
                    attempts += 1
                    time.sleep(10)
                    continue

            stop_time = time.time()
            diff = self.limit_decimals(stop_time - start_time)

            self.log.info('Time taken for latest cycle filtered find operation: ' + str(diff))
            self.update_stat('find_time', 'Latest cycle filtered', str(diff), '')

            self.log.info('Doing oldest cycle filtered find operation')

            start_time = time.time()
            self.backupset.find({
                'filters': [('FileName', '*a*')],
                'page_size': 1000,
                'to_time': self.oldest_cycle_etime
            })
            stop_time = time.time()
            diff = self.limit_decimals(stop_time - start_time)

            self.log.info('Time taken for oldest filtered find operation: ' + str(diff))
            self.update_stat('find_time', 'Oldest cycle filtered', str(diff), '')

            self.push_stats_database()

            self.log.info('***** Browse tests completed *****')

        except Exception as exp:
            self.log.error('Got exception while running test: [{0}]'.format(exp))
            self.log.error(str(traceback.format_exc()))

    def create_stats_table(self):
        """Creates the two tables required to store the performance statistics"""

        resp = self.history_db.execute("SELECT * FROM INFORMATION_SCHEMA.TABLES "
                                       "WHERE TABLE_NAME = 'IndexingPerfStats'")

        if len(resp.rows) == 0:
            self.history_db.execute('''
                CREATE TABLE [dbo].[IndexingPerfStats](
                    [id] [bigint] IDENTITY(1,1) NOT NULL,
                    [attempt] [bigint] NOT NULL,
                    [service_pack] [varchar](max) NULL,
                    [type] [varchar](max) NULL,
                    [name] [varchar](max) NULL,
                    [value] [float] NULL,
                    [comments] [varchar](max) NULL
                )
            ''')

            self.log.info('Indexing performance stats table created successfully')
        else:
            self.log.info('Indexing performance stats table already exists')

        resp = self.history_db.execute("SELECT * FROM INFORMATION_SCHEMA.TABLES "
                                       "WHERE TABLE_NAME = 'IndexingProcessStats'")

        if len(resp.rows) == 0:
            self.history_db.execute('''
                CREATE TABLE [dbo].[IndexingProcessStats](
                    [attempt] [bigint] NULL,
                    [service_pack] [bigint] NULL,
                    [test] [varchar](max) NULL,
                    [time] [bigint] NULL,
                    [name] [varchar](max) NULL,
                    [handles] [bigint] NULL,
                    [threads] [bigint] NULL,
                    [memory] [bigint] NULL
                )
            ''')

            self.log.info('Indexing process stats table created successfully')
        else:
            self.log.info('Indexing process stats table already exists')

    def push_stats_database(self):
        """Pushes the collected data to the DB"""

        self.log.info('Pushing stats to database')
        query = "insert into IndexingPerfStats " \
                "(attempt, service_pack, type, name, value, comments) values "

        for stat in self.current_stats:
            values = "','".join([self.attempt, self.isc_sp,
                                 stat['type'], stat['name'], stat['value'], stat['comments']])
            query += "('" + values + "'),"

        final_query = query[:-1]
        self.log.info('Inserting records: ' + final_query)
        self.history_db.execute(final_query)

        self.current_stats = []

    def update_stat(self, stype, name, value, comments):
        """Records the collected stat in memory"""

        self.current_stats.append({
            'type': stype,
            'name': name,
            'value': value,
            'comments': comments
        })

    def get_attempt(self):
        """Gets the last attempt for the current service pack of the current run"""

        self.log.info('Getting last performance test attempt ID for SP: {0}'.format(self.isc_sp))
        resp = self.history_db.execute("select max(attempt) as 'attempt' from IndexingPerfStats "
                                       "where service_pack = {0}".format(self.isc_sp))
        attempt = resp.rows[0]['attempt']

        if attempt is None:
            return '1'
        else:
            return str(int(attempt) + 1)

    def get_log_file(self, name):
        """Gets the log file path on IndexServer MA for the given log name"""

        return self.isc_machine.join_path(self.isc_obj.install_directory, 'Log Files', name)

    def get_oldest_cycle_time(self):
        """Gets the oldest cycle's job end time"""

        query = ("select top 1 * from jmbkpstats where appid = {0}"
                 " and fullCycleNum = (select fullcyclenum from jmbkpstats where "
                 "jobid = (select min(jobid) from archfile where appid = {0} and "
                 "filetype = 2)) order by jobid asc".format(self.subclient.subclient_id))

        resp = self.cs_db.execute(query)
        self.log.info(query)
        self.log.info(resp.rows)
        return resp.rows[0]['servEndDate']

    def get_last_two_sps(self):
        """Gets the last two service packs for which numbers were collected"""

        resp = self.history_db.execute('select max(service_pack) as "sp" from IndexingPerfStats')
        if resp.rows and resp.rows[0]['sp'] is not None:
            self.high_sp = resp.rows[0]['sp']
            self.low_sp = str(int(self.high_sp) - 1)

        self.log.info('Comparing stats for service packs {0} and {1}'.format(
            self.low_sp, self.high_sp))

    def read_log(self, name, words):
        """Reads the log files for the given words"""

        tries = 1
        while tries <= 5:
            try:
                log_file = self.get_log_file(name)
                out = self.isc_machine.find_lines_in_file(log_file, words)
                return list(filter(len, out))
            except socket.gaierror as e:
                self.log.error('Got exception in getaddrinfo. Retrying again [{0}]'.format(e))
                tries += 1
                time.sleep(60)

    def age_checkpoints(self, backupset_guid):
        """Ages all the checkpoints of the DB"""

        query = "delete from archfile where name like '%{0}%'".format(backupset_guid)
        self.log.info('Age checkpoint query: [{0}]'.format(query))
        self.cs_db.execute(query)

    @staticmethod
    def limit_decimals(value, digits=3):
        """Returns a number with restricted numbers after decimal point"""

        format_specifier = '{0:.' + str(digits) + 'f}'
        return format_specifier.format(value)

    @staticmethod
    def format_number(num, units=None, divide_by=1000):
        """Formats the given number with units"""

        if units is None:
            units = ['', 'K', 'M', 'T']

        try:
            num = int(num)
        except ValueError:
            num = 0

        unit = ''
        for unit in units:
            if num < divide_by:
                return '{0}{1}'.format(int(num), unit)

            num /= divide_by

        return '{0}{1}'.format(int(num), unit)

    def start_perf_counter(self):
        """Starts the powershell based performance counter on the IndexServer machine"""

        counter_file = 'c:\\perf_counter.ps1'
        counter_report_file = 'c:\\perf_stat_result.csv'
        counter_code = """
function GetIndexingProcID([String] $name = ""){
    $ids = Get-WmiObject Win32_Process | Where-Object {$_.CommandLine -like "*$name*"} | select-object -ExpandProperty ProcessID 
    return @($ids)[0]
}

function WriteProcessInfo([Int] $id = "", [String] $name = "")
{
    if($id -eq ""){
        return $false;
    }
    $pi = Get-WmiObject -class Win32_PerfFormattedData_PerfProc_Process | where-object {$_.idprocess -eq $id}
    $table = @{}
    $table.Add("id", $pi.IDProcess)
    $table.Add("name", $name)
    $table.Add("handles", $pi.HandleCount)
    $table.Add("threads", $pi.ThreadCount)
    $table.Add("memory", $pi.WorkingSetPrivate)

    $ourObject = New-Object -TypeName psobject -Property $table
    $ourObject | export-csv -path "c:\perf_stat_result.csv" -NoTypeInformation -append -Force
}

while ($true){
    $isp_id = GetIndexingProcID "IndexServer"
    #$lmp_id = GetIndexingProcID "LogManager"

    WriteProcessInfo $isp_id "IndexServer"
    #WriteProcessInfo $lmp_id "LogManager"
    start-sleep -Seconds 30
}"""

        self.isc_machine.delete_file(counter_report_file)

        self.log.info('Creating performance counter file')
        self.isc_machine.execute_command(
            "set-content -path '" + counter_file + "' -value '" + counter_code + "'")

        self.log.info('Starting performance counter')
        counter_id_cmd = self.isc_machine.execute_command(
            '(start-process -filepath "powershell.exe" -ArgumentList "{0}" '
            '-Verb RunAs -passthru).ID'.format(counter_file))

        self.counter_id = counter_id_cmd.formatted_output
        self.log.info('Started performance counter. Process ID [{0}]'.format(self.counter_id))

    def stop_perf_counter(self):
        """Stops the performance counting process on the IndexServer machine"""

        self.log.info('Killing performance counter process')
        self.isc_machine.kill_process(process_id=self.counter_id)

        counter_report_file = 'c:\\perf_stat_result.csv'

        lines = self.isc_machine.read_csv_file(counter_report_file)
        lines = list(lines)

        # Splitting lines to 900 per set as we cannot insert more than 1000 records at once
        lines_set = [lines[i:i+900] for i in range(0, len(lines), 900)]

        self.log.info(self.test_process_id)

        stat_time = 0
        for lines in lines_set:
            query = "insert into IndexingProcessStats " \
                    "(attempt, service_pack, test, time, name, handles, threads, memory) values "

            for line in lines:

                process_id = line['id']
                test_name = self.test_process_id.get(process_id, '')

                values = "','".join(
                    [self.attempt, self.isc_sp, test_name, str(stat_time),
                     line['name'], line['handles'], line['threads'],
                     line['memory']]
                )

                query += "('" + values + "'),"
                stat_time += 1

            final_query = query[:-1]
            self.log.info('Pushing process stats to DB')
            self.log.info('Inserting records: ' + final_query)
            self.history_db.execute(final_query)

        self.counter_id = None

    def set_test_process_id(self, test_name):
        """Records the test done by the current IndexServer process"""

        try:
            self.log.info('Getting running index server process id')
            process_id = self.get_index_server_process_id()
            self.test_process_id[process_id] = test_name
        except Exception as e:
            self.log.info('Got exception while setting index server process if for the test')

    def set_ticks_graph(self, service_pack):
        """Sets the test name on the x axis of the graph"""

        try:

            start_times, test_names = self.get_test_start_time(service_pack=service_pack)

            if not start_times or not test_names:
                return

            ax = plt.gca()
            ax.set_xticks(start_times)
            ax.set_xticklabels(test_names)

        except Exception as e:
            self.log.error('Got exception while labelling x axis: [{0}]'.format(e))

    def get_index_server_change(self):
        """Gets the index server machine to use for the current run"""

        isc_list = self.tcinputs['IndexServer'].split(',')
        isc_list = [i.strip().lower() for i in isc_list]

        if len(isc_list) == 1:
            return isc_list[0]

        current_is = self.backupset.index_server.client_name.lower()

        try:
            current_is_idx = isc_list.index(current_is)
        except ValueError:
            return isc_list[0]

        try:
            return isc_list[current_is_idx+1]
        except IndexError:
            return isc_list[0]

    def get_index_server_process_id(self):
        """Gets the currently running IndexServer process ID"""

        out = self.isc_machine.execute_command("""@(Get-WmiObject Win32_Process | Where-Object {$_.CommandLine -like "*IndexServer*"} | select-object -ExpandProperty ProcessID)[0]""")
        self.log.info('Current IndexServer process ID: ' + out.formatted_output)

        return out.formatted_output

    def get_time_from_logs(self, words, log_file='Browse.log'):
        """Gets the time value for the log line"""

        while True:
            self.log.info('Looking for words {0}'.format(words))
            log_lines = self.read_log(log_file, words)

            if log_lines:
                break
            else:
                self.log.info('Words not found. Trying again.')
                time.sleep(180)

        line = log_lines[-1]
        matches = re.findall('[0-9]+\/[0-9]+ [0-9]+\:[0-9]+\:[0-9]+', line)
        time_raw = matches[0]
        time_format = '%m/%d %H:%M:%S'

        return datetime.strptime(time_raw, time_format)

    def get_restore_vector_time(self, job_id, job_type):
        """Gets the time taken to create restore vector"""

        time.sleep(10)
        self.log.info('Job ID for restore vector creation [{0}]'.format(job_id))
        job_id_search = ' {0} '.format(job_id)

        if job_type == 'restore':
            end_text = 'Time taken to create restore'
        else:
            end_text = 'Successfully created restore vector at'

        self.log.info('Checking if restore vector creation STARTED in Browse.log')
        start_time = self.get_time_from_logs([job_id_search, 'Received Browse Request From Client'])

        self.log.info('Checking if restore vector creation ENDED in Browse.log')
        end_time = self.get_time_from_logs([job_id_search, end_text])

        time_diff = end_time-start_time
        return str(time_diff.seconds)

    def get_test_start_time(self, service_pack):
        """Gets the start time of the test"""

        resp = self.history_db.execute("""select test, MIN(time) as 'start_time' from 
        indexingprocessstats where service_pack = '{0}' and name = 'IndexServer' and 
        attempt = (select max(attempt) from indexingprocessstats where service_pack = '{0}')
        group by test""".format(service_pack))

        start_times = []
        test_names = []

        for row in resp.rows:
            test_name = row['test']
            start_time = row['start_time']

            start_times.append(start_time)
            test_names.append(test_name)

        return start_times, test_names

    def get_compare_results(self):
        """Runs query to compares the performance stat results"""

        resp = self.history_db.execute("""
IF object_id('tempdb.dbo.#prevsp') is not null
    DROP TABLE #prevsp

IF object_id('tempdb.dbo.#cursp') is not null
    DROP TABLE #cursp

CREATE TABLE #prevsp (service_pack varchar(max), type varchar(max), name varchar(max), value float)
CREATE TABLE #cursp (service_pack varchar(max), type varchar(max), name varchar(max), value float, id bigint)

declare @curspnum varchar(max) = (select max(service_pack) from IndexingPerfStats)
declare @prevspnum varchar(max) = @curspnum-1
declare @avgattempts bigint = 3

insert into #prevsp
select service_pack, type, name, avg(value) from IndexingPerfStats
where service_pack = @prevspnum
and attempt > (select max(attempt) from IndexingPerfStats where service_pack = @prevspnum)-@avgattempts
group by service_pack, type, name

insert into #cursp
select service_pack, type, name, avg(value), max(id) from IndexingPerfStats
where service_pack = @curspnum
and attempt > (select max(attempt) from IndexingPerfStats where service_pack = @curspnum)-@avgattempts
group by service_pack, type, name

--select * from #prevsp
--select * from #cursp

select cursp.type, cursp.name, prevsp.value as 'spl', cursp.value as 'sph', 
(select comments from IndexingPerfStats where id = cursp.id) as 'comments'
from #cursp cursp
left join #prevsp prevsp on prevsp.type = cursp.type and prevsp.name = cursp.name
where cursp.service_pack = @curspnum
order by cursp.id asc

DROP TABLE #prevsp
DROP TABLE #cursp
        """)

        return resp.rows

    def get_compare_process_stats(self, stat):
        """Runs query to compares the process stats results"""

        resp = self.history_db.execute("""
IF object_id('tempdb.dbo.#prevsp') is not null
    DROP TABLE #prevsp

IF object_id('tempdb.dbo.#cursp') is not null
    DROP TABLE #cursp

declare @curspnum varchar(max) = (select max(service_pack) from IndexingProcessStats)
declare @prevspnum varchar(max) = @curspnum-1
declare @curspattempt int = (select max(attempt) from IndexingProcessStats where service_pack = @curspnum)
declare @prevspattempt int = (select max(attempt) from IndexingProcessStats where service_pack = @prevspnum)

CREATE TABLE #prevsp (service_pack varchar(max), test varchar(max), value float)
CREATE TABLE #cursp (service_pack varchar(max), test varchar(max), value float)

insert #prevsp
select service_pack, test, avg({0}) as 'value' from indexingprocessstats where service_pack = @prevspnum and attempt = @prevspattempt group by service_pack, test

insert #cursp
select service_pack, test, avg({0}) as 'value' from indexingprocessstats where service_pack = @curspnum and attempt = @curspattempt group by service_pack, test

select cursp.test, prevsp.value as 'spl', cursp.value as 'sph'
from #cursp cursp
join #prevsp prevsp on prevsp.test = cursp.test

DROP TABLE #prevsp
DROP TABLE #cursp
        """.format(stat))

        return resp.rows

    def get_process_graph_data(self, service_pack, process='IndexServer', metric='memory'):
        """Retreives the process stats from the DB"""

        resp = self.history_db.execute("""select * from indexingprocessstats where 
            service_pack = '{0}' and name = '{1}' and attempt = (select max(attempt) from 
            indexingprocessstats where service_pack = '{0}') order by time
            """.format(service_pack, process))

        x = []
        y = []

        for idx, row in enumerate(resp.rows):
            x.append(idx)
            y.append(row[metric])

        return x, y

    def create_graph(self, graph_type):
        """Creates the image tag for the graph"""

        ax = plt.gca()
        ax.set_xlabel('Time')

        if graph_type == 'memory':
            ax.set_ylabel('Memory usage (MB)')
            ax.set_yticklabels([self.limit_decimals(tick / (1024 * 1024), 2)
                                for tick in ax.get_yticks()])

        if graph_type == 'handles':
            ax.set_ylabel('Handles')

        if graph_type == 'threads':
            ax.set_ylabel('Threads')

        fig = plt.gcf()
        fig.set_size_inches(15, 3)
        fig.tight_layout()

        plt.legend()

        bytes_io = BytesIO()
        plt.savefig(bytes_io)
        bytes_io.seek(0)
        base64_data = base64.b64encode(bytes_io.read())
        string_data = base64_data.decode('utf-8')
        plt.clf()

        return '<img src="data:image/png;base64,' + string_data + '"/>'

    def create_graph_stats(self, stat, service_pack, color):
        """Creates graphs for the process stats"""

        label = 'SP' + service_pack
        isp_memory = self.get_process_graph_data(service_pack=service_pack, metric=stat)
        plt.plot(isp_memory[0], isp_memory[1], label=label, color=color)
        self.set_ticks_graph(service_pack)

        return self.create_graph(stat)

    def build_table_perf_stats(self):
        """Generates HTML for the performance stats table"""

        html = """<table border="1"><tr>
        <th>Test</th>
        <th>SP{0}</th>
        <th>SP{1}</th>
        <th>% Change</th>
        <th>Comments</th>
        </tr>""".format(
            self.low_sp, self.high_sp
        )

        last_type = None
        type_dict = {
            'playback_time': 'Playback time',
            'find_time': 'Find time',
            'restore_rv_time': 'Restore vector creation time',
            'sfull_rv_time': 'Synthetic full job vector creation time',
            'compaction_time': 'Compaction time',
        }

        rows = self.get_compare_results()

        for row in rows:
            lsp_raw = row['spl'] if row['spl'] is not None else 0
            hsp_raw = row['sph'] if row['sph'] is not None else 0
            lsp = float(lsp_raw)
            hsp = float(hsp_raw)
            percent_increase = 0

            try:
                percent_increase = (((hsp-lsp)/lsp)*100)
            except ZeroDivisionError:
                pass

            state_table = {
                0: ['darkGreen', '&#9660;'],
                1: ['#a96500', '&#9650;'],
                2: ['red', '&#9650;']
            }

            state = 0
            if hsp > lsp:
                state = 2 if percent_increase > 10 else 1

            unit = 's'
            if hsp > 60 and lsp > 60:
                hsp = hsp/60
                lsp = lsp/60
                unit = 'min'

            hsp_color = state_table[state][0]
            hsp_value = str(self.limit_decimals(hsp, 2))
            lsp_value = str(self.limit_decimals(lsp, 2))
            p_increase = (str(self.limit_decimals(percent_increase, 2)) + '% ' +
                          state_table[state][1])

            if last_type != row['type']:
                html += '<tr><th colspan="5" style="background-color: #62a0d0; padding: 5px 10px">'
                html += type_dict.get(row['type'], '') + '</th></tr>'

            html += '<tr>'
            html += '<td>' + row['name'] + '</td>'
            html += '<td>' + lsp_value + unit + '</td>'
            html += '<td>' + hsp_value + unit + '</td>'
            html += '<td style="color:' + hsp_color + ';">' + p_increase + '</td>'
            html += '<td>' + row['comments'] + '</td>'
            html += '</tr>'

            last_type = row['type']

        html += '</table>'

        return html

    def build_table_process_stats(self, stat):

        html = """<table border="1"><tr>
        <th>Test (average {2})</th>
        <th>SP{0}</th>
        <th>SP{1}</th>
        </tr>""".format(
            self.low_sp, self.high_sp, stat
        )

        rows = self.get_compare_process_stats(stat)

        for row in rows:
            if stat == 'memory':
                lsp_value = self.limit_decimals(int(row['spl']) / (1024 * 1024), 2) + 'MB'
                hsp_value = self.limit_decimals(int(row['sph']) / (1024 * 1024), 2) + 'MB'
            else:
                lsp_value = self.limit_decimals(row['spl'], 0)
                hsp_value = self.limit_decimals(row['sph'], 0)

            html += '<tr>'
            html += '<td>' + row['test'] + '</td>'
            html += '<td>' + lsp_value + '</td>'
            html += '<td>' + hsp_value + '</td>'
            html += '</tr>'

        html += '</table>'

        return html

    def build_email(self):
        """Generates the HTML for the email report"""

        if self.low_sp is None or self.high_sp is None:
            self.log.info('Stats not available for one of the SP. Not building report')
            return ''

        html = """<style>table{border-collapse: collapse;} th{  
        background: #2c5e84; color: #fff; } th, td{ padding: 10px; } </style>"""

        html += '<h2>SP{0} vs SP{1} performance stats</h2>'.format(self.low_sp, self.high_sp)
        html += '<p>Note: Below numbers are an average of last three attempts.</p>'
        html += self.build_table_perf_stats()

        html += '<h2>IndexServer process stats during above tests</h2>'
        html += '<p>Note: Before every test, IndexServer process is killed and newly launched</p>'

        html += '<h3>RAM consumption</h3>'

        html += self.build_table_process_stats('memory')
        html += self.create_graph_stats('memory', self.low_sp, 'orange')
        html += self.create_graph_stats('memory', self.high_sp, 'blue')

        html += '<h3>Handles opened</h3>'

        html += self.build_table_process_stats('handles')
        html += self.create_graph_stats('handles', self.low_sp, 'orange')
        html += self.create_graph_stats('handles', self.high_sp, 'blue')

        html += '<h3>Threads launched</h3>'

        html += self.build_table_process_stats('threads')
        html += self.create_graph_stats('threads', self.low_sp, 'orange')
        html += self.create_graph_stats('threads', self.high_sp, 'blue')

        html += '<h3>Setup</h3>'
        html += 'CS: {0}<br/>'.format(self.commcell.commserv_hostname)
        html += 'MA: {0}'.format(self.isc_obj.client_hostname)

        return html

    def send_email(self):
        """Sends the report email"""

        self.log.info('Building report and sending email')

        html = self.build_email()

        sp_text = ''
        if self.low_sp is not None and self.high_sp is not None:
            sp_text = ' - SP{0} vs SP{1}'.format(self.low_sp, self.high_sp)

        from AutomationUtils.mailer import Mailer
        mailer = Mailer(mailing_inputs={}, commcell_object=self.commcell)
        mailer.mail('Indexing performance comparison' + sp_text, html)

    def tear_down(self):
        """Cleanup operation"""
        if self.counter_id is not None:
            self.log.info('Killing performance counter process')
            self.isc_machine.kill_process(process_id=self.counter_id)

        self.name = self.name + ' - SP' + self.isc_sp

        self.isc_machine.disconnect()
