# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
""" Custom Report Alert """
import datetime


from Web.AdminConsole.adminconsole import AdminConsole
from Web.AdminConsole.Reports.manage_reports import ManageReport
from Web.AdminConsole.Adapter.WebConsoleAdapter import WebConsoleAdapter
from Web.AdminConsole.Reports.manage_alerts import ManageAlerts

from Web.Common.cvbrowser import BrowserFactory
from Web.Common.cvbrowser import Browser
from Web.Common.exceptions import CVTestCaseInitFailure, CVTestStepFailure
from Web.Common.page_object import TestStep

from Web.WebConsole.Reports.cte import ConfigureAlert
from Web.WebConsole.webconsole import WebConsole
from Web.WebConsole.Reports.Custom import viewer

from AutomationUtils.cvtestcase import CVTestCase

from Reports.Custom.utils import CustomReportUtils
from Reports.reportsutils import get_reports_config
REPORTS_CONFIG = get_reports_config()


class AlertDetails:
    """Set alert details"""
    column_name = None
    condition = None
    hours = None
    minutes = None
    ampm = None

    def __init__(self, column_name, condition, report_name, report_url):
        AlertDetails.column_name = column_name
        AlertDetails.condition = condition
        self.report_name = report_name
        self.report_url = report_url
        self.generate_alert_time()

    @classmethod
    def generate_alert_time(cls):
        """Generate alert time to set in alert"""
        now = datetime.datetime.now()
        now_plus_2mins = now + datetime.timedelta(minutes=2)
        AlertDetails.hours = str(int(datetime.datetime.strftime(now_plus_2mins, "%I")))
        AlertDetails.minutes = str(int(datetime.datetime.strftime(now_plus_2mins, "%M")))
        if len(AlertDetails.minutes) == 1:
            AlertDetails.minutes = "0" + AlertDetails.minutes
        AlertDetails.ampm = str(datetime.datetime.strftime(now_plus_2mins, "%p"))


class TestCase(CVTestCase):
    """
    TestCase class used to execute the test case from here.
    """
    test_step = TestStep()

    def __init__(self):
        super(TestCase, self).__init__()
        self.name = "Admin Console Report Alerts"
        self.browser = None
        self.webconsole = None
        self.manage_report = None
        self.navigator = None
        self.alert = None
        self.admin_console = None
        self.utils = CustomReportUtils(self)

    def init_tc(self):
        """Initial configuration for the test case"""
        try:
            self.browser = BrowserFactory().create_browser_object(name="ClientBrowser")
            self.browser.open()
            self.webconsole = WebConsole(self.browser, self.commcell.webconsole_hostname)
            self.admin_console = AdminConsole(self.browser, self.commcell.webconsole_hostname)
            self.admin_console.login()
            self.navigator = self.admin_console.navigator
            self.manage_report = ManageReport(self.admin_console)
            self.navigator.navigate_to_reports()
            self.cleanup_alerts()
            self.manage_report.access_report(REPORTS_CONFIG.REPORTS.CUSTOM[0])
        except Exception as exception:
            raise CVTestCaseInitFailure(exception) from exception

    def cleanup_alerts(self):
        """ Deletes the alert if exists """
        self.manage_report.view_alerts()
        self.alert = ManageAlerts(self.admin_console)
        if self.name in self.alert.get_all_alerts():
            self.alert.delete_alerts([self.name])
        self.navigator.navigate_to_reports()

    @test_step
    def create_alert(self):
        """Create alert"""
        report_viewer = viewer.CustomReportViewer(self.webconsole)
        table = viewer.DataTable("Summary")
        report_viewer.associate_component(table)
        columns = table.get_table_columns()
        if not columns:
            raise CVTestStepFailure(f"Report [{self.name}] might be empty. Please verify!")
        condition_string = table.get_table_data()[columns[0]][0]
        self.log.info("Creating alert for [%s] report for [%s] column with condition string:"
                      "[%s]", self.name, columns[0], condition_string)
        table.configure_alert()
        alert = AlertDetails(columns[0], condition_string, report_name=self.name,
                             report_url=self.browser.driver.current_url)
        adapter = WebConsoleAdapter(self.admin_console, self.browser)
        alert_window = ConfigureAlert(adapter)
        alert_window.set_time(alert.hours, alert.minutes, alert.ampm)
        alert_window.create_alert(alert_name=alert.report_name, column_name=columns[0],
                                  column_value=condition_string)

    @test_step
    def delete_alert(self):
        """Delete alerts"""
        self.navigator.navigate_to_reports()
        self.manage_report.view_alerts()
        if self.name not in self.alert.get_all_alerts():
            raise CVTestStepFailure(f"Alert {self.name} is missing after creation")
        self.alert.delete_alerts([self.name])

    def run(self):
        try:
            self.init_tc()
            self.create_alert()
            self.delete_alert()
        except Exception as err:
            self.utils.handle_testcase_exception(err)
        finally:
            AdminConsole.logout_silently(self.admin_console)
            Browser.close_silently(self.browser)
