# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright ©2016 Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    setup()         --  setup function of this test case

    run()           --  run function of this test case

    tear_down()     --  tear down function of this test case
"""
import datetime
import time
import calendar
from AutomationUtils import logger, constants
from AutomationUtils.cvtestcase import CVTestCase
from MediaAgents.MAUtils.mahelper import MMHelper

class TestCase(CVTestCase):
    """Class for executing this test case"""

    def __init__(self):
        """Initializes test case class object
        """
        super(TestCase, self).__init__()
        self.name = "Extended retention basic case"
        self.tcinputs = {
            "MediaAgentName": None,
            "MountPath": None,
            "ContentPath": None,
            "SqlSaPassword": None
        }

    def setup(self):
        """Setup function of this test case"""
        self._log = logger.get_log()

    def run(self):
        """Run function of this test case"""
        try:
            self._log.info("Started executing {0} testcase".format(self.id))

            self._log.info(self.name)
            self.library_name = str(self.id) + "_lib"
            self.storage_policy_name = str(self.id) + "_SP"
            self.backupset_name = str(self.id) + "_BS"
            self.subclient_name = str(self.id) + "_SC"

            # initialize MMHelper class
            mmhelper = MMHelper(self)

            try:
                mmhelper.cleanup()
            except Exception as e:
                self._log.info("error while cleanup - ignoring")
                pass

            # calculation to get last friday and month's first day
            d = datetime.date.today()
            last_friday = d
            oneday = datetime.timedelta(days=1)
            while last_friday.weekday() != calendar.FRIDAY:
                last_friday -= oneday
            last_friday_timestamp = int(time.mktime(last_friday.timetuple()))
            self._log.info("last fri:" + str(last_friday_timestamp))

            y, m = d.year, d.month
            a, m = divmod(m-1, 12)
            month_day = datetime.date(y + a, m + 1, 1)
            month_day_timestamp = int(time.mktime(month_day.timetuple()))
            self._log.info("month first: "+ str(month_day_timestamp))

            (self.disk_library,
             self.storage_policy,
             self._backupset,
             self._subclient) = mmhelper.setup_environment()

            # update retention to 1 day, 0 cycle
            self._log.info("setting retention to: 1 day, 0 cycle")
            self.copy = self.storage_policy.get_copy('Primary')
            self.copy.copy_retention = (1, 0, 1)
            self.copy.copy_retention_managed_disk_space = False

            # update extended retention: 2 days and 1 week
            self._log.info("setting extended retnetion to: last full (1) 2 days - Daily full, "
                           "(2) 7 days - Weekly full "
                           "(3) 30 days - Monthly full")
            self.copy.extended_retention_rules = [1, True, "EXTENDED_DAY", 2, 0]
            self.copy.extended_retention_rules = [2, True, "EXTENDED_WEEK", 7, 0]
            self.copy.extended_retention_rules = [3, True, "EXTENDED_MONTH", 30, 0]

            # Day 1 -------------------
            self._log.info("--------- Day 1 -----------")
            # Run 3 FULL backup - Day 1
            self._log.info("Run 3 Full Backup jobs - Day 1")
            backup_jobs_day1 = []
            for i in range(0, 3):
                self._log.info("Running full backup J{0}...".format(i+1))
                job = self._subclient.backup("FULL")
                self._log.info("Backup job{0}: {1}".format(i+1, job.job_id))
                if not job.wait_for_completion():
                    raise Exception(
                        "Failed to run FULL backup with error: {0}".format(job.delay_reason)
                    )
                self._log.info("Backup job completed.")
                backup_jobs_day1 += [job.job_id]

            # Run the script to move backup to 1 day
            for job in backup_jobs_day1:
                mmhelper.move_job_start_time(job, 1)

            # Day 2 -------------------
            self._log.info("--------- Day 2 -----------")
            # Run 3 FULL backup - Day 2
            self._log.info("Run 3 Full Backup jobs - Day 2")
            time2 = int(time.time())

            backup_jobs_day2 = []
            for i in range(0, 3):
                self._log.info("Running full backup J{0}...".format(i+4))
                job = self._subclient.backup("FULL")
                self._log.info("Backup job{0}: {1}".format(i+4, job.job_id))
                if not job.wait_for_completion():
                    raise Exception(
                        "Failed to run FULL backup with error: {0}".format(job.delay_reason)
                    )
                self._log.info("Backup job completed.")
                backup_jobs_day2 += [job.job_id]

            # Run Data aging 1
            self._log.info("Running Data aging...")
            da_job = self.commcell.run_data_aging('Primary', self.storage_policy_name)
            self._log.info("data aging job: {0}".format(da_job.job_id))
            if not da_job.wait_for_completion():
                raise Exception(
                    "Failed to run data aging with error: {0}".format(da_job.delay_reason)
                )
            self._log.info("Data aging job completed.")

            # Validate - J1 and J2 are aged
            self._log.info("VALIDATION: all but last Day 1 jobs are aged")
            for i in range(0, 2):
                retcode = mmhelper.validate_job_prune(backup_jobs_day1[i], self.copy.copy_id)
                if retcode:
                    self._log.info("Validation success")
                else:
                    raise Exception(
                        "Backup job {0} is expected to age".format(backup_jobs_day1[i])
                    )

            # Validate - J3 should not age
            self._log.info("VALIDATION: Daily full - backup job J3 not aged")
            retcode = mmhelper.validate_job_prune(backup_jobs_day1[2], self.copy.copy_id)
            if not retcode:
                self._log.info("Validation success")
            else:
                raise Exception(
                    "Backup job {0} is not expected to age".format(backup_jobs_day1[2])
                )

            self._log.info("VALIDATION: Daily full retention flag set in JMDataStats table")
            retcode = mmhelper.validate_job_retentionflag(backup_jobs_day1[2], "EXTENDED_DAY")
            if retcode:
                self._log.info("Validation success")
            else:
                raise Exception(
                    """Backup job {0} is not set with Daily full retention flag
                    """.format(backup_jobs_day1[2])
                )

            # Run the script to move backup to 1 day
            mmhelper.move_job_start_time(backup_jobs_day1[2], 1)
            for job in backup_jobs_day2:
                mmhelper.move_job_start_time(job, 1)

            # Day 3 -------------------
            self._log.info("--------- Day 3 -----------")
            time3 = int(time.time())

            # Run 3 FULL backup - Day 3
            self._log.info("Run 3 Full Backup jobs - Day 3")
            backup_jobs_day3 = []
            for i in range(0, 3):
                self._log.info("Running full backup J{0}...".format(i+7))
                job = self._subclient.backup("FULL")
                self._log.info("Backup job{0}: {1}".format(i+7, job.job_id))
                if not job.wait_for_completion():
                    raise Exception(
                        "Failed to run FULL backup with error: {0}".format(job.delay_reason)
                    )
                self._log.info("Backup job completed.")
                backup_jobs_day3 += [job.job_id]

            # Run Data aging 2
            self._log.info("Running Data aging...")
            da_job = self.commcell.run_data_aging('Primary', self.storage_policy_name)
            self._log.info("data aging job: {0}".format(da_job.job_id))
            if not da_job.wait_for_completion():
                raise Exception(
                    "Failed to run data aging with error: {0}".format(da_job.delay_reason)
                )
            self._log.info("Data aging job completed.")

            # Validation Day3 - J6 should not age
            self._log.info("VALIDATION: Daily full - backup job J6 not yet aged")
            retcode = mmhelper.validate_job_prune(backup_jobs_day2[2], self.copy.copy_id)
            if not retcode:
                self._log.info("Validation success")
            else:
                raise Exception(
                    "Backup job {0} is not expected to age".format(backup_jobs_day2[2])
                )

            self._log.info("VALIDATION: Daily full retention flag set in JMDataStats table")
            retcode = mmhelper.validate_job_retentionflag(backup_jobs_day2[2], "EXTENDED_DAY")
            if retcode:
                self._log.info("Validation success")
            else:
                raise Exception(
                    """Backup job {0} is not set with Daily full retention flag
                    """.format(backup_jobs_day2[2])
                )

            self._log.info("VALIDATION: Daily full - backup job J3 should be aged")
            retcode = mmhelper.validate_job_prune(backup_jobs_day1[2], self.copy.copy_id)
            if retcode:
                self._log.info("Validation success")
            else:
                raise Exception(
                    "Backup job {0} is expected to age".format(backup_jobs_day1[2])
                )

            # Run the script to move backup to 4 day
            mmhelper.move_job_start_time(backup_jobs_day2[2], 4)
            for job in backup_jobs_day3:
                mmhelper.move_job_start_time(job, 4)

            # move time flags
            time2 -= (4 * 24 * 60 * 60)
            time3 -= (4 * 24 * 60 * 60)
            self._log.info("time2: " + str(time2))
            self._log.info("time3: " + str(time3))

            # Day 7 ---------------------
            self._log.info("--------- Day 7 -----------")
            # Run 3 FULL backup - Day 7
            self._log.info("Run 3 Full Backup jobs - Day 7")
            time4 = int(time.time())

            backup_jobs_day7 = []
            for i in range(0, 3):
                self._log.info("Running full backup J{0}...".format(i+10))
                job = self._subclient.backup("FULL")
                self._log.info("Backup job{0}: {1}".format(i+10, job.job_id))
                if not job.wait_for_completion():
                    raise Exception(
                        "Failed to run FULL backup with error: {0}".format(job.delay_reason)
                    )
                self._log.info("Backup job completed.")
                backup_jobs_day7 += [job.job_id]

            # Run Data aging 3
            self._log.info("Running Data aging...")
            da_job = self.commcell.run_data_aging('Primary', self.storage_policy_name)
            self._log.info("data aging job: {0}".format(da_job.job_id))
            if not da_job.wait_for_completion():
                raise Exception(
                    "Failed to run data aging with error: {0}".format(da_job.delay_reason)
                )
            self._log.info("Data aging job completed.")

            # Validation Weekly full
            retained_jobs = [[backup_jobs_day2[2], time2],
                             [backup_jobs_day3[2], time3],
                             [backup_jobs_day7[2], time4]]
            if time2 > last_friday_timestamp:
                self._log.info("""VALIDATION: Weekly full job to retain: {0}
                """.format(backup_jobs_day7[2]))
                retcode = mmhelper.validate_job_prune(backup_jobs_day7[2], self.copy.copy_id)
                # retained_jobs.remove([backup_jobs_day7[2], time4])
                retained_jobs.remove([backup_jobs_day2[2], time2])
                retained_jobs.remove([backup_jobs_day3[2], time3])
                if not retcode:
                    self._log.info("Validation success")
                else:
                    raise Exception(
                        "Backup job {0} is not expected to age".format(backup_jobs_day7[2])
                    )

                self._log.info("VALIDATION: Weekly full retention flag set in JMDataStats table")
                retcode = mmhelper.validate_job_retentionflag(backup_jobs_day7[2], "EXTENDED_WEEK")
                if retcode:
                    self._log.info("Validation success")
                else:
                    raise Exception(
                        """Backup job {0} is not set with Weekly full retention flag
                        """.format(backup_jobs_day7[2])
                    )

            elif time3 > last_friday_timestamp:
                self._log.info("""VALIDATION: Weekly full job to retain: {0}
                                """.format(backup_jobs_day2[2]))
                retcode = mmhelper.validate_job_prune(backup_jobs_day2[2], self.copy.copy_id)
                # retained_jobs.remove([backup_jobs_day2[2], time2])
                retained_jobs.remove([backup_jobs_day3[2], time3])
                if not retcode:
                    self._log.info("Validation success")
                else:
                    raise Exception(
                        "Backup job {0} is not expected to age".format(backup_jobs_day2[2])
                    )

                self._log.info("VALIDATION: Weekly full retention flag set in JMDataStats table")
                retcode = mmhelper.validate_job_retentionflag(backup_jobs_day2[2], "EXTENDED_WEEK")
                if retcode:
                    self._log.info("Validation success")
                else:
                    raise Exception(
                        """Backup job {0} is not set with Weekly full retention flag
                        """.format(backup_jobs_day2[2])
                    )

            elif time4 > last_friday_timestamp:
                self._log.info("""VALIDATION: Weekly full job to retain: {0}
                                """.format(backup_jobs_day3[2]))
                retcode = mmhelper.validate_job_prune(backup_jobs_day3[2], self.copy.copy_id)
                # retained_jobs.remove([backup_jobs_day3[2], time3])
                retained_jobs.remove([backup_jobs_day2[2], time2])
                if not retcode:
                    self._log.info("Validation success")
                else:
                    raise Exception(
                        "Backup job {0} is not expected to age".format(backup_jobs_day3[2])
                    )

                self._log.info("VALIDATION: Weekly full retention flag set in JMDataStats table")
                retcode = mmhelper.validate_job_retentionflag(backup_jobs_day3[2], "EXTENDED_WEEK")
                if retcode:
                    self._log.info("Validation success")
                else:
                    raise Exception(
                        """Backup job {0} is not set with Weekly full retention flag
                        """.format(backup_jobs_day3[2])
                    )

            else:
                self._log.info("""VALIDATION: Weekly full job to retain: {0}
                                """.format(backup_jobs_day7[2]))
                retcode = mmhelper.validate_job_prune(backup_jobs_day7[2], self.copy.copy_id)
                # retained_jobs.remove([backup_jobs_day7[2], time4])
                retained_jobs.remove([backup_jobs_day2[2], time2])
                retained_jobs.remove([backup_jobs_day3[2], time3])
                if not retcode:
                    self._log.info("Validation success")
                else:
                    raise Exception(
                        "Backup job {0} is not expected to age".format(backup_jobs_day7[2])
                    )

                self._log.info("VALIDATION: Weekly full retention flag set in JMDataStats table")
                retcode = mmhelper.validate_job_retentionflag(backup_jobs_day7[2], "EXTENDED_WEEK")
                if retcode:
                    self._log.info("Validation success")
                else:
                    raise Exception(
                        """Backup job {0} is not set with Weekly full retention flag
                        """.format(backup_jobs_day7[2])
                    )

            # Run the script to move backup to 23 days
            for job in retained_jobs:
                if job[0] != backup_jobs_day7[2]:
                    mmhelper.move_job_start_time(job[0], 23)
            for i in range(len(retained_jobs)):
                retained_jobs[i][1] -= (23 * 24 * 60 * 60)
            self._log.info(retained_jobs)
            for job in backup_jobs_day7:
                mmhelper.move_job_start_time(job, 23)

            # Day 30 ---------------------
            self._log.info("--------- Day 30 -----------")
            # Run 3 FULL backup - Day 30
            self._log.info("Run 3 Full Backup jobs - Day 30")
            backup_jobs_day30 = []
            for i in range(0, 3):
                self._log.info("Running full backup J{0}...".format(i + 13))
                job = self._subclient.backup("FULL")
                self._log.info("Backup job{0}: {1}".format(i + 13, job.job_id))
                if not job.wait_for_completion():
                    raise Exception(
                        "Failed to run FULL backup with error: {0}".format(job.delay_reason)
                    )
                self._log.info("Backup job completed.")
                backup_jobs_day30 += [job.job_id]

            # Run Data aging 4
            self._log.info("Running Data aging...")
            da_job = self.commcell.run_data_aging('Primary', self.storage_policy_name)
            self._log.info("data aging job: {0}".format(da_job.job_id))
            if not da_job.wait_for_completion():
                raise Exception(
                    "Failed to run data aging with error: {0}".format(da_job.delay_reason)
                )
            self._log.info("Data aging job completed.")

            # Validation Monthly full
            length = len(retained_jobs)
            if retained_jobs[0][1] > month_day_timestamp:
                self._log.info("""VALIDATION: Monthly full job to retain: {0}
                                """.format(backup_jobs_day30[2]))
                retcode = mmhelper.validate_job_prune(backup_jobs_day30[2], self.copy.copy_id)
                if not retcode:
                    self._log.info("Validation success")
                else:
                    raise Exception(
                        "Backup job {0} is not expected to age".format(backup_jobs_day30[2])
                    )

                self._log.info("VALIDATION: Monthly full retention flag set in JMDataStats table")
                retcode = mmhelper.validate_job_retentionflag(backup_jobs_day30[2],
                                                              "EXTENDED_MONTH")
                if retcode:
                    self._log.info("Validation success")
                else:
                    raise Exception(
                        """Backup job {0} is not set with Monthly full retention flag
                        """.format(backup_jobs_day30[2])
                    )

            elif length == 2:
                self._log.info(length)
                if retained_jobs[1][1] > month_day_timestamp:
                    self._log.info("""VALIDATION: Monthly full job to retain: {0}
                                    """.format(retained_jobs[0][0]))
                    retcode = mmhelper.validate_job_prune(retained_jobs[0][0], self.copy.copy_id)
                    if not retcode:
                        self._log.info("Validation success")
                    else:
                        raise Exception(
                            "Backup job {0} is not expected to age".format(retained_jobs[0][0])
                        )

                    self._log.info("""VALIDATION: Monthly full retention flag set in JMDataStats
                                    table""")
                    retcode = mmhelper.validate_job_retentionflag(retained_jobs[0][0],
                                                                  "EXTENDED_MONTH")
                    if retcode:
                        self._log.info("Validation success")
                    else:
                        raise Exception(
                            """Backup job {0} is not set with Monthly full retention flag
                            """.format(retained_jobs[0][0])
                        )

                else:
                    self._log.info("""VALIDATION: Monthly full job to retain: {0}
                                    """.format(retained_jobs[1][0]))
                    retcode = mmhelper.validate_job_prune(retained_jobs[1][0], self.copy.copy_id)
                    if not retcode:
                        self._log.info("Validation success")
                    else:
                        raise Exception(
                            "Backup job {0} is not expected to age".format(retained_jobs[1][0])
                        )

                    self._log.info("""VALIDATION: Monthly full retention flag set in
                                    JMDataStats table""")
                    retcode = mmhelper.validate_job_retentionflag(retained_jobs[1][0],
                                                                  "EXTENDED_MONTH")
                    if retcode:
                        self._log.info("Validation success")
                    else:
                        raise Exception(
                            """Backup job {0} is not set with Monthly full retention flag
                            """.format(retained_jobs[1][0])
                        )
            else:
                self._log.info("""VALIDATION: Monthly full job to retain: {0}
                                """.format(retained_jobs[0][0]))
                retcode = mmhelper.validate_job_prune(retained_jobs[0][0], self.copy.copy_id)
                if not retcode:
                    self._log.info("Validation success")
                else:
                    raise Exception(
                        "Backup job {0} is not expected to age".format(retained_jobs[0][0])
                    )

                self._log.info("VALIDATION: Monthly full retention flag set in JMDataStats table")
                retcode = mmhelper.validate_job_retentionflag(retained_jobs[0][0],
                                                              "EXTENDED_MONTH")
                if retcode:
                    self._log.info("Validation success")
                else:
                    raise Exception(
                        """Backup job {0} is not set with Monthly full retention flag
                        """.format(retained_jobs[0][0])
                    )

            # cleanup
            try:
                self._log.info("********* cleaning up ***********")
                mmhelper.cleanup()
            except Exception as e:
                self._log.info("something went wrong while cleanup.")
                pass

        except Exception as exp:
            self._log.error('Failed to execute test case with error: ' + str(exp))
            self.result_string = str(exp)
            self.status = constants.FAILED

    def tear_down(self):
        """Tear down function of this test case"""
        pass
