# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright ©2016 Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    setup()         --  setup function of this test case

    run()           --  run function of this test case

    tear_down()     --  tear down function of this test case
"""

from AutomationUtils import logger, constants
from AutomationUtils.cvtestcase import CVTestCase
from MediaAgents.MAUtils.mahelper import MMHelper


class TestCase(CVTestCase):
    """Class for executing this test case"""

    def __init__(self):
        """Initializes test case class object
        """
        super(TestCase, self).__init__()
        self.name = "Spool copy basic case"
        self.tcinputs = {
            "MediaAgentName": None,
            "MountPath": None,
            "ContentPath": None,
            "SqlSaPassword": None
        }

    def setup(self):
        """Setup function of this test case"""
        self._log = logger.get_log()

    def run(self):
        """Run function of this test case"""
        try:
            self._log.info("Started executing {0} testcase".format(self.id))

            self._log.info(self.name)
            self.library_name = str(self.id) + "_lib"
            self.storage_policy_name = str(self.id) + "_SP"
            self.backupset_name = str(self.id) + "_BS"
            self.subclient_name = str(self.id) + "_SC"
            self.sec_copy_name = str(self.id) + "_SEC"

            # initialize MMHelper class
            mmhelper = MMHelper(self)

            try:
                mmhelper.cleanup()
            except Exception as e:
                self._log.info("error while cleanup - ignoring")
                pass
            (self.library,
             self.storage_policy,
             self._backupset,
             self._subclient) = mmhelper.setup_environment()
            self.sec_copy = mmhelper.configure_secondary_copy(self.sec_copy_name)

            # update retention to 1 day, 0 cycle
            self._log.info("setting secondary copy retention: 1 day, 0 cycle")
            self.sec_copy.copy_retention = (1, 0, 1)

            # update primary to spool copy
            self._log.info("setting primary copy retention to 0 day, 0 cycle (Spool copy)")
            self.copy = self.storage_policy.get_copy('Primary')
            self.copy.copy_retention = (0, 0, 0)

            # disable managed disk space
            self.copy.copy_retention_managed_disk_space = False

            # note copy ids
            self.primary_copy_id = self.copy.get_copy_id()
            self.secondary_copy_id = self.sec_copy.get_copy_id()

            # Run FULL backup
            self._log.info("Running full backup...")
            job = self._subclient.backup("FULL")
            self._log.info("Backup job: " + str(job.job_id))
            if not job.wait_for_completion():
                raise Exception(
                    "Failed to run FULL backup with error: {0}".format(job.delay_reason)
                )
            self._log.info("Backup job completed.")
            backup_job = job.job_id

            # Run Incr backup
            self._log.info("Running incr backup...")
            incrjob = self._subclient.backup("incremental")
            self._log.info("Incremental job: " + str(incrjob.job_id))
            if not incrjob.wait_for_completion():
                raise Exception(
                    "Failed to run Incr backup with error: {0}".format(incrjob.delay_reason)
                )
            self._log.info("Incremental Backup job completed.")
            incr_job = incrjob.job_id

            self._log.info("VALIDATION: backup job not yet aged")
            # validate backup
            self._log.info("Validating full backup...")
            retcode = mmhelper.validate_job_prune(backup_job, self.copy.copy_id)
            if not retcode:
                self._log.info("Validation success")
            else:
                raise Exception(
                    "Backup job {0} is not expected to age".format(backup_job)
                )

            # Run aux copy
            self._log.info("Running Auxcopy job...")
            aux_job = self.storage_policy.run_aux_copy(self.sec_copy_name,
                                                       self.tcinputs["MediaAgentName"],
                                                       False, 0)
            self._log.info("Auxcopy job: " + str(aux_job.job_id))
            if not aux_job.wait_for_completion():
                raise Exception(
                    "Failed to run Auxcopy with error: {0}".format(aux_job.delay_reason)
                )

            # run data aging
            da_job = self.commcell.run_data_aging('Primary', self.storage_policy_name)
            self._log.info("data aging job: " + str(da_job.job_id))
            if not da_job.wait_for_completion():
                raise Exception(
                    "Failed to run data aging with error: {0}".format(da_job.delay_reason)
                )
            self._log.info("Data aging job completed.")

            self._log.info("VALIDATION: backup, incr job should be aged")
            # validate backup
            self._log.info("Validating full backup...")
            retcode = mmhelper.validate_job_prune(backup_job, self.copy.copy_id)
            if retcode:
                self._log.info("Validation success")
            else:
                raise Exception(
                    "Backup job {0} did not age".format(backup_job)
                )
            # validate incr
            self._log.info("Validating incr backup...")
            retcode = mmhelper.validate_job_prune(incr_job, self.copy.copy_id)
            if retcode:
                self._log.info("Validation success")
            else:
                raise Exception(
                    "Incr job {0} did not age".format(backup_job)
                )

            # cleanup
            try:
                self._log.info("********* cleaning up ***********")
                mmhelper.cleanup()
            except Exception as e:
                self._log.info("something went wrong while cleanup.")
                pass

        except Exception as exp:
            self._log.error('Failed to execute test case with error: ' + str(exp))
            self.result_string = str(exp)
            self.status = constants.FAILED

    def tear_down(self):
        """Tear down function of this test case"""
        pass
