# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

This testcase verifies that automatic synthetic full schedule when estimated free space is greater
than 50%

TestCase:
    __init__()                  --  Initializes the TestCase class

    setup()                     --  All testcase objects are initializes in this method

    run()                       --  Contains the core testcase logic and it is the one executed

    tear_down()                 --  Cleans the data created for Indexing validation

"""

import traceback
import time

from AutomationUtils import constants
from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils.machine import Machine

from Indexing.testcase import IndexingTestcase
from Indexing.helpers import IndexingHelpers

from Server.JobManager.jobmanager_helper import JobManager


class TestCase(CVTestCase):
    """This testcase verifies that automatic synthetic full schedule when estimated free space is
    greater than 50%"""

    def __init__(self):
        """Initializes test case class object"""

        super(TestCase, self).__init__()
        self.name = 'Indexing - Automatic SFULL based on estimated free space'
        self.show_to_user = False

        self.tcinputs = {
            'StoragePolicyName': None
        }

        self.backupset = None
        self.subclient = None
        self.storage_policy = None

        self.cl_machine = None
        self.cl_delim = None
        self.idx_tc = None
        self.idx_help = None

        self.seconds_to_move = 86400 * 8

    def setup(self):
        """All testcase objects are initializes in this method"""

        try:

            self.backupset_name = self.tcinputs.get('Backupset', 'auto_sfull_space_reclaim')
            self.subclient_name = self.tcinputs.get('Subclient', self.id)
            self.storagepolicy_name = self.tcinputs.get('StoragePolicyName')

            self.cl_machine = Machine(self.client, self.commcell)
            self.cl_delim = self.cl_machine.os_sep

            self.idx_tc = IndexingTestcase(self)
            self.idx_help = IndexingHelpers(self.commcell)

            self.backupset = self.idx_tc.create_backupset(self.backupset_name)

            self.subclient = self.idx_tc.create_subclient(
                name=self.subclient_name,
                backupset_obj=self.backupset,
                storage_policy=self.storagepolicy_name
            )

            self.log.info('********** Turning off windows time service **********')
            self.cl_machine.toggle_time_service(stop=True)

            self.indexing_level = self.idx_help.get_agent_indexing_level(self.agent)

        except Exception as exp:
            self.log.error(str(traceback.format_exc()))
            raise Exception(exp)

    def run(self):
        """Contains the core testcase logic and it is the one executed"""

        try:
            self.log.info("Started executing {0} testcase".format(self.id))
            sc_content = self.subclient.content

            self.idx_tc.new_testdata(sc_content)
            self.idx_tc.create_only_files(
                sc_content, base_dir='to_delete', count=5, size=(102400000, 204800000))
            self.idx_tc.run_backup(self.subclient, 'full')

            self.idx_tc.edit_testdata(sc_content)
            self.idx_tc.run_backup(self.subclient, 'incremental')

            self.log.info('********** Creating space reclaim schedule ********')
            self.idx_tc.run_backup(self.subclient, 'synthetic_full', schedule_pattern={
                'schedule_name': 'space_reclaim-schedule',
                'freq_type': 'automatic',
                'run_synthetic_full': 'space_reclaim'
            })

            self.idx_tc.run_backup(self.subclient, 'synthetic_full')

            self.idx_tc.edit_testdata(sc_content)
            self.idx_tc.run_backup(self.subclient, 'incremental')

            self.log.info('********** Deleting large files directory ********')
            for path in sc_content:
                large_file_dir = self.cl_delim.join([path, 'to_delete'])
                self.cl_machine.remove_directory(large_file_dir)

            self.idx_tc.edit_testdata(sc_content)
            last_inc_job = self.idx_tc.run_backup(self.subclient, 'incremental')

            if self.indexing_level == 'backupset':
                quota_size = self.idx_help.get_quota_size(backupset_obj=self.backupset)
            else:
                quota_size = self.idx_help.get_quota_size(subclient_obj=self.subclient)

            self.log.info('********** Quota size [{0}] **********'.format(quota_size))

            application_size = self.idx_tc.get_application_size(
                subclient_obj=self.subclient, cycle_num=None)
            self.log.info('********** Application size [{0}] **********'.format(application_size))

            savings = ((application_size-quota_size)/application_size)*100
            self.log.info('********** Savings seen is [{0}%] **********'.format(savings))
            self.log.info('Expecting synthetic full job to run automatically after 7 days')

            time_before = self.cl_machine.current_time()
            self.log.info('Time before changing [{0}]'.format(time_before))

            self.log.info('********** Changing system time **********')
            self.cl_machine.change_system_time(self.seconds_to_move)

            self.log.info('Waiting for 1 minute before verifying change in system time')
            time.sleep(60)

            try:
                time_after = self.cl_machine.current_time()
                self.log.info('Time after changing [{0}]'.format(time_after))
            except Exception:
                pass

            total_attempts = 10
            attempt = 1
            job_started = None

            self.log.info('********** Checking if job has started **********')
            while attempt <= total_attempts:
                self.log.info('Waiting for 5 minutes before check. Attempt [{0}/{1}]'.format(
                    attempt, total_attempts
                ))
                time.sleep(300)

                last_job = self.idx_tc.get_last_job(self.subclient)

                if last_job.job_id == last_inc_job.job_id:
                    attempt += 1
                    continue

                job_started = last_job
                break

            if job_started:
                self.log.info('Automatic synthetic full job started job id [{0}]'.format(
                    job_started.job_id))
            else:
                raise Exception('Synthetic full job is not started automatically '
                                'after waiting for 50 minutes')

            self.log.info('Waiting for synthetic full job to complete')
            sfull_jm = JobManager(job_started)

            sfull_jm.wait_for_state('completed')

            self.log.info('Verifying job find results')
            self.backupset.idx.record_job(job_started)
            self.idx_tc.verify_job_find_results(job_started, self.backupset.idx)

            self.log.info('********** Synthetic full job completed successfully **********')

        except Exception as exp:
            self.log.error('Test case failed with error: ' + str(exp))
            self.result_string = str(exp)
            self.status = constants.FAILED
            self.log.error(str(traceback.format_exc()))

    def tear_down(self):
        """Cleans the data created for Indexing validation and reset time"""

        self.backupset.idx.cleanup()

        self.log.info('********** Turning on windows time service **********')
        self.cl_machine.toggle_time_service(stop=False)

        self.log.info('********** Resetting time on the setup **********')
        self.cl_machine.change_system_time(-self.seconds_to_move)
