# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    setup()         --  setup function of this test case

    run()           --  run function of this test case

    tear_down()     --  tear down function of this test case
"""
import time
from AutomationUtils import constants
from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils.machine import Machine
from AutomationUtils.idautils import CommonUtils
from MediaAgents.MAUtils.mahelper import MMHelper
from MediaAgents.MAUtils.mahelper import DedupeHelper


class TestCase(CVTestCase):
    """Class for executing this test case"""

    def __init__(self):
        """Initializes test case class object
        """
        super(TestCase, self).__init__()
        self.name = "Corrupt the DDBbackup store after backup phase"
        self.show_to_user = False
        self.tcinputs = {
            "MediaAgentName": None,
            "MountPath": None,
            "DedupeStorePath": None,
        }
        self.library_name = None
        self.storage_policy_name = None
        self.backupset_name = None
        self.subclient_name = None
        self.content_path1 = None
        self.content_path2 = None
        self.mount_path = None
        self.ddb_path = None
        self.mmhelper = None
        self.cs_machine = None
        self.common_util = None
        self.dedupehelper = None
        self.client_machine = None
        self.library = None
        self.storage_policy = None
        self.backupset = None
        self.subclient = None
        self.copy = None
        self.store = None
        self.ddb_sp = None
        self.ddb_copy = None
        self.ddb_subclient = None
        self.ddb_backupset = None
        self.subclient2 = None
        self.job1 = None
        self.job2 = None
        self.job3 = None

    def setup(self):
        """Setup function of this test case"""
        self.library_name = str(self.id) + "_lib"
        self.storage_policy_name = str(self.id) + "_SP"
        self.backupset_name = str(self.id) + "_BS"
        self.subclient_name = str(self.id) + "_SC"
        self.content_path1 = self.tcinputs['DedupeStorePath'] + "_data1"
        self.content_path2 = self.tcinputs['DedupeStorePath'] + "_data2"
        self.mount_path = self.tcinputs['MountPath']
        self.ddb_path = self.tcinputs['DedupeStorePath']
        self.mmhelper = MMHelper(self)
        self.cs_machine = Machine(self.commcell.commserv_client, self.commcell)
        self.common_util = CommonUtils(self)
        self.dedupehelper = DedupeHelper(self)
        self.client_machine = Machine(self.client.client_name, self.commcell)

    def run(self):
        """Run function of this test case"""
        try:
            # Run FULL backup
            self._log.info(self.name)
            # create content
            self.mmhelper.create_uncompressable_data(self.client.client_name,
                                                     self.content_path1,
                                                     0.5)
            self.mmhelper.create_uncompressable_data(self.client.client_name,
                                                     self.content_path2,
                                                     0.5)

            # create library, dedupe sp, backupset and subclient
            self.library = self.mmhelper.configure_disk_library(self.library_name,
                                                                self.tcinputs["MediaAgentName"],
                                                                self.mount_path)
            self.storage_policy = self.dedupehelper.configure_dedupe_storage_policy(self.storage_policy_name,
                                                                                    self.library_name,
                                                                                    self.tcinputs["MediaAgentName"],
                                                                                    self.ddb_path)
            self.backupset = self.mmhelper.configure_backupset(self.backupset_name, self.agent)
            self.subclient = self.mmhelper.configure_subclient(self.backupset_name,
                                                               self.subclient_name,
                                                               self.storage_policy_name,
                                                               self.content_path1,
                                                               self.agent)

            self.copy = self.storage_policy.get_copy('Primary')
            self.copy.copy_retention = (1, 1, 1)
            self.copy.copy_retention_managed_disk_space = False

            # another SP
            self.ddb_sp = self.dedupehelper.configure_dedupe_storage_policy(
                self.storage_policy_name + "_DDB",
                self.library_name,
                self.tcinputs['MediaAgentName'],
                str(self.tcinputs['DedupeStorePath']) + "_DDB")

            # another SC
            self.subclient2 = self.mmhelper.configure_subclient(self.backupset_name,
                                                                self.subclient_name + "_DDB",
                                                                self.storage_policy_name + "_DDB",
                                                                self.content_path2,
                                                                self.agent)
            self.ddb_copy = self.storage_policy.get_copy('Primary')
            self.ddb_copy.copy_retention = (1, 1, 1)
            self.ddb_copy.copy_retention_managed_disk_space = False

            ddb_subclient_name = "DDBBackup"
            ddb_backupset_name = "defaultBackupSet"
            self.log.info("Create DDBbackup backupset object.")
            self.ddb_backupset = self._agent.backupsets.get(ddb_backupset_name)
            self.log.info("Create DDBbackup subclient object.")
            self.ddb_subclient = self.ddb_backupset.subclients.get(ddb_subclient_name)

            # reassociate ddb subclient to ddb sp
            self.ddb_subclient.storage_policy = self.ddb_sp.storage_policy_name

            sidb_id = self.dedupehelper.get_sidb_ids(self.storage_policy.storage_policy_id,
                                                     "Primary")

            # backups
            self.log.info("Running backups...")
            self.job1 = self.subclient.backup('FULL')
            self.job2 = self.subclient2.backup('FULL')
            if not self.job1.wait_for_completion():
                raise Exception(
                    "Failed to run FULL backup with error: {0}".format(self.job1.delay_reason)
                )
            self._log.info("Backup job1 completed.")
            if not self.job2.wait_for_completion():
                raise Exception(
                    "Failed to run FULL backup with error: {0}".format(self.job2.delay_reason)
                )
            self._log.info("Backup job2 completed.")
            self.job3 = self.subclient.backup('FULL')
            if not self.job3.wait_for_completion():
                raise Exception(
                    "Failed to run FULL backup with error: {0}".format(self.job3.delay_reason)
                )
            self._log.info("Backup job3 completed.")

            self.log.info("Starting DDB backup...")
            job_ddb_backup = self.ddb_subclient.backup('FULL')

            self.log.info("Polling DDB backup job status for UnQuiescing done")
            log_line = r"UnQuiescing SIDB engines"
            log_file = r"clBackup.log"
            if job_ddb_backup:
                # parse log to find string: "Requested by Job-Id "+str(backupJobId)
                count = 0
                (matched_line, matched_string) = self.dedupehelper.parse_log(
                    self.tcinputs['MediaAgentName'],
                    log_file,
                    log_line,
                    job_ddb_backup.job_id, 1)
                while not matched_string:
                    time.sleep(2)
                    count += 1
                    (matched_line, matched_string) = self.dedupehelper.parse_log(
                        self.tcinputs['MediaAgentName'],
                        log_file,
                        log_line,
                        job_ddb_backup.job_id, single_file=1)
                    if count == 300:
                        raise Exception("DDB Backup did not start in time")
                # Marking DDB for recovery
                self.log.info("marking DDB as corrupt...")
                self.storage_policy.mark_for_recovery(sidb_id[0],
                                                      sidb_id[1],
                                                      self.tcinputs["MediaAgentName"],
                                                      self.tcinputs["DedupeStorePath"])

                time.sleep(10)
                self.log.info("Running Recon job")
                self.storage_policy.run_recon("Primary",
                                              self.storage_policy.storage_policy_name,
                                              sidb_id[0])

                recon_job = self.dedupehelper.poll_ddb_reconstruction(self.storage_policy.storage_policy_name,
                                                                      'Primary')
                recon_job_id = recon_job.job_id

                self.log.info("wait for DDB backup to complete...")
                if not job_ddb_backup.wait_for_completion():
                    self._log.info("DDB backup JPR: %s", job_ddb_backup.delay_reason)
                self._log.info("DDB backup completed!")

                # VALIDATION 2
                self.log.info("Checking LastSnapJobId in idxSidbSubStore table")
                query = """SELECT LastSnapJobId
                        FROM idxSidbSubStore
                        WHERE SIDBStoreId =  {0}
                        AND SubStoreId = {1}
                        """.format(str(sidb_id[0]), str(sidb_id[1]))
                db_response = self.mmhelper.execute_select_query(query)
                updated_snap_job = db_response[0][0]
                self.log.info("Current LastSnapJobId is : %s", updated_snap_job)
                if updated_snap_job != recon_job_id:
                    self.log.info("VALIDATION : PASS : LastSnapJobId is not updated.")
                else:
                    raise Exception("VALIDATION : FAILED : LastSnapJobId is updated.")

        except Exception as exp:
            self.log.error('Failed to execute test case with error: %s', str(exp))
            self.result_string = str(exp)
            self.status = constants.FAILED

    def tear_down(self):
        self._log.info("cleaning up content and jobs")
        self.client_machine.remove_directory(self.content_path1)
        self.client_machine.remove_directory(self.content_path2)
        # delete all jobs
        try:
            self.copy.delete_job(self.job1.job_id)
            self.copy.delete_job(self.job3.job_id)
            self.ddb_copy.delete_job(self.job2.job_id)
        except Exception as exp:
            pass
