# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    setup()         --  setup function of this test case

    run()           --  run function of this test case

    tear_down()     --  tear down function of this test case
"""
import time
from datetime import datetime
from cvpysdk.policies.storage_policies import StoragePolicy
from AutomationUtils import constants
from AutomationUtils.cvtestcase import CVTestCase
from MediaAgents.MAUtils.mahelper import MMHelper
from MediaAgents.MAUtils.mahelper import DedupeHelper
from AutomationUtils.idautils import CommonUtils


class TestCase(CVTestCase):
    """Class for executing this test case"""

    def __init__(self):
        """Initializes test case class object
        """
        super(TestCase, self).__init__()
        self.name = "Retaining multiple DDB backups"
        self.show_to_user = False
        self.tcinputs = {
            "MediaAgentName": None,
            "MountPath": None,
            "DedupeStorePath": None,
            "SqlSaPassword": None
        }

    def setup(self):
        """Setup function of this test case"""
        self.common_util = CommonUtils(self)
        self.mmhelper = MMHelper(self)
        self.dedupehelper = DedupeHelper(self)
        self.library_name = str(self.id) + "_lib"

    def run(self):
        """Run function of this test case"""
        try:
            sp_name = str(self.id) + "_SP"
            subclient_name = "DDBBackup"
            backupset_name = "defaultBackupSet"

            self.log.info("Create DDBbackup backupset object.")
            self.backupset = self._agent.backupsets.get(backupset_name)
            self.log.info("Create DDBbackup subclient object.")
            self.subclient = self.backupset.subclients.get(subclient_name)

            # SP configuration
            ddb_sp = self.subclient.storage_policy
            if ddb_sp:
                self.log.info("SP associated with the DDBbackup subclient : {0}".format(ddb_sp))
                ddb_sp_obj = self.commcell.policies.storage_policies.get(ddb_sp)
            else:
                self.mmhelper.configure_disk_library()

                self.log.info("check SP: {0}".format(sp_name))
                if not self.commcell.storage_policies.has_policy(sp_name):
                    self.log.info("adding Storage policy...")
                    current_time = datetime.strftime(datetime.now(), '%Y-%m-%d_%H-%M-%S')
                    self.commcell.storage_policies.add(sp_name, self.library_name,
                                                       self.tcinputs["MediaAgentName"],
                                                       self.tcinputs["DedupeStorePath"]+current_time)
                else:
                    self.log.info("Storage policy exists!")
                self.log.info("Storage policy config done.")
                ddb_sp_obj = self.commcell.policies.storage_policies.get(sp_name)
                self.log.info("Setting storage policy for the {} subclient".format(subclient_name))
                self.subclient.storage_policy = ddb_sp_obj.storage_policy_name

            sidb_id = self.dedupehelper.get_sidb_ids(ddb_sp_obj.storage_policy_id, "Primary")

            # setting 'MMCONFIG_RETAIN_NO_OF_DDB_BACKUPS' to 3
            self.log.info("setting 'MMCONFIG_RETAIN_NO_OF_DDB_BACKUPS' to 3")
            query = """UPDATE MMCONFIGS
                    SET VALUE = 3
                    WHERE NAME = 'MMCONFIG_RETAIN_NO_OF_DDB_BACKUPS'
                    """
            self.mmhelper.execute_update_query(query)
            self.log.info("Checking if value set correctly")
            query = """SELECT value
                    FROM MMCONFIGS
                    WHERE name = 'MMCONFIG_RETAIN_NO_OF_DDB_BACKUPS'
                    """
            db_response = self.mmhelper.execute_select_query(query)
            if db_response[0][0] == '3':
                self.log.info(
                    "VALIDATION : PASSED : 'MMCONFIG_RETAIN_NO_OF_DDB_BACKUPS' set to 3")
            else:
                raise Exception(
                    "VALIDATION : FAILED : 'MMCONFIG_RETAIN_NO_OF_DDB_BACKUPS' NOT set to 3")

            # Run FULL backup
            self.log.info("Running full backup...")
            job_list = []
            for iterator in range(0, 3):
                job = self.common_util.subclient_backup(self.subclient, "Full")
                job_list.append(str(job.job_id))
                time.sleep(30)

            self.log.info("Running data ageing job..")
            self.commcell.run_data_aging()
            time.sleep(10)

            # VALIDATION 1
            self.log.info("Checking LastSnapJobId in IdxSIDBSubStoreBackupInfo table")
            query = """SELECT LastSnapJobId
                    FROM IdxSIDBSubStoreBackupInfo
                    WHERE SIDBStoreId =  {0}
                    AND SubStoreId = {1}
                    """.format(str(sidb_id[0]), str(sidb_id[1]))
            db_response = self.mmhelper.execute_select_query(query)
            if (job_list[0] in str(db_response)
                    and job_list[1] in str(db_response)) and job_list[2] not in db_response:
                self.log.info("""VALIDATION : PASSED : Only DDBbackup job {0},{1}
                              are in IdxSIDBSubStoreBackupInfo table as expected.
                              """.format(str(job_list[0]), str(job_list[1])))
            else:
                raise Exception("""VALIDATION : FAILED :
                                Correct DDBbackup job IDs not present
                                in IdxSIDBSubStoreBackupInfo table""")

            # VALIDATION 2
            self.log.info("Checking LastSnapJobId in idxSidbSubStore table")
            query = """SELECT LastSnapJobId
                    FROM idxSidbSubStore
                    WHERE SIDBStoreId =  {0}
                    AND SubStoreId = {1}
                    """.format(str(sidb_id[0]), str(sidb_id[1]))
            db_response = self.mmhelper.execute_select_query(query)
            if job_list[2] in str(db_response[0]):
                self.log.info("""VALIDATION : PASSED : DDBbackup job {0} present in
                              idxSidbSubStore table as expected.""".format(str(job_list[2])))
            else:
                raise Exception("""VALIDATION : FAILED :
                                Correct DDBbackup job ID not present in idxSidbSubStore table""")

            self.log.info("Updating LastSnapJobId as {0} in IdxSIDBSubStore".format(str(job_list[1])))
            qscript = """-sn DDBBackupJob -si {0} -si {1} -si set -si {2}""".format(str(sidb_id[0]),
                                                                                    str(sidb_id[1]),
                                                                                    str(job_list[1]))
            self.commcell._qoperation_execscript(qscript)
            self.log.info("Updated LastSnapJobId")
            # VALIDATION 3
            self.log.info("Checking correct LastSnapJobId updated in IdxSIDBSubStore ")
            db_response = self.mmhelper.execute_select_query(query)
            if job_list[1] in str(db_response[0]):
                self.log.info("""VALIDATION : PASSED : DDBbackup job {0}
                              present in idxSidbSubStore table as expected.
                              """.format(str(job_list[1])))
            else:
                raise Exception("""VALIDATION : FAILED :
                                Correct DDBbackup job ID not present in idxSidbSubStore table""")

            # mark DDB for recovery
            ddb_sp_obj.mark_for_recovery(sidb_id[0], sidb_id[1], self.tcinputs["MediaAgentName"],
                                         self.tcinputs["DedupeStorePath"])
            self.log.info("DDB marked for recovery")

            # launch reconstruction job
            ddb_sp_obj.run_recon("Primary", ddb_sp_obj.storage_policy_name, sidb_id[0])
            time.sleep(10)
            # wait for reconstruction job to complete
            recon_job_id = self.dedupehelper.get_ddb_recon_details(ddb_sp_obj.storage_policy_id)

            # VALIDATION 4
            self.log.info("""Verifying if updated DDBbackup job id [{0}]
                          was used for the Reconstruction job""".format(job_list[1]))
            string = "RestoreDDBSubStore Submitting restore request from DDB Backup Job"
            log_file_name = "AuxCopyMgr.log"
            (matched_line, matched_string) = self.dedupehelper.parse_log(self.client.client_hostname,
                                                                         log_file_name,
                                                                         string, recon_job_id)
            if matched_line:
                self.log.info("Matched log line : \n{0}".format(matched_line[0]))
                job_id_used = matched_line[0].split("[")[1].split("]")[0]
                if int(job_id_used) == int(job_list[1]):
                    self.log.info("""VALIDATION : PASSED : DDBbackup job id {0}
                                  was used for the Reconstruction job""".format(job_list[1]))
            else:
                raise Exception("VALIDATION : FAILED : Failed to find: {0} ".format(string))

        except Exception as exp:
            self.log.error('Failed to execute test case with error: {0}'.format(str(exp)))
            self.result_string = str(exp)
            self.status = constants.FAILED

    def tear_down(self):
        """Tear down function of this test case"""
        self.log.info("setting 'MMCONFIG_RETAIN_NO_OF_DDB_BACKUPS' to 1")
        query = """UPDATE MMCONFIGS
                            SET VALUE = 1
                            WHERE NAME = 'MMCONFIG_RETAIN_NO_OF_DDB_BACKUPS'
                            """
        self.mmhelper.execute_update_query(query)
        pass
