# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Test case to verify Virtualization dashboard page loading without any errors."""

from AutomationUtils.cvtestcase import CVTestCase
from Web.Common.cvbrowser import BrowserFactory
from Web.Common.cvbrowser import Browser
from Web.Common.exceptions import CVTestCaseInitFailure, CVTestStepFailure
from Web.Common.page_object import TestStep

from Web.AdminConsole.adminconsole import AdminConsole
from Web.AdminConsole.AdminConsolePages.dashboard import (VirtualizationDashboard, Dashboard)

from Reports.utils import TestCaseUtils


class TestCase(CVTestCase):
    """Test case to verify Virtualization dashboard page is loading fine, and all the expected panes are
    present"""
    test_step = TestStep()

    def __init__(self):

        super(TestCase, self).__init__()
        self.name = "Admin console Virtualization Dashboard page verification"
        self.browser = None
        self.utils = None
        self.dashboard = None
        self.report = None
        self.admin_console = None
        self.expected_panes = []

    def init_tc(self):
        """Initialize browser and redirect to page"""
        try:
            self.utils = TestCaseUtils(self)
            self.browser = BrowserFactory().create_browser_object()
            self.browser.open()
            self.admin_console = AdminConsole(self.browser, self.commcell.webconsole_hostname)
            self.admin_console.login()
            self.expected_panes = [pane.value for pane in VirtualizationDashboard
                                   if "pane" in pane.name]
            self.dashboard = Dashboard(self.admin_console)
        except Exception as _exception:
            raise CVTestCaseInitFailure(_exception) from _exception

    @test_step
    def verify_dashboard_navigation(self):
        """Navigate to virtualization dashboard without any errors"""
        self.dashboard.navigate_to_given_dashboard("Virtualization Dashboard")

    @test_step
    def verify_dashboard_page(self):
        """
        Verify virtualization dashboard page is not having any errors/notifications
        """
        if self.browser.driver.title != "Virtualization Dashboard":
            raise CVTestStepFailure("Expecting [Virtualization Dashboard]title of landing page"
                                    ", but [%s] is the currently title of the page" %
                                    self.browser.driver.title)
        notification = self.admin_console.get_notification()
        if notification:
            raise CVTestStepFailure("Dashboard page has Notification error [%s]" % notification)
        self.log.info("Dashboard page load is successful")

    @test_step
    def verify_pane_titles(self):
        """Verify all panes are present in dashboard page"""
        titles = self.dashboard.get_dash_pane_titles()
        if sorted(titles) != sorted(self.expected_panes):
            raise CVTestStepFailure("Expected [%s] panes are not present in dashboard page. "
                                    "Existing panes are [%s]" % (str(self.expected_panes),
                                                                 str(titles)))
        self.log.info("Dashboard pane's titles are verified successfully")

    @staticmethod
    def get_mapped_title(pane_name, entity_name=None):
        """Get expected titles of pane details page"""

        _mapped_items = [{"p_name": VirtualizationDashboard.pane_hypervisors.value,
                          "entity": None,
                          "title": "Hypervisors"},
                         {"p_name": VirtualizationDashboard.pane_vms.value,
                          "entity": VirtualizationDashboard.entity_protected.value,
                          "title": "Virtual machines"},
                         {"p_name": VirtualizationDashboard.pane_vms.value,
                          "entity": VirtualizationDashboard.entity_not_protected.value,
                          "title": "Unprotected VMs"},
                         {"p_name": VirtualizationDashboard.pane_vms.value,
                          "entity": VirtualizationDashboard.entity_backed_up_with_error.value,
                          "title": "Virtual machines"},
                         {"p_name": VirtualizationDashboard.pane_sla.value,
                          "entity": None,
                          "title": "SLA"},
                         {"p_name": VirtualizationDashboard.pane_jobs_in_last_24_hours.value,
                          "entity": VirtualizationDashboard.entity_running.value,
                          "title": "Active jobs"},
                         {"p_name": VirtualizationDashboard.pane_jobs_in_last_24_hours.value,
                          "entity": VirtualizationDashboard.entity_success.value,
                          "title": "Job history"},
                         {"p_name": VirtualizationDashboard.pane_jobs_in_last_24_hours.value,
                          "entity": VirtualizationDashboard.entity_failed.value,
                          "title": "Job history"},
                         {"p_name": VirtualizationDashboard.pane_jobs_in_last_24_hours.value,
                          "entity": VirtualizationDashboard.entity_events.value,
                          "title": "Events"},
                         {"p_name": VirtualizationDashboard.pane_last_week_backup_job_summary.value,
                          "entity": None,
                          "title": "Job history"},
                         {"p_name": VirtualizationDashboard.pane_largest_hypervisors.value,
                          "entity": None,
                          "title": "Hypervisors"}]
        for pane in _mapped_items:
            if pane["p_name"] == pane_name and pane["entity"] == entity_name:
                return pane["title"]
        raise CVTestStepFailure("[%s]Pane is not mapped" % pane_name)

    @test_step
    def verify_pane_details_page(self):
        """Access details page of pane and verify the page titles are correct"""
        panes_dict = \
            {
                VirtualizationDashboard.pane_hypervisors: None,
                VirtualizationDashboard.pane_vms: [VirtualizationDashboard.entity_protected,
                                                   VirtualizationDashboard.entity_not_protected,
                                                   VirtualizationDashboard.entity_backed_up_with_error],
                VirtualizationDashboard.pane_sla: None,
                VirtualizationDashboard.pane_jobs_in_last_24_hours:
                    [VirtualizationDashboard.entity_running,
                        VirtualizationDashboard.entity_success,
                        VirtualizationDashboard.entity_failed,
                        VirtualizationDashboard.entity_events],
                VirtualizationDashboard.pane_last_week_backup_job_summary: None,
                VirtualizationDashboard.pane_largest_hypervisors: None
            }
        for pane, entities in panes_dict.items():
            if entities:
                for each_value in entities:
                    self.log.info("Accessing details page of Pane:%s, Entity:%s", pane.value,
                                  each_value.value)
                    self.dashboard.access_details_page(pane_name=pane.value,
                                                       entity_name=each_value.value)
                    self.validate_title(pane.value, entity=each_value.value)
            else:
                self.log.info("Accessing details page of Pane:%s", pane.value)
                self.dashboard.access_details_page(pane_name=pane.value)
                self.validate_title(pane.value, entity=entities)

    def validate_title(self, pane, entity):
        """Access pane details page and verify page titles"""
        expected_title = self.get_mapped_title(pane, entity_name=entity)
        handles = self.browser.driver.window_handles
        size = len(handles)
        page_title = None
        if size > 1:
            parent_handle = self.browser.driver.current_window_handle
            for x in range(size):
                if handles[x] != parent_handle:
                    self.browser.driver.switch_to_window(handles[x])
                    page_title = self.dashboard.get_page_title()
                    self.browser.driver.close()
                    break
            self.browser.driver.switch_to_window(parent_handle)
            self.admin_console.wait_for_completion()
        else:
            page_title = self.dashboard.get_page_title()
            self.browser.driver.back()
            self.admin_console.wait_for_completion()

        if page_title == expected_title:
            self.log.info("Verified title for [%s] pane, [%s] entity", pane, entity)
        else:
            raise CVTestStepFailure("Title did not match for [%s] pane, [%s] entity, [%s] title is"
                                    " Expected, [%s] title is found on page" %
                                    (pane, entity, expected_title, page_title))

    def run(self):
        try:
            self.init_tc()
            self.verify_dashboard_navigation()
            self.verify_dashboard_page()
            self.verify_pane_titles()
            self.verify_pane_details_page()
        except Exception as err:
            self.utils.handle_testcase_exception(err)
        finally:
            AdminConsole.logout_silently(self.admin_console)
            Browser.close_silently(self.browser)
