# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Infrastructure Load report"""

from datetime import (
    datetime,
    timedelta
)

from cvpysdk.security.role import Roles
from cvpysdk.security.user import Users, User

from AutomationUtils.cvtestcase import CVTestCase

from Reports.Custom.sql_utils import SQLQueries
from Reports.Custom.utils import CustomReportUtils

from Web.Common.cvbrowser import (
    BrowserFactory,
    Browser
)
from Web.Common.exceptions import CVTestCaseInitFailure, CVTestStepFailure
from Web.Common.page_object import TestStep
from Web.WebConsole.Reports.Custom.viewer import (
    CustomReportViewer,
    DataTable
)
from Web.WebConsole.Reports.Metrics.report import MetricsReport
from Web.WebConsole.Reports.home import ReportsHomePage

from Web.WebConsole.webconsole import WebConsole


class TestCase(CVTestCase):
    """ TestCase class used to execute the test case from here."""
    test_step = TestStep()
    QUERY = """select cl.displayName
            from MMHost M
            inner join app_client cl
            on cl.id = M.ClientId
            where M.RolesBitMask & 8  =8 or M.RolesBitMask & 128  = 128  or M.clientid=2"""

    def __init__(self):
        super(TestCase, self).__init__()
        self.name = "Infrastructure Load report"
        self.show_to_user = False
        self.browser = None
        self.webconsole = None
        self.utils = CustomReportUtils(self)
        self.data = None
        self.automation_username = "tc53795"
        self.automation_password = "######"

    def init_tc(self):
        """ Initial configuration for the test case. """
        try:
            self.browser = BrowserFactory().create_browser_object()
            self.browser.open()
            self.create_role_and_user()
            self.webconsole = WebConsole(self.browser, self.commcell.webconsole_hostname)
            self.utils.webconsole = self.webconsole
            self.webconsole.login()
            self.webconsole.goto_reports()
        except Exception as exception:
            raise CVTestCaseInitFailure(exception)from exception

    def create_role_and_user(self):
        """Create role and user if does not exist"""
        roles = Roles(self.commcell)
        if roles.has_role("Report Management") is False:
            self.log.info("'Report Management' Role does not exist, creating it")
            roles.add("Report Management", ["Report Management"], ["Global"])

        users = Users(self.commcell)
        if users.has_user(self.automation_username) is False:
            self.log.info(f"{self.automation_username} does not exist, creating it")
            users.add(
                self.automation_username,
                self.automation_username,
                "reports@testing.com",
                None,
                self.automation_password
            )

    @test_step
    def verify_report_loading(self):
        """Verifies Loading of the Infrastructure Load report"""
        home_page = ReportsHomePage(self.webconsole)
        home_page.access_hidden_report("Infrastructure Load")
        report = MetricsReport(self.webconsole)
        if report.is_no_data_error_exists() or self.webconsole.get_all_error_notifications():
            raise CVTestStepFailure("Improper loading of Infrastructure Load Report")
        report_viewer = CustomReportViewer(self.webconsole)
        table = DataTable("Performance")
        report_viewer.associate_component(table)
        self.data = table.get_table_data()

    @test_step
    def verify_media_agents(self):
        """Verifies Media Agents"""
        media_agents = self.utils.cre_api.execute_sql(
            sql=TestCase.QUERY, database_name="CommServ", as_json=True).get("displayName")
        if set(media_agents) < set(self.data["Client"]) is False:
            raise CVTestStepFailure(f"Missing Clients are: {set(media_agents)-set(self.data['Client'])}")

        self.data["Time"] = [item[1] for item in list(zip(self.data["Client"], self.data["Time"]))
                             if item[0] in media_agents]
        self.data["Client"] = media_agents

    @test_step
    def verify_time_info(self):
        """Verifies time info"""
        ma_time = list(map(lambda time_: datetime.strptime(time_, "%b %d, %Y, %I:%M:%S %p"), self.data["Time"]))
        current_time = datetime.now()
        error = list()
        for index, time in enumerate(ma_time):
            seconds = (current_time - time).total_seconds()
            if seconds > 7200:
                error.append(f"Media Agent: {self.data['Client'][index]} was last updated"
                             f" {str(timedelta(seconds=seconds))} ago")

        if error:
            raise CVTestStepFailure("\n".join(error))

    @test_step
    def verify_security(self):
        """Verifies security by restricting permissions to user"""
        client_name = [self.data['Client'][0]]
        dict_ = {'assoc1':
                 {
                     'clientName': client_name,
                     'role': ['Report Management']
                 }
                 }
        user = User(self.commcell, self.automation_username)
        user.update_security_associations(dict_, "UPDATE")
        security = CustomReportViewer(self.webconsole).open_security()
        security.associate_security_permission([self.automation_username])
        security.update()
        self.webconsole.logout()
        self.webconsole.login(self.automation_username, self.automation_password)
        self.webconsole.goto_reports()
        self.verify_report_loading()
        SQLQueries.validate_list_equality(client_name, self.data['Client'])
        self.verify_time_info()

    def run(self):
        try:
            self.init_tc()
            self.verify_report_loading()
            self.verify_media_agents()
            self.verify_time_info()
            self.verify_security()
        except Exception as err:
            self.utils.handle_testcase_exception(err)

        finally:
            WebConsole.logout_silently(self.webconsole)
            Browser.close_silently(self.browser)
