# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""
Testcase: This test case verifies whether VMWare replication job kill produces the need sync state on the
admin console and then verifies whether the destination VM has been cleaned up on the destination hypervisor

Sample JSON: {
      "ClientName": "<source hypervisor>",
      "source_vm": "vm",
      "recovery_target": "target",
      "storage_name": "storage",
      "guest_username": "",
      "guest_password": ""
}
"""
from time import sleep

from AutomationUtils.cvtestcase import CVTestCase
from Reports.utils import TestCaseUtils
from VirtualServer.VSAUtils import VirtualServerUtils
from VirtualServer.VSAUtils.LiveSyncUtils import LiveSyncUtils
from Web.AdminConsole.DR.group_details import ReplicationDetails
from Web.AdminConsole.DR.replication import ReplicationGroup
from Web.AdminConsole.adminconsole import AdminConsole
from Web.Common.cvbrowser import BrowserFactory
from Web.Common.exceptions import CVTestStepFailure, CVTestCaseInitFailure
from Web.Common.page_object import TestStep


class TestCase(CVTestCase):
    """This class is used to automate the replication of Vmware and job kill"""
    test_step = TestStep()
    _AGENT_NAME = 'virtual server'
    _INSTANCE_NAME = 'vmware'
    _BACKUPSET_NAME = 'defaultBackupSet'

    def __init__(self):
        """Initialises the objects and TC inputs"""
        super(TestCase, self).__init__()
        self.name = "Verify that cleanup is perform on the destination and status is set to " + \
                    "“Need Sync” when full VM replication Failed/Killed."
        self.tcinputs = {
            "ClientName": None,
            "source_vm": None,
            "recovery_target": None,
            "storage_name": None,
            "guest_username": None,
            "guest_password": None
        }
        self.utils = None

        self.browser = None
        self.admin_console = None

        self.replication_group = None
        self.group_details = None

        self._source_subclient = None
        self.replication_job_id = None

    @property
    def group_name(self):
        """Returns the virtualization group name"""
        return f"Group_TC_{self.id}"

    @property
    def schedule(self):
        """Returns the name of the schedule"""
        return self.replication_group.get_schedule_name_by_replication_group(self.group_name)

    @property
    def source_subclient(self):
        """Returns the VSA Auto subclient after initialising all the testcase objects"""
        try:
            hasattr(self, 'subclient')
        except:
            self.agent = self.client.agents.get(self._AGENT_NAME)
            self.instance = self.agent.instances.get(self._INSTANCE_NAME)
            self.backupset = self.agent.backupsets.get(self._BACKUPSET_NAME)
            self.subclient = self.backupset.subclients.get(self.group_name)
        if not self._source_subclient:
            self._source_subclient = VirtualServerUtils.subclient_initialize(self)
        return self._source_subclient

    @property
    def destination_vm_name(self):
        """Gets the destination VM name from the source subclient"""
        return self.source_subclient.get_live_sync_destination_vms(self.schedule)[0]

    @property
    def backup_job_id(self):
        """Get the latest backup job obj"""
        try:
            hasattr(self, 'subclient')
        except:
            self.agent = self.client.agents.get(self._AGENT_NAME)
            self.instance = self.agent.instances.get(self._INSTANCE_NAME)
            self.backupset = self.agent.backupsets.get(self._BACKUPSET_NAME)
            self.subclient = self.backupset.subclients.get(self.group_name)
        return self.subclient.find_latest_job().job_id

    def login(self):
        """Logs in to admin console"""
        self.browser = BrowserFactory().create_browser_object()
        self.browser.open()
        self.admin_console = AdminConsole(self.browser, machine=self.inputJSONnode['commcell']['webconsoleHostname'])
        self.admin_console.login(self.inputJSONnode['commcell']['commcellUsername'],
                                 self.inputJSONnode['commcell']['commcellPassword'])

        self.replication_group = ReplicationGroup(self.admin_console)
        self.group_details = ReplicationDetails(self.admin_console)

    def logout(self):
        """Logs out of the admin console and closes the browser"""
        self.admin_console.logout_silently(self.admin_console)
        self.browser.close_silently(self.browser)

    def setup(self):
        """Sets up the Testcase"""
        try:
            self.utils = TestCaseUtils(self)

            self.login()
        except Exception as exp:
            raise CVTestCaseInitFailure(f'Failed to initialise testcase {str(exp)}')

    def wait_for_sync(self, num_minutes: int):
        """Waits for sync to happen for number of minutes"""
        for _ in range(num_minutes // 2):
            sleep(120)
            self.admin_console.refresh_page()
        if num_minutes % 2:
            sleep(60)

    @test_step
    def delete_virtualization_group(self, wait_for_sync: bool = False):
        """Deletes the virtualization group if it exists already"""
        self.admin_console.navigator.navigate_to_replication_groups()
        if self.replication_group.has_group(self.group_name):
            self.log.info('The replication group %s exists, deleting it now', self.group_name)
            self.replication_group.delete_group(self.group_name)
            if wait_for_sync:
                self.wait_for_sync(5)
            else:
                self.admin_console.refresh_page()
            if self.replication_group.has_group(self.group_name):
                raise CVTestStepFailure(f"Deletion of replication group {self.group_name} failed")
        self.log.info('The replication group %s does not exist', self.group_name)

    @test_step
    def configure_replication_group(self):
        """Configures the replication group with default options"""
        vmware_configure = self.replication_group.configure_vmware()
        vmware_configure.content.set_name(self.group_name)
        vmware_configure.content.select_vm_from_browse_tree(self.tcinputs['ClientName'],
                                                            {"VMs and templates": [self.tcinputs['source_vm']]})
        sleep(5)
        vmware_configure.next()

        vmware_configure.target.select_recovery_target(self.tcinputs['recovery_target'])
        vmware_configure.target.unconditionally_overwrite_vm(True)
        vmware_configure.next()

        vmware_configure.storage_cache.select_storage(self.tcinputs['storage_name'])

        vmware_configure.next()
        vmware_configure.next()

        sleep(5)
        vmware_configure.finish()

    @test_step
    def verify_backup_job_completion(self):
        """Waits for backup job to complete and then verify its details"""
        self.log.info('Waiting to 3 minutes to let the jobs trigger')
        sleep(180)
        self.log.info('Waiting for backup job %s to finish', self.backup_job_id)
        job_obj = self.commcell.job_controller.get(self.backup_job_id)
        job_obj.wait_for_completion()
        self.utils.assert_comparison(job_obj.status, 'Completed')

    @test_step
    def verify_replication_job_kill(self):
        """Waits for replication job to be killed in one minute"""
        self.log.info("Waiting 2 minutes to let live sync update")
        sleep(120)
        live_sync_utils = LiveSyncUtils(self.source_subclient, self.schedule)
        job_obj = live_sync_utils.get_recent_replication_job(self.backup_job_id)
        self.replication_job_id = job_obj.job_id
        self.log.info("Waiting for replication job id %s to be killed", job_obj.job_id)
        job_obj.kill(wait_for_job_to_kill=True)

    @test_step
    def verify_admin_console_state(self):
        """Verifies that the admin console shows the status need sync and the failed backup job ID is shown"""
        self.admin_console.navigator.navigate_to_replication_groups()
        self.replication_group.access_group(self.group_name)
        vm_details = self.group_details.overview.get_vm_details(self.tcinputs['source_vm'])
        self.utils.assert_comparison(vm_details['Status'], 'Sync Pending')
        self.utils.assert_comparison(vm_details['Last replication job'], self.replication_job_id)
        if vm_details['Last successful replication job'] == self.replication_job_id:
            raise CVTestStepFailure(f"Replication job [{self.replication_job_id}] succeeded even after killing it")

    @test_step
    def verify_cleanup(self):
        """Verifies that the VM doesn't exist and the schedule is deleted"""
        destination_vm_name = self.source_subclient.get_live_sync_destination_vms(self.schedule)[0]
        destination_subclient = self.source_subclient.get_live_sync_destination_subclient(self.schedule)
        if destination_subclient.hvobj.check_vms_exist([destination_vm_name]):
            raise CVTestStepFailure(f"The VM name[{destination_vm_name}] exists")

    def run(self):
        """Runs the testcase in order"""
        try:
            self.delete_virtualization_group(wait_for_sync=True)
            self.configure_replication_group()

            self.logout()
            self.verify_backup_job_completion()
            self.verify_replication_job_kill()
            self.verify_cleanup()

            self.login()
            self.verify_admin_console_state()
            self.delete_virtualization_group(wait_for_sync=False)

        except Exception as exp:
            self.utils.handle_testcase_exception(exp)

    def tear_down(self):
        """Cleans up the testcase"""
        self.logout()
