# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------


from AutomationUtils.cvtestcase import CVTestCase
from Reports.Custom.sql_utils import (
    SQLQueries, ValueProcessors
)
from Reports.Custom.utils import CustomReportUtils
from Web.Common.cvbrowser import (
    Browser, BrowserFactory
)
from Web.Common.exceptions import (
    CVTestCaseInitFailure
)
from Web.Common.page_object import TestStep
from Web.WebConsole.Reports.Custom import builder
from Web.WebConsole.Reports.Custom import viewer
from Web.WebConsole.Reports.Custom.builder import Datasets
from Web.WebConsole.Reports.navigator import Navigator
from Web.WebConsole.webconsole import WebConsole


class TestCase(CVTestCase):

    test_step = TestStep()

    def __init__(self):
        super(TestCase, self).__init__()
        self.name = "Custom Report: (DataSet) - MySQL DataSource"
        self.applicable_os = self.os_list.WINDOWS
        self.product = self.products_list.SOFTWARESTORE
        self.feature = self.features_list.WEBCONSOLE
        self.show_to_user = True
        self.utils = CustomReportUtils(self)
        self.webconsole: WebConsole = None
        self.browser: Browser = None
        self.rpt_builder = None
        self.table = None
        self.dataset = None

    def init_tc(self):
        try:
            self.utils.cre_api.delete_custom_report_by_name(
                self.name, suppress=True
            )
            self.browser = BrowserFactory().create_browser_object()
            self.browser.open()
            self.webconsole = WebConsole(
                self.browser, self.commcell.webconsole_hostname
            )
            self.webconsole.login()
            Navigator(self.webconsole).goto_report_builder()
            self.rpt_builder = builder.ReportBuilder(self.webconsole)
            self.rpt_builder.set_report_name(self.name)
        except Exception as e:
            raise CVTestCaseInitFailure(e) from e

    @test_step
    def create_dataset(self):
        """Create and Preview dataSet with MySQL DataSource"""
        self.dataset = Datasets.DatabaseDataset()
        self.rpt_builder.add_dataset(self.dataset)
        self.dataset.set_dataset_name("AutomationDataSet")
        self.dataset.set_all_mysql_datasource()
        self.dataset.set_database()
        self.dataset.set_sql_query(SQLQueries.mysql_q())
        received_data = self.dataset.get_preview_data()
        expected_data = SQLQueries.mysql_r()
        SQLQueries.validate_equality(
            received=received_data,
            expected=expected_data,
            value_processor=ValueProcessors.lower_and_unique,
            err_msg="Unexpected Table data when viewed dataset preview"
        )
        self.dataset.save()

    @test_step
    def add_datasource_to_table(self):
        """Add the datasource to any table"""
        data_table = builder.DataTable("AutomationTable")
        self.rpt_builder.add_component(data_table, self.dataset)
        data_table.add_column_from_dataset()
        received_data = data_table.get_table_data()
        expected_data = SQLQueries.mysql_r()
        SQLQueries.validate_equality(
            received=received_data,
            expected=expected_data,
            value_processor=ValueProcessors.lower_and_unique,
            err_msg="Unexpected Table data when viewed from Builder"
        )
        self.rpt_builder.save_and_deploy()
        self.rpt_builder.open_report()
        self.table = viewer.DataTable("AutomationTable")
        rpt_viewer = viewer.CustomReportViewer(self.webconsole)
        rpt_viewer.associate_component(self.table)

    @test_step
    def validate_report_viewer(self):
        """Table data should also be shown on viewer"""
        received_data = self.table.get_table_data()
        expected_data = SQLQueries.mysql_r()
        SQLQueries.validate_equality(
            received=received_data,
            expected=expected_data,
            value_processor=ValueProcessors.lower_and_unique,
            err_msg="Unexpected Table data when viewed from Viewer"
        )

    def run(self):
        try:
            self.init_tc()
            self.create_dataset()
            self.add_datasource_to_table()
            self.validate_report_viewer()
        except Exception as err:
            self.utils.handle_testcase_exception(err)
        finally:
            WebConsole.logout_silently(self.webconsole)
            Browser.close_silently(self.browser)
