# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Install App"""

from Web.Common.page_object import TestStep
from Web.Common.cvbrowser import (
    BrowserFactory, Browser
)
from Web.Common.exceptions import (
    CVTestCaseInitFailure, CVTestStepFailure
)
from AutomationUtils.cvtestcase import CVTestCase
from Web.WebConsole.webconsole import WebConsole
from Web.WebConsole.Store import storeapp
from Web.WebConsole.Reports.Custom import viewer
from Web.WebConsole.Apps.apps import (
    AppsPage, App
)
from Web.API.webconsole import Apps as AppsAPI
from Reports.App.util import AppUtils

_APP_CONFIG = AppUtils.get_config()


class TestCase(CVTestCase):
    test_step = TestStep()

    def __init__(self):
        super().__init__()
        self.name = "Install, Export, Delete, Import operation on App"
        self.browser: Browser = None
        self.webconsole = None
        self.export = None
        self.viewer: viewer.CustomReportViewer = None
        self.store_app: storeapp.StoreApp = None
        self.utils = AppUtils(self)
        self.report_builder = None
        self.apps_page: AppsPage = None
        self.apps_api: AppsAPI = None

    def init_tc(self):
        try:
            self.apps_api = AppsAPI(self.commcell.webconsole_hostname)
            self.browser = BrowserFactory().create_browser_object()
            self.utils.reset_temp_dir()
            self.browser.set_downloads_dir(self.utils.get_temp_dir())
            self.browser.open()
            self.webconsole = WebConsole(
                self.browser, self.commcell.webconsole_hostname
            )
            self.webconsole.login()
            self.webconsole.goto_apps()
            self.apps_page = AppsPage(self.webconsole)
            if self.utils.is_app_installed(_APP_CONFIG.DEFAULT.name):
                app = App(_APP_CONFIG.DEFAULT.name, self.apps_page)
                app.delete()
            self.store_app = storeapp.StoreApp(self.webconsole)
            self.viewer = viewer.CustomReportViewer(self.webconsole)
            self.apps_page.goto_store()
        except Exception as err:
            raise CVTestCaseInitFailure(err) from err

    @test_step
    def install_app_from_store(self):
        """Install app from store"""
        self.store_app.install_app(_APP_CONFIG.DEFAULT.name)
        self.store_app.open_package(
            _APP_CONFIG.DEFAULT.name, category="APPS"
        )
        rpt_name = self.viewer.get_report_name()
        self.webconsole.get_all_unread_notifications()  # To clear error messages
        primary_comp = _APP_CONFIG.DEFAULT.primary.name
        if rpt_name != primary_comp:
            raise CVTestStepFailure(
                f"Expecting report title [{primary_comp}] when "
                f"[{_APP_CONFIG.DEFAULT.name}] is opened, but "
                f"received [{rpt_name}]"
            )
        self.webconsole.goto_applications()
        self.webconsole.goto_apps()

    @test_step
    def export_installed_app(self):
        """Export installed app"""
        App(_APP_CONFIG.DEFAULT.name, self.apps_page).export()
        self.utils.poll_for_tmp_files(ends_with="cvapp.zip", count=1)

    @test_step
    def delete_installed_app(self):
        """Delete installed app"""
        App(_APP_CONFIG.DEFAULT.name, self.apps_page).delete()
        self.browser.driver.refresh()
        if _APP_CONFIG.DEFAULT.name in self.apps_page.get_apps():
            raise CVTestStepFailure(
                "App visible on Apps page after delete"
            )

    @test_step
    def install_exported_file(self):
        """Install app from File"""
        temp_files = self.utils.get_temp_files(ends_with="cvapp.zip")
        if not temp_files:
            raise CVTestStepFailure("Unable to find exported App")
        self.apps_api.import_app_from_path(temp_files[0])
        if self.utils.is_app_installed(_APP_CONFIG.DEFAULT.name) is False:
            raise CVTestStepFailure("App import failed")
        self.browser.driver.refresh()
        if _APP_CONFIG.DEFAULT.name not in self.apps_page.get_apps():
            raise CVTestStepFailure(
                "App not visible on Apps page after import"
            )

    def run(self):
        try:
            self.init_tc()
            self.install_app_from_store()
            self.export_installed_app()
            self.delete_installed_app()
            self.install_exported_file()
        except Exception as e:
            self.utils.handle_testcase_exception(e)
        finally:
            WebConsole.logout_silently(self.webconsole)
            Browser.close_silently(self.browser)
