# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
""""
Main file for executing this test case

TestCase is the only class definied in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    run()           --  Creates a tablespace, user, table with records, Performs Snap Backup,
                        adds more datfailes  - Performs tablespace only Restore and Validate
"""
import sys
from AutomationUtils import constants
from AutomationUtils.cvtestcase import CVTestCase
from Database.OracleUtils.oraclehelper import OracleHelper


class TestCase(CVTestCase):
    """
    Test case class used to run a given test
    """

    def __init__(self):
        """TestCase constructor"""
        super(TestCase, self).__init__()
        self.name = "Oracle Snap - Tablespace only restore"
        #self.product = self.products_list.ORACLE
        #self.feature = self.features_list.DATAPROTECTION
        self.show_to_user = True
        #self.commserver_name = None
        self.oracle_helper = None
        self.status = constants.FAILED
        self.result_string = "Run of test case 52844 is incomplete"
        self.tcinputs = {
            "client_ip": None,
            "SnapEngine": None,
            "DataFileLocation": None,
            "port": 1521,
            "fileOption": None,
            "OracleOptions": None
        }

    def setup(self):
        """Initializes pre-requisites for this test case"""
        self.log.info('CS set to %s in test case: %s', self.commcell,
                      self.id)

    def validation(self, inputs, ts_name, num_of_files):
        """
        Method vaidates the tablespace , datafiles and table content

            Args:
                inputs (dict) -- Database properties like client ip and port
                ts_name (str) --  Tablespace name to be validated
                num_of_files (int) -- expected number of datafiles

            Raise:
                ValueError -- if datatabase status/ TS/ DF is invalid

        """
        self.log.info("****** Validation Start *****")
        if inputs is not None:
            self.oracle_helper = OracleHelper(self.commcell, self.client, self.instance)
            self.oracle_helper.ora_host_name = inputs['client_ip']
            self.oracle_helper.ora_port = inputs['port']
            self.oracle_helper.db_connect(OracleHelper.CONN_SYSDBA)
            self.log.info('DB DBID: %s', self.instance.dbid)
            self.log.info(
                'DB Version: %s',
                self.oracle_helper.ora_version)

        db_status = ''
        for result in self.oracle_helper.db_execute('select open_mode from v$database'):
            db_status = result[0]
        self.log.info('DB Status: %s', db_status)

        if db_status.strip().upper() != 'READ WRITE':
            self.log.exception('Database status is invalid: %s', db_status)
            raise ValueError('Invalid database status: {0}'.format(db_status))

        (tbs, datafiles) = self.oracle_helper.db_tablespace_validate(ts_name)
        if ts_name == tbs and (num_of_files + 1 == datafiles):
            self.log.info("Tablespace and datafile validation successful")
        else:
            raise ValueError('Tablespace not found')

    def run(self):
        """Main function for test case execution"""

        inputs = self.tcinputs
        try:
            self.log.info("Started executing %s Test Case", self.id)

            self.log.info(
                "%(boundary)s %(message)s %(boundary)s",
                {
                    'boundary': "*" * 10,
                    'message': "Initialize helper objects"
                }
            )

            self.log.info(
                "%(boundary)s %(message)s %(boundary)s",
                {
                    'boundary': "*" * 10,
                    'message': "Create SDK objects"
                }
            )

            self.oracle_helper = OracleHelper(
                self.commcell, self.client, self.instance)

            # If service name is different from instance name, set it like below
            # self.oracle_helper.ora_service_name = inputs['service_name']
            self.oracle_helper.ora_host_name = inputs['client_ip']
            self.oracle_helper.ora_port = inputs['port']
            self.oracle_helper.db_connect(OracleHelper.CONN_SYSDBA)
            db_status = ''
            for result in self.oracle_helper.db_execute(
                    'select open_mode from v$database'):
                db_status = result[0]
            self.log.info('DB DBID: %s', self.instance.dbid)
            self.log.info('DB Status: %s', db_status)
            self.log.info('DB Version: %s', self.oracle_helper.ora_version)

            if db_status.strip().upper() != 'READ WRITE':
                self.log.exception('Database status is invalid: %s', db_status)
                raise ValueError('Invalid database status: {0}'.format(db_status))

            # Enable Intellisnap for the client
            self.client.enable_intelli_snap()
            self.log.info("Intellisnap is enabled on the client")

            # Enable Intellisnap for the client
            self.client.enable_intelli_snap()
            self.log.info("Intellisnap is enabled on the client")

            # Enable Intellisnap on the subclient
            self.subclient.enable_intelli_snap(inputs['SnapEngine'])
            self.log.info("Intelli Snap is already configured on the subclient")

            # Edit properties of the subclient
            if inputs['StoragePolicy'] != self.subclient.data_sp:
                if not isinstance(inputs['StoragePolicy'], str):
                    raise Exception(
                        "Subclient storage policy should be a string - and not matching")
                else:
                    self.subclient.storage_policy = inputs['StoragePolicy']
                    self.log.info("Data Storage Property is set")
            else:
                self.log.info("Data Storage Property is correctly set")

            ts_name = 'CV_52845'
            user = 'cv_52845_user'
            table_prefix = "CV_TABLE_"
            table_limit = 1
            num_of_files = 1
            #row_limit = 10

            # create tablespace
            self.oracle_helper.db_create_tablespace(
                ts_name, inputs['DataFileLocation'], num_of_files)

            # create user/schema
            self.oracle_helper.db_create_user(user, ts_name)

            # create table and populate with 10 records
            self.oracle_helper.db_create_table(ts_name, table_prefix, user, table_limit)

            self.oracle_helper.db_execute('alter system switch logfile')
            self.log.info(' STEP 1: Logfile switch complete...')

            self.log.info(" STEP 1: Running full backup on database: %s",
                          self.instance.instance_name)
            job = self.subclient.backup(r'full')
            if not job.wait_for_completion():
                raise Exception(
                    "Failed to run FULL snap backup job with error: {0}".format(
                        job.delay_reason))
            self.log.info(" STEP 1: Full backup JOB ID: %s", job.job_id)

            self.validation(inputs=None, ts_name=ts_name, num_of_files=num_of_files)

            # Add 2 more datafiles to the tablespace
            self.oracle_helper.db_alter_tablespace(ts_name, inputs['DataFileLocation'], 2)

            self.log.info(
                " STEP 2: Running full restore on database: %s",
                self.instance.instance_name)

            #job = self.subclient.restore()
            # Performs subclient level snap restore - The selected tablespaces from
            # the latest backup are restored to current time
            job = self.subclient.restore(
                files=inputs['fileOption'],
                destination_client=inputs['ClientName'],
                common_options=None,
                browse_option=None,
                oracle_options=inputs['OracleOptions'])
            if not job.wait_for_completion():
                raise Exception("Failed to run restore job with error: {0}".format(
                    job.delay_reason))
            self.log.info(" STEP 2: Full DB restore JOB ID: %s", job.job_id)

            self.validation(inputs=inputs, ts_name=ts_name, num_of_files=4)

            self.status = constants.PASSED
            self.result_string = "Run of test case 52844 has completed successfully"
        except Exception as exp:
            self.log.error("""Testcase failed with exception : %s""", str(exp))
            self.log.error("Detailed Exception : %s", sys.exc_info())
            self.result_string = str(exp)
            self.status = constants.FAILED
