# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()            --  initialize TestCase class

    setup()               --  setup function of this test case

    run()                 --  run function of this test case

    run_backup()          -- runs the backup of specified type

    fetch_copied_times()  --  fetches the copied time of the job to Secondary Copy

    fetch_primary_count() -- fetches the count of primary objects for job in Both DDBs

    tear_down()           --  tear down function of this test case

Inputs to be passed in JSON File:

    AgentName:               File System
    ClientName:              Name of the Client machine
    PrimaryCopyMediaAgent:   Name of a MediaAgent machine - we create primary copy here
    SecondaryCopyMediaAgent: Name of a MediaAgent machine - we create secondary copy here

    ************************          IMPORTANT NOTE:         ***************************
    *************************************************************************************
    ** 1. BOTH THE MEDIA AGENTS SHOULD NOT BE SAME,                                    **
    ** 2. DURING RESTORE VALIDATION WE STOP MEDIA MOUNT MANAGER SERVICE ON PRIMARY MA. **
    *************************************************************************************

Steps:

1: Configure the environment: create a library,Storage Policy-with secondary copy space-optimized),
                              a BackupSet,a SubClient
2: Run Backups on the subclient in order: F_I_I_F_SF

3: Run AuxCopy

4: Run the Validations whether SpaceOptimized Copy feature worked as expected
    -Full Backups should be copied first
    -PrimaryObjects count for 1st Full should be equal in both copies
    -PrimaryObjects count for 2nd Full should be more in Secondary Copy
    -PrimaryObjects count for Synthetic fulls should always be 0
    -Total Primary objects count in both copies should be equal
    -Restore Validation from the Secondary Copy

5: CleanUp the environment
"""

from AutomationUtils import constants
from AutomationUtils.machine import Machine
from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils.options_selector import OptionsSelector
from MediaAgents.MAUtils.mahelper import DedupeHelper, MMHelper


class TestCase(CVTestCase):
    """Class for executing this test case"""

    def __init__(self):
        """Initializes test case class object
        """
        super(TestCase, self).__init__()
        self.name = 'Space Optimized AuxCopy'
        self.tcinputs = {
            "AgentName": None,
            "ClientName": None,
            "PrimaryCopyMediaAgent": None,
            "SecondaryCopyMediaAgent": None
        }
        self.mm_helper = None
        self.ma_machine_1 = None
        self.ma_machine_2 = None
        self.dedupe_helper = None
        self.client_machine = None
        self.ddb_path = None
        self.ma_1_path = None
        self.ma_2_path = None
        self.mount_path = None
        self.client_path = None
        self.mount_path_2 = None
        self.content_path = None
        self.restore_path = None
        self.copy_ddb_path = None
        self.subclient = None
        self.storage_policy = None
        self.copy_name = None
        self.library_name = None
        self.subclient_name = None
        self.backupset_name = None
        self.storage_policy_name = None

    def setup(self):
        """Setup function of this test case"""
        self.client_machine = Machine(self.tcinputs['ClientName'], self.commcell)
        self.ma_machine_1 = Machine(self.tcinputs['PrimaryCopyMediaAgent'], self.commcell)
        self.ma_machine_2 = Machine(self.tcinputs['SecondaryCopyMediaAgent'], self.commcell)
        utility = OptionsSelector(self.commcell)
        client_drive = utility.get_drive(self.client_machine, 1024)
        primary_ma_drive = utility.get_drive(self.ma_machine_1, 4096)
        secondary_ma_drive = utility.get_drive(self.ma_machine_2, 4096)
        self.client_path = client_drive + 'test_' + str(self.id) + self.client_machine.os_sep
        self.ma_1_path = primary_ma_drive + 'test_' + str(self.id) + self.ma_machine_1.os_sep
        self.ma_2_path = secondary_ma_drive + 'test_' + str(self.id) + self.ma_machine_2.os_sep
        self.ddb_path = self.ma_1_path + 'DDB'
        self.mount_path = self.ma_1_path + 'MP'
        self.mount_path_2 = self.ma_2_path + 'MP2'
        self.copy_ddb_path = self.ma_2_path + 'copy_DDB'
        self.content_path = self.client_path + 'Content'
        self.restore_path = self.client_path + 'Restores'
        self.copy_name = str(self.id) + '_Copy'
        self.library_name = str(self.id) + '_Lib'
        self.backupset_name = str(self.id) + '_BS'
        self.subclient_name = str(self.id) + '_SC'
        self.storage_policy_name = str(self.id) + '_SP'
        self.mm_helper = MMHelper(self)
        self.dedupe_helper = DedupeHelper(self)

    def run(self):
        """Run Function of this case"""
        try:
            # 1: Configure the environment
            self.mm_helper.create_uncompressable_data(self.tcinputs['ClientName'],
                                                      self.content_path, 0.5)
            self.mm_helper.configure_disk_library(self.library_name,
                                                  self.tcinputs['PrimaryCopyMediaAgent'],
                                                  self.mount_path)
            self.mm_helper.configure_disk_library(self.library_name + '_2',
                                                  self.tcinputs['SecondaryCopyMediaAgent'],
                                                  self.mount_path_2)
            self.storage_policy = self.dedupe_helper.configure_dedupe_storage_policy(
                self.storage_policy_name,
                self.library_name,
                self.tcinputs['PrimaryCopyMediaAgent'],
                self.ddb_path)
            if self.storage_policy.has_copy(self.copy_name):
                self.storage_policy.delete_secondary_copy(self.copy_name)
            # By default we create space optimized copy in Automation
            storage_policy_copy = self.dedupe_helper.configure_dedupe_secondary_copy(
                self.storage_policy,
                self.copy_name,
                self.library_name + '_2',
                self.tcinputs['SecondaryCopyMediaAgent'],
                self.copy_ddb_path,
                self.tcinputs['SecondaryCopyMediaAgent'])
            self.mm_helper.configure_backupset(self.backupset_name)
            self.subclient = self.mm_helper.configure_subclient(content_path=self.content_path)

            # Remove Association with System Created AutoCopy Schedule
            self.mm_helper.remove_autocopy_schedule(self.storage_policy_name, self.copy_name)

            # 2: Run Backups on the subclient in order: F_I_I_F_SF
            self.log.info('Running backupJobs in order F_I_I_F_SF')
            full_1 = self.run_backup("Full")

            self.mm_helper.create_uncompressable_data(self.tcinputs['ClientName'],
                                                      self.content_path, 0.5)
            incremental_1 = self.run_backup("Incremental")

            self.mm_helper.create_uncompressable_data(self.tcinputs['ClientName'],
                                                      self.content_path, 0.5)
            incremental_2 = self.run_backup("Incremental")

            full_2 = self.run_backup("Full")
            synth_job = self.run_backup("Synthetic_full")

            # 3: Run AuxCopy
            self.log.info('Running AuxCopy Job with Scalable Resource Allocation')
            aux_copy_job = self.storage_policy.run_aux_copy()
            if aux_copy_job.wait_for_completion():
                self.log.info('AuxCopy Completed(Id: %s)', aux_copy_job.job_id)
            else:
                raise Exception('AuxCopy Failed(Id: %s)' % aux_copy_job.job_id)

            # 4: Run the Validations
            self.log.info('********************** VALIDATIONS **********************')
            self.log.info('*** CASE 1: Order of Jobs -> Fulls copied 1st, Remaining Later ***')
            self.log.info('Fetching Copied Times')
            time_1 = self.fetch_copied_times(storage_policy_copy.copy_id, full_1.job_id)
            time_2 = self.fetch_copied_times(storage_policy_copy.copy_id, incremental_1.job_id)
            time_3 = self.fetch_copied_times(storage_policy_copy.copy_id, incremental_2.job_id)
            time_4 = self.fetch_copied_times(storage_policy_copy.copy_id, full_2.job_id)
            time_5 = self.fetch_copied_times(storage_policy_copy.copy_id, synth_job.job_id)

            max_copy_time = max(time_1, time_4, time_5)
            min_copy_time = min(time_2, time_3)
            if max_copy_time < min_copy_time:
                self.log.info('SUCCESS Result: Passed')
            else:
                self.status = constants.FAILED
                self.log.error('ERROR Result: Failed')

            count_0, count_1 = self.fetch_primary_count(full_1.job_id)
            count_2, count_3 = self.fetch_primary_count(incremental_1.job_id)
            count_4, count_5 = self.fetch_primary_count(incremental_2.job_id)
            count_6, count_7 = self.fetch_primary_count(full_2.job_id)
            count_8, count_9 = self.fetch_primary_count(synth_job.job_id)

            self.log.info('*** CASE 2: PrimaryObjectsCount for 1st Full: Sec_Copy = Primary ***')
            if count_0 == count_1:
                self.log.info('SUCCESS Result: Passed')
            else:
                self.log.error('ERROR Result: Failed')

            self.log.info('*** CASE 3: PrimaryObjectsCount for 2nd Full: Sec_Copy >= Primary ***')
            if count_6 <= count_7:
                self.log.info('SUCCESS Result: Passed')
            else:
                self.log.error('ERROR Result: Failed')

            self.log.info('*** CASE 4: PrimaryObjectsCount for SFull: Sec_Copy = Primary = 0 ***')
            if count_8 == count_9 == 0:
                self.log.info('SUCCESS Result: Passed')
            else:
                self.log.error('ERROR Result: Failed')

            self.log.info('*** CASE 5: Sum of PrimaryObjectsCount : Sec_Copy = Primary ***')
            total_in_primary = count_0 + count_2 + count_4 + count_6 + count_8
            total_in_secondary = count_1 + count_3 + count_5 + count_7 + count_9
            if total_in_primary == total_in_secondary:
                self.log.info('SUCCESS Result: Passed %d', total_in_primary)
            else:
                self.status = constants.FAILED
                self.log.error('ERROR Result: Failed %d %d', total_in_primary, total_in_secondary)

            self.log.info('*** Running Restore from Secondary copy when Primary MA is offline ***')

            self.log.info('********* STOPPING Media Mount Manager SERVICE ON PRIMARY MA *********')
            self.ma_machine_1.client_object.stop_service(
                'GXMMM(%s)' % self.ma_machine_1.client_object.instance)
            self.log.info('********* STOPPED Media Mount Manager SERVICE ON PRIMARY MA **********')

            self.log.info('Initiating Restore Job')
            restore_job = self.subclient.restore_out_of_place(self.client.client_name,
                                                              self.restore_path,
                                                              [self.content_path],
                                                              copy_precedence=2)
            if restore_job.wait_for_completion():
                self.log.info('Restore Job: %s Completed', restore_job.job_id)
            else:
                raise Exception('Restore job %s Failed' % restore_job.job_id)

            self.log.info('Validating Restored Data from Secondary Copy')
            restored_path = self.restore_path + self.client_machine.os_sep
            difference = self.client_machine.compare_folders(self.client_machine,
                                                             self.content_path,
                                                             restored_path + 'Content')
            if difference:
                raise Exception('Validating Data restored from Secondary Copy Failed')
            self.log.info('Validation SUCCESS')

        except Exception as exe:
            self.status = constants.FAILED
            self.result_string = str(exe)
            self.log.error('Exception Occurred : %s', str(exe))

    def run_backup(self, backup_type):
        """Runs Backup of specified type and waits for job till it completes
                Args:
                        backup_type    (str)  --   Type of backup To Run
                Return:
                        (object)              --   Object of Job Class

        """
        job = self.subclient.backup(backup_level=backup_type)
        if job.wait_for_completion():
            self.log.info('%s Backup job Completed(Id: %s)', backup_type, job.job_id)
        else:
            raise Exception('%s Backup Job Failed(Id: %s)' % (backup_type, job.job_id))
        return job

    def fetch_copied_times(self, copy_id, job_id):
        """Returns copiedTime from JMJobDataStats
                Args:
                        job_id    (str)  --   Id of Backup Job

                        copy_id   (str)  --   Id of Storage Policy Copy
                Return:
                        (int)            --   Copied Time of Job to the Secondary Copy

        """
        query = '''select distinct copiedTime from JMJobDataStats
                where archGrpCopyId = {0} and jobId = {1}
                '''.format(copy_id, job_id)
        self.log.info('Query: %s', query)
        self.csdb.execute(query)
        result = self.csdb.fetch_one_row()
        self.log.info('Copied Time : %s', result[0])
        return int(result[0])

    def fetch_primary_count(self, job_id):
        """Returns Count of primary Objects for the job in Primary and Secondary Copies
                Args:
                        job_id    (str)  --   Id of Backup Job
                Return:
                        (tuple)          --   Count of Primary objects in DDBs of both copies

        """
        result_1 = self.dedupe_helper.get_primary_objects_sec(job_id, 'Primary')
        result_2 = self.dedupe_helper.get_primary_objects_sec(job_id, self.copy_name)
        return int(result_1), int(result_2)

    def tear_down(self):
        """Tear Down Function of this Case"""
        # 5: CleanUp the environment
        try:
            self.log.info('********* STARTING Media Mount Manager SERVICE ON PRIMARY MA *********')
            self.ma_machine_1.client_object.start_service(
                'GXMMM(%s)' % self.ma_machine_1.client_object.instance)
            self.log.info('********* STARTED Media Mount Manager SERVICE ON PRIMARY MA **********')

            if self.agent.backupsets.has_backupset(self.backupset_name):
                self.log.info("Deleting backupset %s", self.backupset_name)
                self.agent.backupsets.delete(self.backupset_name)

            if self.commcell.storage_policies.has_policy(self.storage_policy_name):
                self.log.info("Deleting storage policy  %s", self.storage_policy_name)
                self.commcell.storage_policies.delete(self.storage_policy_name)

            self.commcell.disk_libraries.delete(self.library_name)
            self.commcell.disk_libraries.delete(self.library_name + '_2')
            self.mm_helper.remove_content(self.ma_1_path, self.ma_machine_1)
            if self.tcinputs['PrimaryCopyMediaAgent'] != self.tcinputs['SecondaryCopyMediaAgent']:
                self.mm_helper.remove_content(self.ma_2_path, self.ma_machine_2)
            if self.tcinputs['PrimaryCopyMediaAgent'] != self.tcinputs['ClientName']:
                if self.tcinputs['SecondaryCopyMediaAgent'] != self.tcinputs['ClientName']:
                    self.mm_helper.remove_content(self.client_path, self.client_machine)
        except Exception as exe:
            self.log.error('ERROR in TearDown. Might need to Cleanup Manually: %s', str(exe))
