# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

This testcase verifies that the "unusual job activity alert" is working for added, modified, deleted and root size
change activities.

TestCase:
    __init__()                  --  Initializes the TestCase class

    setup()                     --  All testcase objects are initializes in this method

    run()                      --  Contains the core testcase logic and it is the one executed

    initialize_job_stats_data() --  Takes a copy of the JobStats.csv file initializes the job stats data for further
    processing

    revert_job_stats()          --  Reverts the job stats file to the previous state without anomalous data

    generate_anomalous_stats()  --  Generates anomalous job stats data

"""

import traceback
import time
import random

from AutomationUtils import constants
from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils.machine import Machine

from Indexing.helpers import IndexingHelpers
from Indexing.database import index_db


class TestCase(CVTestCase):
    """This testcase verifies that the "unusual job activity alert" is working for added, modified, deleted and
    root size change activities."""

    def __init__(self):
        """Initializes test case class object"""

        super(TestCase, self).__init__()
        self.name = 'Indexing - Unusual job activity alert'

        self.tcinputs = {}

        self.cl_machine = None
        self.idx_tc = None
        self.idx_help = None
        self.index_db = None
        self.index_server_machine = None

        self.js_raw = None
        self.js_fields = None
        self.js_lines = None
        self.last_job = None

        self.last_event = None

        self.anomalous_jobs = {}

    def setup(self):
        """All testcase objects are initialized in this method"""

        try:
            self.cl_machine = Machine(self.client, self.commcell)

            self.idx_help = IndexingHelpers(self.commcell)

            if self._subclient is not None:
                self.index_db = index_db.get(self.subclient)
            else:
                self.index_db = index_db.get(self.backupset)

            self.js_fields = ['Total Files in Index', 'Job Id', 'Total Commands', 'Added Files', 'Deleted Folders',
                              'Deleted Files', 'Seconds To Playback', 'Start Time', 'End Time', 'Modified Files',
                              'Job Type', 'Hostname', 'Timestamp', 'Status', 'Root Size']

            self.csdb.execute("""
                select top 1 id from evMsg where subsystem='CvStatAnalysis' order by id desc
            """)
            self.last_event = self.csdb.fetch_one_row()[0]
            self.log.info('Last seen anomaly event is [{0}]'.format(self.last_event))

        except Exception as exp:
            self.log.error(str(traceback.format_exc()))
            raise Exception(exp)

    def run(self):
        """Contains the core testcase logic and it is the one executed

            Steps:
                1) Generate anomalous job stats information for added, modified, deleted and root size activities and
                note down the jobs.
                2) Add the anomalous jobs to the jobstats.csv file
                3) Restart IndexServer services
                4) Verify if anomaly events are generated for the expected jobs.

        """
        try:

            self.initialize_job_stats_data()

            self.log.info('********** Generating anomalous data **********')
            anomalous_data = self.generate_anomalous_stats()
            self.log.info('Anomalous data generated is:')
            self.log.info(anomalous_data)

            self.log.info('********** Inserting anomalous data to the job stats file **********')
            self.index_db.isc_machine.append_to_file(self.index_db.job_stats_file, anomalous_data)

            self.log.info('********** Restarting IndexServer MA services **********')
            try:
                self.index_db.index_server.restart_services()
            except Exception as e:
                self.log.info('Got exception [{0}] while trying to restart the services. Ignoring it'.format(e))

            self.log.info('********** Waiting 30 secs for anomaly process to run **********')
            time.sleep(30)

            self.log.info('********** Verifying if anomaly events are generated **********')
            self.csdb.execute("""
                select id from evMsg where subsystem='CvStatAnalysis' and id > '{0}'
            """.format(self.last_event))

            all_events = self.csdb.fetch_all_rows()
            self.log.info('New events {0}'.format(all_events))

            events_generated = {}
            for event in all_events:
                event_id = event[0]
                self.csdb.execute("""select data from evparam where evmsgid = '{0}' and position = 1""".format(
                    event_id
                ))
                job_id = self.csdb.fetch_one_row()[0]
                
                if not job_id:
                    raise Exception('No anomaly is detected !')
                
                self.csdb.execute("""select data from evparam where evmsgid = '{0}' and position = 2""".format(
                    event_id
                ))
                anomaly_type = self.csdb.fetch_one_row()[0]
                events_generated[int(job_id)] = anomaly_type

            self.log.info('Events generated [{0}]'.format(events_generated))
            self.log.info('Expected events [{0}]'.format(self.anomalous_jobs))

            if events_generated != self.anomalous_jobs:
                raise Exception('Some expected anomaly events are not generated. Please check')

            self.log.info('********** All expected anomaly events are generated. **********')

        except Exception as exp:
            self.log.error('Test case failed with error: ' + str(exp))
            self.result_string = str(exp)
            self.status = constants.FAILED
            self.log.error(str(traceback.format_exc()))

        finally:
            self.revert_job_stats()

    def initialize_job_stats_data(self):
        """Takes a copy of the JobStats.csv file initializes the job stats data for further processing"""

        self.js_raw = self.index_db.isc_machine.read_file(self.index_db.job_stats_file)
        self.index_db.isc_machine.create_file(self.index_db.job_stats_file + '.bkp', self.js_raw)

        lines = self.js_raw.split('\r\n')
        self.js_lines = list(filter(None, lines))

        last_line = self.js_lines[-1].split(',')
        self.last_job = dict(zip(self.js_fields, last_line))

    def revert_job_stats(self):
        """Reverts the job stats file to the previous state without anomalous data"""

        self.log.info('********** Reverting job stats to the initial **********')
        self.index_db.isc_machine.create_file(self.index_db.job_stats_file, self.js_raw)

    def generate_anomalous_stats(self):
        """Generates anomalous job stats data"""

        last_job_id = self.last_job['Job Id']
        last_total_files = self.last_job['Total Files in Index']
        last_end_time = None
        isc_name = self.index_db.index_server.client_hostname

        lines = []

        for r in range(20):
            row = []
            for field in self.js_fields:

                added_items = random.randint(100, 200)
                deleted_items = random.randint(0, 10)
                modified_items = random.randint(10, 20)
                total_items = added_items+modified_items+deleted_items
                start_time = int(time.time())-86400 if last_end_time is None else last_end_time+10
                end_time = start_time + 25
                job_id = int(last_job_id) + 1

                if field == 'Total Files in Index':
                    value = int(last_total_files) + 50
                    last_total_files = value
                    row.append(value)

                if field == 'Job Id':
                    last_job_id = job_id
                    row.append(job_id)

                if field == 'Total Commands':
                    row.append(total_items)

                if field == 'Added Files':
                    if r == 2:  # Insert anomaly
                        added_items = 9999999999999
                        self.anomalous_jobs[last_job_id] = 'Added'

                    row.append(added_items)

                if field == 'Deleted Folders':
                    row.append(0)

                if field == 'Deleted Files':
                    if r == 5:  # Insert anomaly
                        deleted_items = 888888888888888
                        self.anomalous_jobs[last_job_id] = 'Deleted'

                    row.append(deleted_items)

                if field == 'Seconds To Playback':
                    row.append(random.randint(10000, 60000)/1000000)

                if field == 'Start Time':
                    row.append(start_time)

                if field == 'End Time':
                    last_end_time = end_time
                    row.append(end_time)

                if field == 'Modified Files':
                    if r == 10:  # Insert anomaly
                        modified_items = 777777777777777
                        self.anomalous_jobs[last_job_id] = 'Modified'

                    row.append(modified_items)

                if field == 'Job Type':
                    row.append(4)

                if field == 'Hostname':
                    row.append(isc_name)

                if field == 'Timestamp':
                    row.append(start_time*1000)

                if field == 'Status':
                    row.append(0)

                if field == 'Root Size':
                    value = added_items * random.randint(2048, 4096)
                    if r == 15:  # Insert anomaly
                        value = 6*(1024**4)  # 6TB
                        self.anomalous_jobs[last_job_id] = 'Root Size Change'

                    row.append(value)

            row = [str(field) for field in row]
            lines.append(','.join(row))

        return '\r\n'.join(lines)
