# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    setup()         --  setup function of this test case

    run()           --  run function of this test case

    run_validations --  runs the required validations for the case

    tear_down()     --  tear down function of this test case

Inputs to be passed in JSON File:

    AgentName:               File System
    ClientName:              Name of the Client machine
    PrimaryCopyMediaAgent:   Name of a MediaAgent machine - we create primary copy here
    SecondaryCopyMediaAgent: Name of a MediaAgent machine - we create secondary copy here

    Note: Both the MediaAgents can be the same machine

Steps:

1: Configure the environment: create a library,Storage Policy-with Primary, Secondary Copy,
                              a BackupSet,a SubClient

2: Run Multiple Backups: More than 10 (as min. value for Batching jobs is 10)

3: Run AuxCopy with total_jobs_to_process as 10

4: While AuxCopy is running, Verify whether Batching worked as expected
   (should not copy more than 10 jobs simultaneously)

5: CleanUp the environment
"""

import time
from AutomationUtils import constants
from cvpysdk.exception import SDKException
from AutomationUtils.machine import Machine
from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils.options_selector import OptionsSelector
from MediaAgents.MAUtils.mahelper import DedupeHelper, MMHelper


class TestCase(CVTestCase):
    """Class for executing this test case"""

    def __init__(self):
        """Initializes test case class object
        """
        super(TestCase, self).__init__()
        self.name = 'Batching with number of jobs picked to process for dash copy'
        self.tcinputs = {
            "AgentName": None,
            "ClientName": None,
            "PrimaryCopyMediaAgent": None,
            "SecondaryCopyMediaAgent": None
        }
        self.mm_helper = None
        self.ma_machine_1 = None
        self.ma_machine_2 = None
        self.dedupe_helper = None
        self.client_machine = None
        self.ddb_path = None
        self.ma_1_path = None
        self.ma_2_path = None
        self.mount_path = None
        self.client_path = None
        self.mount_path_2 = None
        self.content_path = None
        self.copy_ddb_path = None
        self.subclient = None
        self.storage_policy = None
        self.copy_name = None
        self.library_name = None
        self.subclient_name = None
        self.backupset_name = None
        self.storage_policy_name = None

    def setup(self):
        """Setup function of this test case"""
        self.client_machine = Machine(self.tcinputs['ClientName'], self.commcell)
        self.ma_machine_1 = Machine(self.tcinputs['PrimaryCopyMediaAgent'], self.commcell)
        self.ma_machine_2 = Machine(self.tcinputs['SecondaryCopyMediaAgent'], self.commcell)
        utility = OptionsSelector(self.commcell)
        client_drive = utility.get_drive(self.client_machine, 1024)
        primary_ma_drive = utility.get_drive(self.ma_machine_1, 4096)
        secondary_ma_drive = utility.get_drive(self.ma_machine_2, 4096)
        self.client_path = client_drive + 'test_' + str(self.id) + self.client_machine.os_sep
        self.ma_1_path = primary_ma_drive + 'test_' + str(self.id) + self.ma_machine_1.os_sep
        self.ma_2_path = secondary_ma_drive + 'test_' + str(self.id) + self.ma_machine_2.os_sep
        self.ddb_path = self.ma_1_path + 'DDB'
        self.mount_path = self.ma_1_path + 'MP'
        self.mount_path_2 = self.ma_2_path + 'MP2'
        self.copy_ddb_path = self.ma_2_path + 'copy_DDB'
        self.content_path = self.client_path + 'Content'
        self.copy_name = str(self.id) + '_Copy'
        self.library_name = str(self.id) + '_Lib'
        self.backupset_name = str(self.id) + '_BS'
        self.subclient_name = str(self.id) + '_SC'
        self.storage_policy_name = str(self.id) + '_SP'
        self.mm_helper = MMHelper(self)
        self.dedupe_helper = DedupeHelper(self)

    def run(self):
        """Run Function of this case"""
        try:
            # 1: Configure the environment
            self.mm_helper.create_uncompressable_data(self.tcinputs['ClientName'],
                                                      self.content_path, 0.4)
            self.mm_helper.configure_disk_library(self.library_name,
                                                  self.tcinputs['PrimaryCopyMediaAgent'],
                                                  self.mount_path)
            self.mm_helper.configure_disk_library(self.library_name + '_2',
                                                  self.tcinputs['SecondaryCopyMediaAgent'],
                                                  self.mount_path_2)
            self.storage_policy = self.dedupe_helper.configure_dedupe_storage_policy(
                self.storage_policy_name,
                self.library_name,
                self.tcinputs['PrimaryCopyMediaAgent'],
                self.ddb_path)
            storage_policy_copy = self.dedupe_helper.configure_dedupe_secondary_copy(
                self.storage_policy,
                self.copy_name,
                self.library_name + '_2',
                self.tcinputs['SecondaryCopyMediaAgent'],
                self.copy_ddb_path,
                self.tcinputs['SecondaryCopyMediaAgent'])

            self.mm_helper.configure_backupset(self.backupset_name)
            self.subclient = self.mm_helper.configure_subclient(content_path=self.content_path)
            # Remove association for StoragePolicy with System created AutoCopy Schedule
            try:
                schedule = self.commcell.schedule_policies.get('System Created AutoCopy Schedule')
                assoc_list = schedule._associations
                required_index = -1
                for index in range(len(assoc_list)):
                    if 'storagePolicyName' in assoc_list[index]:
                        if assoc_list[index]['storagePolicyName'] == self.storage_policy_name:
                            required_index = index
                            break
                if required_index != -1:
                    assoc_list = assoc_list[required_index]
                    schedule.update_associations([assoc_list], 'deleted')
            except SDKException as exe:
                self.log.info('Exception: %s', str(exe))

            # 2: Run Multiple Backups
            self.log.info('Running Full Backup')
            self.run_backup('Full')

            self.log.info('Running 15 Incremental Backups')
            for _ in range(1, 16):
                self.mm_helper.create_uncompressable_data(self.tcinputs['ClientName'],
                                                          self.content_path, 0.1)
                self.run_backup("Incremental")

            # 3: Run AuxCopy with total_jobs_to_process as 10
            self.log.info('Running AuxCopy Job(jobs to process = 10)')
            aux_copy_job = self.storage_policy.run_aux_copy(total_jobs_to_process=10)

            self.log.info('AuxCopy Job Initiated(Id: %s)', aux_copy_job.job_id)

            # 4: While AuxCopy is running, Verify whether Batching worked as expected
            self.run_validations(aux_copy_job, storage_policy_copy)

        except Exception as exe:
            self.status = constants.FAILED
            self.result_string = str(exe)
            self.log.error('Test Case Failed with Exception : %s', str(exe))

    def run_backup(self, backup_type):
        """Runs Backup of specified type and waits for job till it completes
                Args:
                        backup_type    (str)  --   Type of backup To Run

        """
        job = self.subclient.backup(backup_level=backup_type)
        if job.wait_for_completion():
            self.log.info('%s Backup job Completed(Id: %s)', backup_type, job.job_id)
        else:
            raise Exception('%s Backup Job Failed(Id: %s)' % (backup_type, job.job_id))

    def run_validations(self, aux_copy_job, storage_policy_copy):
        """Runs the Validations for the Case
                Args:
                        aux_copy_job          (object)  --   Object of Job Class for AuxCopy Job

                        storage_policy_copy   (object)  --   Object of StoragePolicyCopy Class

        """
        self.log.info('**************************** VALIDATIONS *********************************')
        self.log.info('*** CASE 1: Verify No. of Jobs Picked simultaneously always <= 10***')
        query = '''select count(distinct BackupJobID)
                from ArchChunkToReplicate where AdminJobID = {0}
                '''.format(aux_copy_job.job_id)
        self.log.info('Query: %s', query)
        job_status = ['Pending', 'Running', 'Waiting']
        elapsed_time = 1800
        while aux_copy_job.status in job_status and elapsed_time > 0:
            self.csdb.execute(query)
            row = self.csdb.fetch_one_row()
            if int(row[0]) <= 10:
                self.log.info('Result: Jobs Picked for AuxCopy are %s', str(row[0]))
            else:
                self.status = constants.FAILED
                self.log.error('ERROR Result: Jobs Picked for AuxCopy > 10:  %s', str(row[0]))
            time.sleep(20)
            elapsed_time -= 20
        if aux_copy_job.status != 'Completed':
            aux_copy_job.kill()
            raise Exception('AuxCopy Job Killed - Run > 30 minutes:Id:  %s' % aux_copy_job.job_id)

        # AuxCopy Completes by now: Run other Validations
        self.log.info('*** CASE 2: Verify All Jobs are copied ***')

        query = '''select count(jobId) from JMJobDataStats
                where status in(101,102,103) and archGrpCopyId = {0}
                '''.format(storage_policy_copy.copy_id)
        self.log.info('Query: %s', query)
        self.csdb.execute(query)
        row = self.csdb.fetch_one_row()
        if int(row[0]) == 0:
            self.log.info('SUCCESS Result: Remaining Jobs are Copied')
        else:
            self.status = constants.FAILED
            self.log.error('ERROR Result: Still some Jobs are not copied')

        validation3 = 'Verify from JMJobOptions that options are set to '
        validation3 += 'populate job list & resubmit aux copy on completion'
        self.log.info('*** CASE 3: %s ***', validation3)
        query = '''select distinct jobId from JMJobOptions
                where attributeId in (69,70) and JobId = {0}
                '''.format(aux_copy_job.job_id)
        self.log.info('Query: %s', query)
        self.csdb.execute(query)
        row = self.csdb.fetch_one_row()
        if int(row[0]) == int(aux_copy_job.job_id):
            self.log.info('SUCCESS Result: Validation Succeeded')
        else:
            self.status = constants.FAILED
            self.log.error('ERROR Result: Validation Failed')

        self.log.info('*** CASE 4: Verify from TM_JobOptions - Count of Jobs to process ***')
        query = '''select value from TM_JobOptions
                where optionId = 1654993746 and JobId = {0}
                '''.format(aux_copy_job.job_id)
        self.log.info('Query: %s', query)
        self.csdb.execute(query)
        row = self.csdb.fetch_one_row()
        if int(row[0]) == 10:
            self.log.info('SUCCESS Result: Validation passed')
        else:
            self.status = constants.FAILED
            self.log.error('ERROR Result: %s: Validation Failed', row[0])

    def tear_down(self):
        """Tear Down Function of this Case"""
        # 5: CleanUp the environment
        try:
            if self.agent.backupsets.has_backupset(self.backupset_name):
                self.log.info("Deleting backupset %s", self.backupset_name)
                self.agent.backupsets.delete(self.backupset_name)

            if self.commcell.storage_policies.has_policy(self.storage_policy_name):
                self.log.info("Deleting storage policy  %s", self.storage_policy_name)
                self.commcell.storage_policies.delete(self.storage_policy_name)

            self.commcell.disk_libraries.delete(self.library_name)
            self.commcell.disk_libraries.delete(self.library_name + '_2')
            self.mm_helper.remove_content(self.ma_1_path, self.ma_machine_1)
            if self.tcinputs['PrimaryCopyMediaAgent'] != self.tcinputs['SecondaryCopyMediaAgent']:
                self.mm_helper.remove_content(self.ma_2_path, self.ma_machine_2)
            if self.tcinputs['PrimaryCopyMediaAgent'] != self.tcinputs['ClientName']:
                if self.tcinputs['SecondaryCopyMediaAgent'] != self.tcinputs['ClientName']:
                    self.mm_helper.remove_content(self.client_path, self.client_machine)
        except Exception as exe:
            self.log.error('ERROR in TearDown. Might need to Cleanup Manually: %s', str(exe))
