# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
""""
Main file for executing this test case

TestCase is the only class definied in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    run()           --  Creates a tablespace, user, table with records, Validates table records,
                        Performs Backup, Drops Table
                        Restores the table and Validates table records.
"""
import sys
from time import sleep
from AutomationUtils import constants
from AutomationUtils.cvtestcase import CVTestCase
from Database.OracleUtils.oraclehelper import OracleHelper


class TestCase(CVTestCase):
    """
    Test case class used to run a given test
    """

    def __init__(self):
        """TestCase constructor"""
        super(TestCase, self).__init__()
        self.name = "Oracle Test Case 52658 - Snap Partial Datafile Restore"
        #self.product = self.products_list.ORACLE
        #self.feature = self.features_list.DATAPROTECTION
        self.show_to_user = True
        #self.commserver_name = None
        self.oracle_helper = None
        self.status = constants.FAILED
        self.result_string = "Run of test case 52658 is incomplete"
        self.tcinputs = {
            "client_ip": None,
            "SnapEngine": None,
            "DataFileLocation": None,
            "port": 1521,
            "fileOption": None,
            "OracleOptions": None
        }

    def setup(self):
        """Initializes pre-requisites for this test case"""
        self.log.info('CS set to %s in test case: %s', self.commcell,
                      self.id)

    def validation(
            self,
            inputs,
            ts_name,
            num_of_files,
            user,
            table,
            multiplier=1,
            row_limit=10):
        """Method vaidates the tablespace , datafiles and table content

        Args:
            inputs (dict) -- Database properties like client ip and port
            ts_name (str) --  Tablespace name to be validated
            num_of_files (int) -- expected number of datafiles
            user (str) -- associated table user if any
            table (str) -- name of the table to be validated for records
            multiplier (int) -- used to denote the run you want to validate for
            row_limit (int) -- row limit set for the table in the test case

        Raise:
            ValueError -- if datatabase status/ TS/ DF is invalid

        """
        self.log.info("****** Validation Start *****")
        self.oracle_helper = OracleHelper(
            self.commcell, self.client, self.instance)
        self.oracle_helper.ora_host_name = inputs['client_ip']
        self.oracle_helper.ora_port = inputs['port']
        self.oracle_helper.db_connect(OracleHelper.CONN_SYSDBA)

        db_status = ''
        for result in self.oracle_helper.db_execute(
                'select open_mode from v$database'):
            db_status = result[0]
        self.log.info('DB DBID: %s', self.instance.dbid)
        self.log.info('DB Status: %s', db_status)
        self.log.info(
            'DB Version: %s',
            self.oracle_helper.ora_version)

        if db_status.strip().upper() != 'READ WRITE':
            message = 'Database status is invalid: {0}'.format(db_status)
            self.log.exception(message)
            raise ValueError('Invalid database status: {0}'.format(db_status))

        (tbs, datafiles) = self.oracle_helper.db_tablespace_validate(ts_name)
        if ts_name == tbs and (num_of_files + 1 == datafiles):
            self.log.info("Tablespace and datafiles validation successful")
        else:
            raise ValueError('Tablespace not found')

        rec = self.oracle_helper.db_table_validate(user, table)
        if multiplier * row_limit == rec:
            self.log.info("Table records validation successful")
        else:
            raise ValueError("Table records not matching")
        self.log.info("****** Validation complete *****")

    def run(self):
        """Main function for test case execution"""

        inputs = self.tcinputs
        try:
            self.log.info("Started executing %s Test Case", self.id)

            self.log.info(
                "%(boundary)s %(message)s %(boundary)s",
                {
                    'boundary': "*" * 10,
                    'message': "Initialize helper objects"
                }
            )

            self.log.info(
                "%(boundary)s %(message)s %(boundary)s",
                {
                    'boundary': "*" * 10,
                    'message': "Create SDK objects"
                }
            )

            self.oracle_helper = OracleHelper(
                self.commcell, self.client, self.instance)

            # If service name is different from instance name, set it like below
            # self.oracle_helper.ora_service_name = inputs['service_name']
            self.oracle_helper.ora_host_name = inputs['client_ip']
            self.oracle_helper.ora_port = inputs['port']
            self.oracle_helper.db_connect(OracleHelper.CONN_SYSDBA)
            db_status = ''
            for result in self.oracle_helper.db_execute(
                    'select open_mode from v$database'):
                db_status = result[0]
            self.log.info('DB DBID: %s', self.instance.dbid)
            self.log.info('DB Status: %s', db_status)
            self.log.info('DB Version: %s', self.oracle_helper.ora_version)

            if db_status.strip().upper() != 'READ WRITE':
                self.log.exception('Database status is invalid: %s', db_status)
                raise ValueError('Invalid database status: {0}'.format(db_status))
            # Enable Intellisnap for the client
            self.client.enable_intelli_snap()
            self.log.info("Intellisnap is enabled on the client")

            # Enable Intellisnap on the subclient
            self.subclient.enable_intelli_snap(inputs['SnapEngine'])
            self.log.info("Intelli Snap is configured on the subclient")

            # Edit properties of the subclient for restore
            if inputs['StoragePolicy'] != self.subclient.data_sp:
                if not isinstance(inputs['StoragePolicy'], str):
                    raise Exception(
                        "Subclient storage policy should be a string - and not matching")
                else:
                    self.subclient.storage_policy = inputs['StoragePolicy']
                    self.log.info("Data Storage Property is set")
            else:
                self.log.info("Data Storage Property is correctly set")

            ts_name = 'CV_52658'
            user = 'cv_52658_user'
            table_prefix = "CV_TABLE_"
            table_limit = 1
            # The number of files to be added to the tablespace excluding the first datafile.
            num_of_files = 1
            row_limit = 10
            # File number to be retrieved
            # If the num_of_file = 3, file_num range = [1,3]

            # create tablespace
            self.oracle_helper.db_create_tablespace(
                ts_name, inputs['DataFileLocation'], num_of_files)

            # create user/schema
            self.oracle_helper.db_create_user(user, ts_name)

            # create table
            self.oracle_helper.db_create_table(ts_name, table_prefix, user, table_limit)

            self.oracle_helper.db_execute('alter system switch logfile')
            self.log.info(' STEP 1: Logfile switch complete...')
            self.log.info(" STEP 1: Running full backup on database: %s",
                          self.instance.instance_name)
            job = self.subclient.backup(r'full')
            if not job.wait_for_completion():
                raise Exception("Failed to run FULL snap backup job with error: {0}"
                                .format(job.delay_reason))
            self.log.info(" STEP 1: Full backup JOB ID: %s", job.job_id)

            # Dropping datafile with the suffix 1 after the backup
            self.oracle_helper.db_execute(
                "alter tablespace {0} drop datafile '{1}{2}{3}.dbf'".format(
                    ts_name, inputs['DataFileLocation'], ts_name, 1))
            self.log.info("Datafile in the Tablespace %s dropped", ts_name)

            sleep(3)

            self.log.info(
                " STEP 1: Running instance level snap restore from full snap backup: %s",
                self.instance.instance_name)

            # Restoring and recovering from latest backup
            # Performs an instance level snap full restore - All the tablespaces from
            # the latest backup are restored
            job = self.instance.restore(
                files=inputs['fileOption'],
                destination_client=inputs['ClientName'],
                common_options=None,
                oracle_options=inputs['OracleOptions'])
            if not job.wait_for_completion():
                raise Exception("Failed to run restore job with error: {0}"
                                .format(job.delay_reason))

            self.log.info(" STEP 1: Full database snap restore JOB ID: %s", job.job_id)

            self.validation(inputs, ts_name, 0, user, table_prefix + "01", 1, row_limit)

            self.status = constants.PASSED
            self.result_string = "Run of test case 52658 has completed successfully"
        except Exception as exp:
            self.log.error("""Testcase failed with exception : %s""", str(exp))
            self.log.error("Detailed Exception : %s", sys.exc_info())
            self.result_string = str(exp)
            self.status = constants.FAILED
