# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
""""
Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    run()           --  Creates a tablespace, user, table with records,
                        Performs Snap Backup, Restore and Validate
                        adds more records - Performs Snap Backup, Restores and Validate
"""
from time import sleep
from cvpysdk.subclient import Subclients
from AutomationUtils import constants
from AutomationUtils.cvtestcase import CVTestCase
from Database.OracleUtils.oraclehelper import OracleHelper


class TestCase(CVTestCase):
    """
    Test case class used to run a given test
    """

    def __init__(self):
        """TestCase constructor"""
        super(TestCase, self).__init__()
        self.name = "Oracle Snap Test Case - Acceptance test for all basic oracle functions"
        self.show_to_user = True
        self.oracle_helper = None
        self.status = constants.FAILED
        self.result_string = "Run of test case 52632 is incomplete"
        self.tcinputs = {
            "SnapEngine": None,
        }

    def setup(self):
        """Initializes pre-requisites for this test case"""
        self.log.info('CS set to %s in test case: %s', self.commcell,
                      self.id)
        ######Establishing Client and Oracle Connection##########
        self.oracle_helper = OracleHelper(
            self.commcell, self.client, self.instance)
        self.oracle_helper.db_connect(OracleHelper.CONN_SYSDBA)

    def run(self):
        """Main function for test case execution"""

        inputs = self.tcinputs
        try:
            self.log.info("Started executing %s Test Case", self.id)

            self.log.info(
                "%(boundary)s %(message)s %(boundary)s",
                {
                    'boundary': "*" * 10,
                    'message': "Initialize helper objects"
                }
            )

            db_status = ''
            for result in self.oracle_helper.db_execute(
                    'select open_mode from v$database'):
                db_status = result[0]
            self.log.info('DB DBID: %s', self.instance.dbid)
            self.log.info('DB Status: %s', db_status)
            self.log.info('DB Version: %s', self.oracle_helper.ora_version)

            if db_status.strip().upper() != 'READ WRITE':
                self.log.exception('Database status is invalid: %s', db_status)
                raise ValueError('Invalid database status: {0}'.format(db_status))

            #####Enable Intellisnap for the client ###################
            self.client.enable_intelli_snap()
            self.log.info("Intellisnap is enabled on the client")

            self.sub_client = self.instance.subclients.get('default')
            storage_policy = self.sub_client.storage_policy

            ###### Create Subclient for the Snapbackup content #######
            self.subclients = Subclients(self.instance)
            if not self.subclients.has_subclient("FullSnap1"):
                self.log.info(' STEP: Creating Subclient for the Snap Backups')
                self.subclients.add(
                    "FullSnap1", storage_policy, None, "OnlineSnap")
            else:
                self.log.info("Subclient named 'FullSnap1' exists - Skipping creation")

            self.subclient = self.subclients.get("FullSnap1")
            ##### Modifying the properties of the subclient for Snap ###
            self.subclient.set_prop_for_orcle_subclient(storage_policy, inputs['SnapEngine'])

            ts_name = 'CV_52632'
            user = 'cv_52632_user'
            table_prefix = "CV_TABLE_"
            table_limit = 1
            num_of_files = 1
            row_limit = 10

            data_file_location = self.oracle_helper.db_fetch_dbf_location()

            # create a sample tablespace
            self.oracle_helper.db_create_tablespace(
                ts_name, data_file_location, num_of_files)

            # create a sample user/schema
            self.oracle_helper.db_create_user(user, ts_name)

            # create table and populate with 10 records
            self.oracle_helper.db_create_table(
                ts_name, table_prefix, user, table_limit)

            self.oracle_helper.db_execute('alter system switch logfile')
            self.log.info(' STEP 1: Logfile switch complete...')

            self.log.info(
                " STEP 1: Running full backup on database: %s",
                self.instance.instance_name)
            job = self.subclient.backup(r'full')
            if not job.wait_for_completion():
                raise Exception(
                    "Failed to run FULL snap backup job with error: {0}".format(
                        job.delay_reason))
            self.log.info(
                " STEP 1: Full backup JOB ID: %s",
                job.job_id)

            self.log.info("Waiting for the restore job to start")
            sleep(15)

            self.log.info(
                "Running instance level snap restore from full snap backup on database: %s",
                self.instance.instance_name)

            # Performs an instance level snap full restore - All the
            # tablespaces from the latest backup are restored
            job = self.instance.restore(
                destination_client=self.client.client_name,
                common_options=None,
                oracle_options=None, tag='SNAP')
            if not job.wait_for_completion():
                raise Exception("Failed to run restore job with error: {0}"
                                .format(job.delay_reason))

            self.log.info(
                " STEP 1: Full database snap restore JOB ID: %s",
                job.job_id)

            self.oracle_helper.validation(
                ts_name,
                num_of_files,
                user,
                table_prefix +
                "01",
                row_limit)

            # populating table with 10 more records
            self.oracle_helper.db_populate_table(
                table_prefix, user, table_limit)

            # Step 2 -- Run incremental backup : It's auto converted to full
            self.log.info(
                " STEP 2: Running Snap full backup on database: %s",
                self.instance.instance_name)
            job = self.subclient.backup(r'incremental')
            if not job.wait_for_completion():
                raise Exception("Failed to run full backup job with error: {0}"
                                .format(job.delay_reason))
            self.log.info(
                " STEP 2: Incremental->Full backup JOB ID: %s", job.job_id)

            self.log.info(
                " STEP 2: Running full restore on database: %s",
                self.instance.instance_name)

            # Performs subclient level snap full restore - All the tablespaces
            # from the latest backup are restored
            job = self.subclient.restore(
                common_options=None,
                destination_client=self.client.client_name,
                oracle_options=None, tag='SNAP')
            if not job.wait_for_completion():
                raise Exception(
                    "Failed to run restore job with error: {0}".format(
                        job.delay_reason))
            self.log.info(
                " STEP 2: Incremental->Full DB restore JOB ID: %s", job.job_id)

            self.oracle_helper.validation(
                ts_name,
                num_of_files,
                user,
                table_prefix +
                "01",
                row_limit)

            self.status = constants.PASSED
            self.result_string = "Run of test case 52632 has completed successfully"
        except Exception as exp:
            self.log.error("""Testcase failed with exception : %s""", exp)
            self.result_string = "Run of test case 52632 failed"
            self.status = constants.FAILED
