# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Custom Reports: Validate Join Datasets"""

from Reports.Custom.utils import CustomReportUtils

from Web.Common.cvbrowser import BrowserFactory, Browser
from Web.Common.exceptions import (
    CVTestStepFailure,
    CVTestCaseInitFailure
)
from Web.Common.page_object import TestStep


from Web.WebConsole.webconsole import WebConsole
from Web.WebConsole.Reports.navigator import Navigator
from Web.WebConsole.Reports.Custom import builder
from Web.WebConsole.Reports.Custom import viewer

from AutomationUtils.cvtestcase import CVTestCase


class TestCase(CVTestCase):
    """ TestCase class used to execute the test case from here."""
    test_step = TestStep()

    def __init__(self):
        super(TestCase, self).__init__()
        self.name = "Custom Reports: Validate Join Datasets"
        self.applicable_os = self.os_list.WINDOWS
        self.product = self.products_list.METRICSREPORTS
        self.feature = self.features_list.WEBCONSOLE
        self.show_to_user = True
        self.browser = None
        self.webconsole = None
        self.table = None
        self.join_dataset = None
        self.report_builder = None
        self.utils = CustomReportUtils(self)

    def init_tc(self):
        """ Initial configuration for the test case. """
        try:
            self.browser = BrowserFactory().create_browser_object()
            self.browser.set_downloads_dir(self.utils.get_temp_dir())
            self.browser.open()
            self.webconsole = WebConsole(self.browser, self.commcell.webconsole_hostname)
            self.utils.webconsole = self.webconsole
            self.webconsole.login()
            self.webconsole.goto_reports()

            self.utils.cre_api.delete_custom_report_by_name(self.name, suppress=True)
            navigator = Navigator(self.webconsole)
            navigator.goto_report_builder()
        except Exception as exception:
            raise CVTestCaseInitFailure(exception)from exception

    @test_step
    def create_join_dataset(self):
        """Creates join dataset from two database datasets"""
        self.report_builder = builder.ReportBuilder(self.webconsole)
        self.report_builder.set_report_name(self.name)

        database_dataset = builder.Datasets.DatabaseDataset()
        self.report_builder.add_dataset(database_dataset)
        database_dataset.set_dataset_name("DataSet A")
        database_dataset.set_sql_query("SELECT '1' [One], '2' [Two], '3' [Three] "
                                       "UNION ALL "
                                       "SELECT '11' [One], '12' [Two], '13' [Three]")
        database_dataset.save()

        database_dataset = builder.Datasets.DatabaseDataset()
        self.report_builder.add_dataset(database_dataset)
        database_dataset.set_dataset_name("DataSet B")
        database_dataset.set_sql_query("SELECT 'a' [One], 'b' [Two],  'c' [Three] ")
        database_dataset.save()

        self.join_dataset = builder.Datasets.JoinDataset()
        self.report_builder.add_dataset(self.join_dataset)
        self.join_dataset.set_dataset_name("Joined DS")
        self.join_dataset.set_sql_query("SELECT * FROM :[DataSet A] UNION ALL "
                                        "SELECT * FROM :[DataSet B]")

    @test_step
    def preview_data(self):
        """Preview the data"""
        result = {'One': ['1', '11', 'a'], 'Two': ['2', '12', 'b'], 'Three': ['3', '13', 'c']}
        data = self.join_dataset.get_preview_data()
        self.validate(data, result)

    @test_step
    def add_post_query_filter(self):
        """Add post query filter"""
        result = {'One': ['1', 'a'], 'Two': ['2', 'b'], 'Three': ['3', 'c']}
        self.join_dataset.set_post_query_filter("SELECT *FROM $this$ WHERE One != '11'")
        data = self.join_dataset.get_preview_data()
        self.validate(data, result)
        self.join_dataset.save()
        self.table = builder.DataTable("Automation Table 51462")
        self.report_builder.add_component(self.table, self.join_dataset)
        self.table.add_column_from_dataset()
        self.report_builder.save(deploy=True)

    @test_step
    def apply_filter(self):
        """Apply filter on table"""
        self.report_builder.open_report()
        report_viewer = viewer.CustomReportViewer(self.webconsole)
        self.table = viewer.DataTable("Automation Table 51462")
        report_viewer.associate_component(self.table)

        data = self.table.get_table_data()
        result = {'One': ['1', 'a'], 'Two': ['2', 'b'], 'Three': ['3', 'c']}
        self.validate(data, result)

        self.table.set_filter("One", "a")

        data = self.table.get_table_data()
        result = {'One': ['a'], 'Two': ['b'], 'Three': ['c']}
        self.validate(data, result)

    @test_step
    def export_as_csv(self):
        """Export to CSV"""
        row_1 = ['One', 'Two', 'Three']
        row_2 = ['a', 'b', 'c']
        self.utils.reset_temp_dir()
        self.table.export_to_csv()
        self.utils.poll_for_tmp_files("csv")
        content = self.utils.get_csv_content(self.utils.get_temp_files()[0])
        if content[3] != row_1 or content[4] != row_2:
            raise CVTestStepFailure("Unexpected data [%s\n%s] \n expected [%s\n%s]" % (
                str(content[3]), str(content[4]), str(row_1), str(row_2)))

    def validate(self, data, result):
        """Validates the data against the given result"""
        if data != result:
            raise CVTestStepFailure("Unexpected data [%s], expected [%s]" % (
                str(data), str(result)))

    def run(self):
        try:
            self.init_tc()
            self.create_join_dataset()
            self.preview_data()
            self.add_post_query_filter()
            self.apply_filter()
            self.export_as_csv()

        except Exception as err:
            self.utils.handle_testcase_exception(err)

        finally:
            WebConsole.logout_silently(self.webconsole)
            Browser.close_silently(self.browser)
