# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

from AutomationUtils.cvtestcase import CVTestCase
from Reports.storeutils import StoreUtils
from Web.Common.cvbrowser import (
    BrowserFactory,
    Browser
)
from Web.Common.exceptions import (
    CVTestStepFailure,
    CVTestCaseInitFailure
)
from Web.Common.page_object import TestStep
from Web.WebConsole.Store.storeapp import ReadMe
from Web.WebConsole.Store.storeapp import StoreApp
from Web.WebConsole.webconsole import WebConsole


class TestCase(CVTestCase):

    test_step = TestStep()

    def __init__(self):
        super(TestCase, self).__init__()
        self.name = "Store: Price Filters"
        self.applicable_os = self.os_list.WINDOWS
        self.product = self.products_list.SOFTWARESTORE
        self.feature = self.features_list.WEBCONSOLE
        self.store: StoreApp = None
        self.readme: ReadMe = None
        self.webconsole: WebConsole = None
        self.browser = None
        self.inputs = StoreUtils.get_store_config()
        self.utils = StoreUtils(self)

    def init_tc(self):
        try:
            self.utils.store_server_api  # to initialize API
            self.browser = BrowserFactory().create_browser_object()
            self.browser.open()
            self.webconsole = WebConsole(
                self.browser,
                self.commcell.webconsole_hostname
            )
            self.webconsole.login()
            self.webconsole.goto_store()
            self.store = StoreApp(self.webconsole)
        except Exception as e:
            raise CVTestCaseInitFailure(e) from e

    def _validate_if_filter_exists(self, category):
        self.log.info("")
        self.log.info(f"Validating filters for category {category}")
        self.store.filter_by(category=category)
        filters = self.store.get_all_available_filters()
        if self.inputs.PriceFilters.sort() != filters.sort():
            raise CVTestStepFailure(
                f"Expected {self.inputs.PriceFilters}, received {filters}"
            )

    def _validate_if_filter_works(self, category, sub_category, quick_filter):
        self.log.info("")
        self.log.info(
            f"Validating packages inside Category - [{category}], "
            f"Subcategory - [{sub_category}], quick filter [{quick_filter}]"
        )
        packages = self.store.get_all_packages(
            category=category,
            sub_category=sub_category,
            quick_filter=quick_filter
        ).sort()
        quick_filter = "%" if quick_filter is "All" else quick_filter
        db_pkgs = self.utils.get_pkgs_from_server_db(
            category=category,
            sub_category=sub_category,
            price=quick_filter
        ).sort()
        if packages != db_pkgs:
            raise CVTestStepFailure(
                f"One or more package don't belong to Category {category} "
                f"and Sub-Category {sub_category}; Packages are {packages}"
            )

    @test_step
    def step1(self):
        """All, Free, Premium and Proactive Filters should exist for Reports, Workflows, Alerts, Apps and GDPR"""
        categories = ["Reports", "Workflows", "Alerts", "Apps", "GDPR"]
        for category in categories:
            self._validate_if_filter_exists(category)

    @test_step
    def step2(self):
        """When we click on the filter, only packages matching the filter should be displayed"""
        categories = ["Reports", "Workflows", "Alerts", "Apps", "GDPR"]
        for catg in categories:
            sub_categories = self.utils.get_subcategories_from_server_db(catg)[:2]
            for sub_catg in sub_categories:
                quick_filters = ["All", "Premium", "Free", "Proactive Support"]
                for quick_filter in quick_filters:
                    self._validate_if_filter_works(
                        catg, sub_catg, quick_filter
                    )

    def run(self):
        try:
            self.init_tc()
            self.step1()
            self.step2()
        except Exception as e:
            self.utils.handle_testcase_exception(e)
        finally:
            WebConsole.logout_silently(self.webconsole)
            Browser.close_silently(self.browser)
