# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    setup()         --  setup function of this test case

    run()           --  run function of this test case

    tear_down()     --  tear down function of this test case
"""

import sys
from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils import constants
from Application.Exchange.ExchangeMailbox.exchange_mailbox import ExchangeMailbox
from Application.Exchange.ExchangeMailbox.data_generation import TestData
from Application.Exchange.ExchangeMailbox.constants import (
    CLEANUP_POLICY_DEFAULT,
    CLEANUP_POLICY_OLDER_THAN,
    CLEANUP_POLICY_LARGER_THAN,
    CLEANUP_POLICY_ADD_RECALL_LINK,
    CLEANUP_POLICY_LEAVE_MESSAGE_BODY,
    CLEANUP_POLICY_SKIP_UNREAD_MESSAGE,
    CLEANUP_POLICY_HAS_ATTACHMENTS,
    ARCHIVE_POLICY_DEFAULT,
    RETENTION_POLICY_DEFAULT
)


class TestCase(CVTestCase):
    """Class for executing and verifying Recall Link Validation"""

    def __init__(self):
        """Initializes test case class object

            Properties to be initialized:
                name            (str)       --  name of this test case

                show_to_user    (bool)      --  test case flag to determine if the test case is
                                                    to be shown to user or not

                    Accept:
                        True    -   test case will be shown to user from commcell gui

                        False   -   test case will not be shown to user

                    default: False

                tcinputs    (dict)      --  dict of test case inputs with input name as dict key
                                                and value as input type

                        Ex: {
                             "MY_INPUT_NAME": None

                        }

                log     (object)    --  Object of the logger module

        """
        super(TestCase, self).__init__()
        self.name = "Exchange Mailbox Agent : Recall Link Validation"
        self.show_to_user = True
        self.mailboxes_list = []
        self.cleanup_policies_list = []
        self.subclient_content_list = []
        self.smtp_list = []
        self.exmbclient_object = None

    def setup(self):
        """Setup function of this test case"""

        self.log.info('Creating Exchange Mailbox client object.')
        self.exmbclient_object = ExchangeMailbox(self)

        self.log.info(
            "--------------------------TEST DATA CREATION-----------------------------------"
        )
        testdata = TestData(self.exmbclient_object)
        self.mailboxes_list = testdata.create_mailbox()
        self.smtp_list = testdata.import_pst()

        # List of mailboxes
        self.exmbclient_object.users = self.smtp_list

        self.log.info("Mailbox list: ")
        self.log.info(self.mailboxes_list)

        # Creating Exchange Client
        self._client = self.exmbclient_object.cvoperations.add_exchange_client()
        self._subclient = self.exmbclient_object.cvoperations.subclient

        archive_policy_default = ARCHIVE_POLICY_DEFAULT % (self.id)
        retention_policy_default = RETENTION_POLICY_DEFAULT % (self.id)
        cleanup_policy_default = CLEANUP_POLICY_DEFAULT % (self.id)
        cleanup_policy_older_than = CLEANUP_POLICY_OLDER_THAN % (self.id)
        cleanup_policy_larger_than = CLEANUP_POLICY_LARGER_THAN % (self.id)
        cleanup_policy_add_recall_link = CLEANUP_POLICY_ADD_RECALL_LINK % (self.id)
        cleanup_policy_leave_message_body = CLEANUP_POLICY_LEAVE_MESSAGE_BODY % (self.id)
        cleanup_policy_skip_unread_message = CLEANUP_POLICY_SKIP_UNREAD_MESSAGE % (self.id)
        cleanup_policy_has_attachments = CLEANUP_POLICY_HAS_ATTACHMENTS % (self.id)

        # Creating default Exchange Configuration Policies
        self.archive_policy = self.exmbclient_object.cvoperations.add_exchange_policy(
            self.exmbclient_object.cvoperations.get_policy_object
            (archive_policy_default, "Archive"))
        self.cleanup_policy_default = self.exmbclient_object.cvoperations.add_exchange_policy(
            self.exmbclient_object.cvoperations.get_policy_object(
                cleanup_policy_default, "Cleanup"))
        self.cleanup_policies_list.append(self.cleanup_policy_default)
        self.retention_policy = self.exmbclient_object.cvoperations.add_exchange_policy(
            self.exmbclient_object.cvoperations.get_policy_object(
                retention_policy_default, "Retention"))

        # Creating Cleanup policy with rule - message older than
        cleanup_policy_object = self.exmbclient_object.cvoperations.get_policy_object(
            cleanup_policy_older_than, "Cleanup")
        cleanup_policy_object.collect_messages_days_after = 30
        self.cleanup_policy_older_than = (
            self.exmbclient_object.cvoperations.add_exchange_policy(cleanup_policy_object))
        self.cleanup_policies_list.append(self.cleanup_policy_older_than)

        # Creating Cleanup policy with rule - message larger than
        cleanup_policy_object = self.exmbclient_object.cvoperations.get_policy_object(
            cleanup_policy_larger_than, "Cleanup")
        cleanup_policy_object.collect_messages_larger_than = 10
        self.cleanup_policy_larger_than = (
            self.exmbclient_object.cvoperations.add_exchange_policy(cleanup_policy_object))
        self.cleanup_policies_list.append(self.cleanup_policy_larger_than)

        # Creating Cleanup policy with rule - Add recall link
        cleanup_policy_object = self.exmbclient_object.cvoperations.get_policy_object(
            cleanup_policy_add_recall_link, "Cleanup")
        cleanup_policy_object.add_recall_link = True
        self.cleanup_policy_add_recall_link = (
            self.exmbclient_object.cvoperations.add_exchange_policy(cleanup_policy_object))
        self.cleanup_policies_list.append(self.cleanup_policy_add_recall_link)

        # Creating Cleanup policy with rule - Leave Message body
        cleanup_policy_object = self.exmbclient_object.cvoperations.get_policy_object(
            cleanup_policy_leave_message_body, "Cleanup")
        cleanup_policy_object.leave_message_body = True
        self.cleanup_policy_leave_message_body = (
            self.exmbclient_object.cvoperations.add_exchange_policy(cleanup_policy_object))
        self.cleanup_policies_list.append(self.cleanup_policy_leave_message_body)

        # Creating Cleanup policy with rule - Skip unread messages
        cleanup_policy_object = self.exmbclient_object.cvoperations.get_policy_object(
            cleanup_policy_skip_unread_message, "Cleanup")
        cleanup_policy_object.skip_unread_messages = True
        self.cleanup_policy_skip_unread_message = (
            self.exmbclient_object.cvoperations.add_exchange_policy(cleanup_policy_object))
        self.cleanup_policies_list.append(self.cleanup_policy_skip_unread_message)

        # Creating Cleanup policy with rule - Collect messages with attachments
        cleanup_policy_object = self.exmbclient_object.cvoperations.get_policy_object(
            cleanup_policy_has_attachments, "Cleanup")
        cleanup_policy_object.collect_messages_with_attachments = True
        self.cleanup_policy_has_attachments = (
            self.exmbclient_object.cvoperations.add_exchange_policy(cleanup_policy_object))
        self.cleanup_policies_list.append(self.cleanup_policy_has_attachments)

    def run(self):
        """Run function of this test case"""
        try:

            self.log.info("Creating subclient contents with mailbox names and its associated "
                          "Exchange Configuration Policies")

            for iterator in range(0, len(self.cleanup_policies_list)):
                subclient_content = {
                    'mailboxNames': [self.mailboxes_list[iterator]],
                    'is_auto_discover_user': True,
                    'archive_policy': self.archive_policy,
                    'cleanup_policy': self.cleanup_policies_list[iterator],
                    'retention_policy': self.retention_policy,
                }
                self.subclient_content_list.append(subclient_content)

            self.log.info(
                "--------------------------CREATE USER ASSOCIATION"
                "-----------------------------------"
            )
            self.log.info("Creating active_directory object")
            active_directory = self.exmbclient_object.active_directory

            for iterator in range(0, len(self.cleanup_policies_list)):
                self.log.info("User association for Mailbox name: %s",
                              self.subclient_content_list[iterator]['mailboxNames'])
                active_directory.set_user_assocaitions(self.subclient_content_list[iterator])

            self.log.info(
                "--------------------------READING MAILBOX PROPERTIES BEFORE BACKUP"
                "-----------------------------------"
            )

            before_backup_object = self.exmbclient_object.exchange_lib
            before_backup_object.get_mailbox_prop()

            self.log.info(
                "--------------------------RUNNING BACKUP"
                "-----------------------------------"
            )
            self.exmbclient_object.cvoperations.run_backup()

            self.log.info(
                "--------------------------RUNNING CLEANUP"
                "-----------------------------------"
            )

            self.exmbclient_object.cvoperations.cleanup()
            self.log.info(
                "--------------------------READING MAILBOX PROPERTIES AFTER CLEANUP"
                "-----------------------------------"
            )

            after_cleanup_object = self.exmbclient_object.exchange_lib
            after_cleanup_object.get_mailbox_prop()

            cleanup = self.exmbclient_object.cleanup

            cleanup.validate_recall(
                before_backup_object.mailbox_prop,
                after_cleanup_object.mailbox_prop)

        except Exception as ex:
            self.log.error('Error {} on line {}. Error {}'.format(
                type(ex).__name__, sys.exc_info()[-1].tb_lineno, ex))
            self.result_string = str(ex)
            self.status = constants.FAILED
