# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

""""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:
    __init__()      --  initialize TestCase class

    setup()         --  setup function of this test case

    run()           --  run function of this test case

    tear_down()     --  tear down function of this test case

"""
import calendar
import time
import datetime

from AutomationUtils import constants
from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils.machine import Machine
from Server.JobManager.jobmanager_helper import JobManager

from dynamicindex.Datacube.dcube_sync_helper import SyncHelper
from dynamicindex.Datacube.dcube_solr_helper import SolrHelper
from dynamicindex.utils import constants as dynamic_constants


class TestCase(CVTestCase):
    """Class for executing this test case"""

    def __init__(self):
        """Initializes test case class object

            Properties to be initialized:

                name            (str)       --  name of this test case

                tcinputs    (dict)          --  test case inputs with input name as dict key
                                                and value as input type

        """
        super(TestCase, self).__init__()
        self.accessnode_clientid = None
        self.accessnode_client_obj = None
        self.properties = []
        self.datasource_type = dynamic_constants.FILE_SYSTEM_DSTYPE
        self.datasource_id = None
        self.datasource_obj = None
        self.timestamp = None
        self.crawl_jobid = None
        self.coreid = None
        self.solr_core_name = None
        self.total_crawlcount = 0
        self.dcube_helper = None
        self.dssync_regname = "nLastDCubeSyncTime"
        self.sync_helper = None
        self.file_ds_prefix = "DC_file_"
        self.name = "DataCube FileSystem Sync Status to CS DataBase"
        self.tcinputs = {
            "DataSourcename": None,
            "IndexServer": None,
            "IncludedirectoriesPath": None,
            "DoincrementalScan": None,
            "UserName": None,
            "Password": None,
            "PushonlyMetadata": None,
            "Accessnodeclient": None
        }

    def setup(self):
        """Setup function of this test case"""

        try:
            self.sync_helper = SyncHelper(self)
            self.dcube_helper = SolrHelper(self)
            self.timestamp = calendar.timegm(time.gmtime())
            self.tcinputs['DataSourcename'] = self.tcinputs['DataSourcename'] +\
                str(self.timestamp)
            self.log.info("Going to create File System Datasource via Rest API : %s",
                          str(self.tcinputs['DataSourcename']))

            self.accessnode_client_obj = self.commcell.clients.get(
                self.tcinputs['Accessnodeclient'])
            self.log.info("Client object Initialised")
            self.accessnode_clientid = self.accessnode_client_obj.client_id
            self.log.info("Accessnode Client id : %s", str(self.accessnode_clientid))

            prop_name = ['includedirectoriespath', 'doincrementalscan', 'username', 'password', 'pushonlymetadata',
                         'accessnodeclientid', 'createclient', 'candelete', 'appname', 'excludefilters',
                         'minumumdocumentsize', 'maximumdocumentsize']
            prop_value = [self.tcinputs['IncludedirectoriesPath'], self.tcinputs['DoincrementalScan'],
                          self.tcinputs['UserName'], self.tcinputs['Password'], self.tcinputs['PushonlyMetadata'],
                          self.accessnode_clientid, "archiverClient", "true", "DATACUBE", "", "0", "52428800"]
            self.properties = [{"propertyName": prop_name[x], "propertyValue":prop_value[x]}
                               for x in range(0, len(prop_name))]

            self.commcell.datacube.datasources.add(
                self.tcinputs['DataSourcename'],
                self.tcinputs['IndexServer'],
                self.datasource_type, self.properties)
            self.log.info("File system datasource created successfully")

            self.datasource_obj = self.commcell.datacube.datasources.get(
                self.tcinputs['DataSourcename'])
            self.datasource_id = self.datasource_obj.datasource_id
            self.log.info("Created DataSource id : %s", str(self.datasource_id))

            self.log.info("Going to start crawl job for this data source")
            self.crawl_jobid = self.datasource_obj.start_job()
            self.log.info("Started crawl job with id : %s", str(self.crawl_jobid))
            if self.crawl_jobid is None:
                raise Exception("Something went wrong with datasource start job")

            self.log.info("Going to Monitor crawl job for this data source")
            job_manager = JobManager(_job=self.crawl_jobid, commcell=self.commcell)
            if job_manager.wait_for_state('completed', 10, 60, True):
                self.log.info("Job completed")
            else:
                self.log.error("Crawl job for FS datasource failed")
                raise Exception("Crawl job for FS datasource failed")

            self.log.info("Going to Get Status for this data source")
            datasource_status = self.datasource_obj.get_status()
            self.total_crawlcount = datasource_status['status']['totalcount']
            if self.total_crawlcount is not None and self.total_crawlcount == 0:
                raise Exception("Total crawled data count is zero. Please check")

            self.log.info("Crawled Data Count : %s", str(self.total_crawlcount))

        except Exception as ee:
            self.log.error('Setup for the test case failed.')
            self.log.exception(ee)
            self.result_string = str(ee)
            self.status = constants.FAILED
            raise Exception("Test case setup(Creation of Datasource failed). Please check")

    def run(self):
        """Run function of this test case"""
        try:

            current_time = datetime.datetime.now()
            self.sync_helper.set_dcube_sync_registry(current_time, 2, 24, True)
            statusxml = self.sync_helper.get_syncxml_by_csrestart(self.datasource_id)
            self.cs_machineobj = Machine(self.commcell.clients.get(2),
                                         self.commcell)
            reg_key = "CommServe"
            updated_synctime = self.cs_machineobj.get_registry_value(
                reg_key, self.dssync_regname)

            self.log.info("Going to validate the dcube registry")
            self.log.info("Sync time got from registry : %s", updated_synctime)
            self.log.info("Sync time we considered : %s", str(int(current_time.timestamp())))
            if int(updated_synctime) <= int(current_time.timestamp()):
                self.log.info("Sync time not updated properly in registry")
                raise Exception("Sync time not updated properly in registry")

            self.log.info("Dcube sync registry updated properly in registry")
            self.coreid = self.dcube_helper.get_coreid_datasource(self.datasource_id)
            self.solr_core_name = f"{self.file_ds_prefix}{self.tcinputs['DataSourcename']}_{str(self.coreid)}"
            self.log.info("Core id for this Datasource is : %s", str(self.coreid))

            solr_url = self.dcube_helper.get_solr_baseurl(self.tcinputs['IndexServer'], 1)
            self.log.info("Base solr URl : %s", solr_url)

            stats_response = self.dcube_helper.get_corestats(solr_url, self.solr_core_name)
            self.log.info("Core stats response : %s", str(stats_response))

            fsstats_response = self.dcube_helper.get_fs_sync_facets(solr_url, self.solr_core_name)
            self.log.info("FS Core specific stats response : %s", str(fsstats_response))

            # validation starts
            testresult = self.sync_helper.verify_sync_xml(
                statusxml, stats_response, str(int(current_time.timestamp())), fsstats_response)
            if testresult:
                self.log.info("Sync xml validation passed")
            else:
                raise Exception("Sync xml validation failed")

        except Exception as exp:
            self.log.error('Failed to execute test case with error: ')
            self.result_string = str(exp)
            self.log.exception(exp)
            self.status = constants.FAILED

    def tear_down(self):
        """Tear down function of this test case"""
        if self.datasource_obj is not None and self.status == constants.PASSED:
            self.log.info("Delete the data source with id = %s", str(self.datasource_id))
            self.datasource_obj.delete_datasource()
