# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Main file for executing this test case

TestCase is the only class defined in this file.

TestCase: Class for executing this test case

TestCase:

    __init__()      --  initialize TestCase class

    setup()         --  setup function of this test case

    run()           --  run function of this test case

    tear_down()     --  tear down function of this test case

Inputs:

    Hostname    --  the client to install software on

    Username    --  username of the host machine

    Password    --  password of the host machine

    Packages       --   comma separated string of packages to be uninstalled
        Example: Packages = "File System, Oracle"

    **Note**

            * If  "All" is given in Packages, it will install all the packages

            * Install or uninstall can be done only on packages listed on the Admin Console

            * This test case is for new client without multiple instances

            * Give the full hostname of the client machine as input

"""

from AutomationUtils.cvtestcase import CVTestCase
from AutomationUtils import constants
from AutomationUtils.options_selector import OptionsSelector
from AutomationUtils.machine import Machine
from Server.NameChange.name_change_helper import NameChangeHelper
from Web.Common.cvbrowser import BrowserFactory, Browser
from Web.AdminConsole.Helper.DeploymentHelper import DeploymentHelper
from Web.AdminConsole.adminconsole import AdminConsole
from Install import installer_constants
from Install.install_helper import InstallHelper


class TestCase(CVTestCase):
    """Class for installing new windows client and adding software to the client in Admin Console"""

    def __init__(self):
        """Initializes test case class object"""
        super(TestCase, self).__init__()
        self.name = "Install- Admin Console - Install new windows client"
        self.factory = None
        self.browser = None
        self.driver = None
        self.login_obj = None
        self.deployment_helper = None
        self.namechangehelper = None
        self.admin_console = None
        self.tcinputs = {
            "Hostname": None,
            "Username": None,
            "Password": None,
            "Packages": None,
            "HyperVHostname": None,
            "VMHostname": None,
            "HyperVUsername": None,
            "HyperVPassword": None,
            "VMUsername": None,
            "VMPassword": None
        }
        self.install_helper = None

    def setup(self):
        """Initializes pre-requisites for this test case"""
        self.factory = BrowserFactory()
        self.browser = self.factory.create_browser_object()
        self.browser.open()
        self.driver = self.browser.driver
        self.admin_console = AdminConsole(self.browser, self.inputJSONnode['commcell']['webconsoleHostname'])
        self.admin_console.login(self.inputJSONnode['commcell']['commcellUsername'],
                                 self.inputJSONnode['commcell']['commcellPassword'],
                                 stay_logged_in=True)
        self.deployment_helper = DeploymentHelper(self, self.admin_console)
        self.namechangehelper = NameChangeHelper(self)
        self.client_machine = Machine(machine_name=self.tcinputs.get('Hostname'),
                                      username=self.tcinputs.get("Username"),
                                      password=self.tcinputs.get("Password"))
        self.install_helper = InstallHelper(self.commcell)
        self.install_helper.revert_snap(
            server_name=self.tcinputs.get('HyperVHostname'),
            username=self.tcinputs.get('HyperVUsername'),
            password=self.tcinputs.get('HyperVPassword'),
            vm_name=self.tcinputs.get('VMHostname')
        )

    def run(self):
        """Main function for test case execution"""

        try:
            install_path = OptionsSelector.get_drive(self.client_machine)
            self.deployment_helper.add_server_new_windows_or_unix_server(
                hostname=self.tcinputs.get('Hostname'),
                username=self.tcinputs.get('Username'),
                password=self.tcinputs.get('Password'),
                packages=['File System'],
                log_path=installer_constants.DB2LOGLOCATION,
                install_path=install_path
            )
            packages = self.deployment_helper.get_package_names()
            self.namechangehelper.change_client_display_name(
                client_name=self.deployment_helper.get_client_name_from_hostname(self.tcinputs.get('Hostname')),
                new_name=self.deployment_helper.get_client_name_from_hostname(self.tcinputs.get('Hostname')))
            self.deployment_helper.action_add_software(
                client_name=self.deployment_helper.get_client_name_from_hostname(
                    self.tcinputs.get('Hostname')),
                select_all_packages="ALL" in packages,
                packages=[self.tcinputs.get('Packages')]
            )
            # Validating the Install location
            client = self.commcell.clients.get(self.tcinputs.get("Hostname"))
            installed_path = client.install_directory
            if installed_path.startswith(install_path):
                self.log.info("Software Succcessfully installed at the location: %s" % installed_path)
            else:
                self.log.error("Software installed at the location:%s" % installed_path)
                raise Exception("Failed to install at the location:%s" % install_path)

        except Exception as exp:
            self.log.error('Failed with error: %s', exp)
            self.result_string = str(exp)
            self.status = constants.FAILED

        finally:
            AdminConsole.logout_silently(self.admin_console)
            Browser.close_silently(self.browser)
